document.addEventListener('DOMContentLoaded', () => {
  const apiKeyInput = document.getElementById('apiKey');
  const geminiApiKeyInput = document.getElementById('geminiApiKey');
  const modelSelect = document.getElementById('model');
  const reasoningSelect = document.getElementById('reasoning');
  const verbositySelect = document.getElementById('verbosity');
  const saveBtn = document.getElementById('save');
  const statusDiv = document.getElementById('status');
  const openaiKeySection = document.getElementById('openai-key-section');
  const geminiKeySection = document.getElementById('gemini-key-section');

  // Helper to check if model is Gemini
  function isGeminiModel(model) {
    return model.startsWith('gemini-');
  }

  // Update visible API key section based on model
  function updateApiKeySection() {
    const model = modelSelect.value;
    if (isGeminiModel(model)) {
      openaiKeySection.style.display = 'none';
      geminiKeySection.style.display = 'block';
    } else {
      openaiKeySection.style.display = 'block';
      geminiKeySection.style.display = 'none';
    }
  }

  // Load saved settings
  chrome.storage.sync.get(['openaiApiKey', 'geminiApiKey', 'openaiModel', 'openaiReasoning', 'openaiVerbosity'], (result) => {
    if (result.openaiApiKey) {
      apiKeyInput.value = result.openaiApiKey;
    }
    if (result.geminiApiKey) {
      geminiApiKeyInput.value = result.geminiApiKey;
    }
    if (result.openaiModel) {
      modelSelect.value = result.openaiModel;
    }
    if (result.openaiReasoning) {
      reasoningSelect.value = result.openaiReasoning;
    }
    if (result.openaiVerbosity) {
      verbositySelect.value = result.openaiVerbosity;
    }
    updateApiKeySection();
  });

  // Update API key section when model changes
  modelSelect.addEventListener('change', updateApiKeySection);

  // Save settings
  saveBtn.addEventListener('click', () => {
    const model = modelSelect.value;
    const reasoning = reasoningSelect.value;
    const verbosity = verbositySelect.value;
    const isGemini = isGeminiModel(model);

    const apiKey = isGemini ? geminiApiKeyInput.value.trim() : apiKeyInput.value.trim();

    if (!apiKey) {
      statusDiv.className = 'status error';
      statusDiv.textContent = `Please enter ${isGemini ? 'a Gemini' : 'an OpenAI'} API key`;
      return;
    }

    // Validate API key format
    if (isGemini) {
      if (!apiKey.startsWith('AIza')) {
        statusDiv.className = 'status error';
        statusDiv.textContent = 'Invalid Gemini API key format (should start with AIza)';
        return;
      }
    } else {
      if (!apiKey.startsWith('sk-')) {
        statusDiv.className = 'status error';
        statusDiv.textContent = 'Invalid OpenAI API key format (should start with sk-)';
        return;
      }
    }

    const settings = {
      openaiModel: model,
      openaiReasoning: reasoning,
      openaiVerbosity: verbosity
    };

    // Save the appropriate API key
    if (isGemini) {
      settings.geminiApiKey = apiKey;
    } else {
      settings.openaiApiKey = apiKey;
    }

    chrome.storage.sync.set(settings, () => {
      statusDiv.className = 'status success';
      statusDiv.textContent = 'Settings saved successfully!';
      setTimeout(() => {
        statusDiv.className = 'status';
      }, 2000);
    });
  });
});
