(function() {
  'use strict';

  let isPendingSave = false;
  let needsSaveBeforeTranslate = false;

  // Language code to full name mapping
  const LANGUAGE_NAMES = {
    'en': 'English',
    'en_US': 'English',
    'en_GB': 'English (UK)',
    'pt_BR': 'Portuguese (Brazilian)',
    'fr_FR': 'French',
    'de_DE': 'German',
    'es_ES': 'Spanish',
    'tr_TR': 'Turkish',
    'it_IT': 'Italian',
    'pl_PL': 'Polish',
    'nl_NL': 'Dutch',
    'el': 'Greek',
    'el_GR': 'Greek',
    'ru_RU': 'Russian',
    'ja': 'Japanese',
    'zh_CN': 'Chinese (Simplified)',
    'ko_KR': 'Korean',
    'ar': 'Arabic',
    'hi_IN': 'Hindi',
    'sv_SE': 'Swedish',
    'da_DK': 'Danish',
    'fi': 'Finnish',
    'no_NO': 'Norwegian',
    'cs_CZ': 'Czech',
    'hu_HU': 'Hungarian',
    'ro_RO': 'Romanian',
    'uk': 'Ukrainian',
    'vi': 'Vietnamese',
    'th': 'Thai',
    'id_ID': 'Indonesian',
    'ms_MY': 'Malay',
    'he_IL': 'Hebrew'
  };

  // Detect the source language dynamically
  function getSourceLanguageCode() {
    // Method 1: Get from readonly textarea's container ID (most reliable in TranslatePress)
    const readonlyTextarea = document.querySelector('textarea.trp-translation-input[readonly]');
    if (readonlyTextarea) {
      const container = readonlyTextarea.closest('[id^="trp-language-"]');
      if (container) {
        return container.id.replace('trp-language-', '');
      }
    }

    // Method 2: Get from HTML lang attribute
    const htmlLang = document.documentElement.lang;
    if (htmlLang) {
      // Convert format: "de-DE" -> "de_DE"
      return htmlLang.replace('-', '_');
    }

    // Method 3: Try iframe's HTML lang
    try {
      const previewFrame = document.getElementById('trp-preview-iframe');
      if (previewFrame && previewFrame.contentDocument) {
        const iframeLang = previewFrame.contentDocument.documentElement.lang;
        if (iframeLang) {
          return iframeLang.replace('-', '_');
        }
      }
    } catch (e) {}

    // Fallback to English (original behavior)
    return 'en_US';
  }

  // Check if a language code matches the source language
  function isSourceLanguage(langCode) {
    const sourceLang = getSourceLanguageCode();
    // Handle variations: en, en_US, en_GB should all match if source is any English variant
    const normalizedSource = sourceLang.split('_')[0].toLowerCase();
    const normalizedTarget = langCode.split('_')[0].toLowerCase();
    return normalizedSource === normalizedTarget;
  }

  // Check if we're on a TranslatePress editor page
  function isTranslatePressEditor() {
    return document.querySelector('.trp-translation-input') !== null ||
           document.querySelector('[class*="trp-"]') !== null;
  }

  // Get the source text from the editor
  function getSourceText(container) {
    // Look for the source textarea (readonly)
    const englishContainer = container.querySelector('.trp-language-container');
    if (englishContainer) {
      const textarea = englishContainer.querySelector('textarea.trp-translation-input[readonly]');
      if (textarea) {
        return textarea.value || textarea.getAttribute('value') || '';
      }
    }

    // Fallback: look for first readonly textarea
    const readonlyTextarea = container.querySelector('textarea.trp-translation-input[readonly]');
    if (readonlyTextarea) {
      return readonlyTextarea.value || readonlyTextarea.getAttribute('value') || '';
    }

    return '';
  }

  // Get current source text dynamically (for when user switches blocks)
  function getCurrentSourceText() {
    // Method 1: Find the readonly textarea in the source container
    const sourceTextarea = document.querySelector('.trp-language-container textarea.trp-translation-input[readonly]');
    if (sourceTextarea) {
      const value = sourceTextarea.value || sourceTextarea.getAttribute('value') || '';
      if (value) return value;
    }

    // Method 2: Try to get from data-dictionary-entry attribute
    const footer = document.querySelector('.trp-translation-input-footer[data-dictionary-entry]');
    if (footer) {
      try {
        const dictEntry = JSON.parse(footer.dataset.dictionaryEntry || '{}');
        if (dictEntry.original) {
          return dictEntry.original;
        }
      } catch (e) {}
    }

    // Method 3: Look for any visible readonly textarea
    const allReadonly = document.querySelectorAll('textarea.trp-translation-input[readonly]');
    for (const textarea of allReadonly) {
      const value = textarea.value || textarea.getAttribute('value') || '';
      if (value) return value;
    }

    return '';
  }

  // Find all target language containers
  function getTargetLanguages() {
    const languages = [];
    const langContainers = document.querySelectorAll('[id^="trp-language-"]');

    langContainers.forEach(container => {
      const id = container.id;
      const langCode = id.replace('trp-language-', '');

      // Skip the source language (dynamically detected)
      if (isSourceLanguage(langCode)) {
        return;
      }

      const textarea = container.querySelector('textarea.trp-translation-input:not([readonly])');
      if (textarea) {
        languages.push({
          code: langCode,
          name: LANGUAGE_NAMES[langCode] || langCode,
          textarea: textarea,
          container: container
        });
      }
    });

    return languages;
  }

  // Get the page title from the preview iframe
  function getPageTitle() {
    try {
      const previewFrame = document.getElementById('trp-preview-iframe');
      if (previewFrame) {
        // Try accessing contentDocument
        if (previewFrame.contentDocument) {
          return previewFrame.contentDocument.title;
        }
        // Fallback: try accessing contentWindow.document
        if (previewFrame.contentWindow && previewFrame.contentWindow.document) {
          return previewFrame.contentWindow.document.title;
        }
      } else {
        console.log('TRP AI: Preview iframe not found');
      }
    } catch (e) {
      console.log('TRP AI: Could not access preview iframe title:', e);
    }
    return '';
  }

  // Check if model is GPT-5 family
  function isGPT5Family(model) {
    return model.startsWith('gpt-5');
  }

  // Check if model is Gemini family
  function isGeminiFamily(model) {
    return model.startsWith('gemini-');
  }

  // Retry wrapper for fetch requests
  async function fetchWithRetry(url, options, maxRetries = 3) {
    let lastError;
    for (let attempt = 1; attempt <= maxRetries; attempt++) {
      try {
        const response = await fetch(url, options);
        return response;
      } catch (error) {
        lastError = error;
        console.log(`Fetch attempt ${attempt}/${maxRetries} failed:`, error.message);
        if (attempt < maxRetries) {
          // Wait before retry: 1s, 2s, 3s (exponential backoff)
          await new Promise(r => setTimeout(r, attempt * 1000));
        }
      }
    }
    throw lastError;
  }

  // Run promises with concurrency limit
  async function runWithConcurrency(tasks, concurrency = 3) {
    const results = [];
    const executing = new Set();

    for (const task of tasks) {
      const promise = Promise.resolve().then(() => task());
      results.push(promise);
      executing.add(promise);

      const cleanup = () => executing.delete(promise);
      promise.then(cleanup, cleanup);

      if (executing.size >= concurrency) {
        await Promise.race(executing);
      }
    }

    return Promise.all(results);
  }

  // Check if language is Greek
  function isGreekLanguage(langCode) {
    return langCode === 'el' || langCode === 'el_GR' || langCode.toLowerCase() === 'greek';
  }

  // Translate text using OpenAI API
  async function translateWithOpenAI(text, targetLanguage, apiKey, model, reasoning, verbosity, pageTitle = '', isSlug = false, langCode = '') {
    let systemPrompt = `You are a professional translator working for PureThemes.net, a WordPress theme developer company.`;

    if (pageTitle) {
      systemPrompt += ` You are translating website content from this page: "${pageTitle}".`;
    }

    systemPrompt += ` Your translations should be:
- Tech-savvy and accurate for WordPress/web development context
- Natural and fluent, avoiding clunky or literal translations
- Similar in length to the source text
- Preserving ALL HTML tags exactly as they appear in the source (like <strong>, <em>, <a>, etc.)

IMPORTANT: Return ONLY the translated text. Do not add any explanations, notes, or commentary.`;

    // Special instruction for Greek permalinks/slugs - use Greeklish (Latin characters)
    if (isSlug && isGreekLanguage(langCode)) {
      systemPrompt += `

CRITICAL: You are translating a URL permalink/slug. For Greek translations, you MUST use "Greeklish" (Greek words written in Latin/ASCII characters) instead of Greek alphabet characters. This ensures URL compatibility.
Examples:
- "about-us" → "sxetika-me-emas" (NOT "σχετικά-με-εμάς")
- "contact" → "epikoinonia" (NOT "επικοινωνία")
- "services" → "ypiresies" (NOT "υπηρεσίες")
Use standard Greeklish transliteration rules.`;
    }

    const userPrompt = `Translate the following text to ${targetLanguage}. Keep all HTML tags intact and in the same positions:

${text}`;

    // Build request body based on model family
    let requestBody;

    if (isGPT5Family(model)) {
      // GPT-5/5.1 API format
      requestBody = {
        model: model,
        messages: [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: userPrompt }
        ],
        max_completion_tokens: 2000,
        verbosity: verbosity || 'medium'
      };

      // Add reasoning parameter
      // GPT-5 Mini requires reasoning (doesn't support 'none'), use 'minimal' instead
      if (reasoning) {
        if (model === 'gpt-5-mini' && reasoning === 'none') {
          requestBody.reasoning_effort = 'minimal';
        } else {
          requestBody.reasoning_effort = reasoning;
        }
      }
    } else {
      // GPT-4 family API format (legacy)
      requestBody = {
        model: model,
        messages: [
          { role: 'system', content: systemPrompt },
          { role: 'user', content: userPrompt }
        ],
        temperature: 1,
        max_tokens: 2000
      };
    }

    const response = await fetchWithRetry('https://api.openai.com/v1/chat/completions', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'Authorization': `Bearer ${apiKey}`
      },
      body: JSON.stringify(requestBody)
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error?.message || 'API request failed');
    }

    const data = await response.json();
    return data.choices[0].message.content.trim();
  }

  // Translate text using Google Gemini API
  async function translateWithGemini(text, targetLanguage, apiKey, model, reasoning, verbosity, pageTitle = '', isSlug = false, langCode = '') {
    let systemPrompt = `You are a professional translator working for PureThemes.net, a WordPress theme developer company.`;

    if (pageTitle) {
      systemPrompt += ` You are translating website content from this page: "${pageTitle}".`;
    }

    systemPrompt += ` Your translations should be:
- Tech-savvy and accurate for WordPress/web development context
- Natural and fluent, avoiding clunky or literal translations
- Similar in length to the source text
- Preserving ALL HTML tags exactly as they appear in the source (like <strong>, <em>, <a>, etc.)

IMPORTANT: Return ONLY the translated text. Do not add any explanations, notes, or commentary.`;

    // Special instruction for Greek permalinks/slugs - use Greeklish (Latin characters)
    if (isSlug && isGreekLanguage(langCode)) {
      systemPrompt += `

CRITICAL: You are translating a URL permalink/slug. For Greek translations, you MUST use "Greeklish" (Greek words written in Latin/ASCII characters) instead of Greek alphabet characters. This ensures URL compatibility.
Examples:
- "about-us" → "sxetika-me-emas" (NOT "σχετικά-με-εμάς")
- "contact" → "epikoinonia" (NOT "επικοινωνία")
- "services" → "ypiresies" (NOT "υπηρεσίες")
Use standard Greeklish transliteration rules.`;
    }

    const userPrompt = `Translate the following text to ${targetLanguage}. Keep all HTML tags intact and in the same positions:

${text}`;

    // Map reasoning level to Gemini thinking config
    const thinkingLevelMap = {
      'none': 'none',
      'low': 'low',
      'medium': 'medium',
      'high': 'high'
    };

    // Build Gemini API request
    const requestBody = {
      contents: [{
        parts: [{ text: systemPrompt + '\n\n' + userPrompt }]
      }]
    };

    // Add thinking config for reasoning (Gemini 3 feature)
    if (reasoning && reasoning !== 'none') {
      requestBody.generationConfig = {
        thinkingConfig: {
          thinkingLevel: thinkingLevelMap[reasoning] || 'low'
        }
      };
    }

    // Gemini model names mapping
    const geminiModelMap = {
      'gemini-3-pro': 'gemini-3-pro-preview',
      'gemini-3-flash': 'gemini-3-flash-preview'
    };

    const apiModel = geminiModelMap[model] || model;
    const apiUrl = `https://generativelanguage.googleapis.com/v1beta/models/${apiModel}:generateContent?key=${apiKey}`;

    const response = await fetchWithRetry(apiUrl, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(requestBody)
    });

    if (!response.ok) {
      const error = await response.json();
      throw new Error(error.error?.message || 'Gemini API request failed');
    }

    const data = await response.json();

    // Extract text from Gemini response format
    if (data.candidates && data.candidates[0] && data.candidates[0].content) {
      const parts = data.candidates[0].content.parts;
      // Filter out thinking parts if present, get only text parts
      const textParts = parts.filter(p => p.text && !p.thought);
      if (textParts.length > 0) {
        return textParts.map(p => p.text).join('').trim();
      }
    }

    throw new Error('Unexpected Gemini API response format');
  }

  // Set value on textarea and trigger events
  function setTextareaValue(textarea, value) {
    // Focus the element first
    textarea.focus();

    // 1. Try native setter (React 15/16+)
    const nativeInputValueSetter = Object.getOwnPropertyDescriptor(
      window.HTMLTextAreaElement.prototype, 'value'
    ).set;
    
    if (nativeInputValueSetter) {
      nativeInputValueSetter.call(textarea, value);
    } else {
      textarea.value = value;
    }

    textarea.setAttribute('value', value);

    // 2. Dispatch events
    // Some frameworks listen to 'input', others 'change', some 'keydown'/'keyup'
    const events = ['keydown', 'keypress', 'input', 'keyup', 'change'];
    events.forEach(eventType => {
      const event = new Event(eventType, { bubbles: true, cancelable: true });
      textarea.dispatchEvent(event);
    });

    // 3. Vue.js specific
    if (textarea.__vue__) {
      textarea.__vue__.$emit('input', value);
    }

    // 4. Blur to finalize
    textarea.blur();
  }

  // Create the Translate All button for save container
  function createTranslateAllButton() {
    const btn = document.createElement('button');
    btn.className = 'trp-ai-translate-all-btn';
    btn.id = 'trp-ai-translate-all';
    btn.innerHTML = `
      <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
        <path d="M12.87 15.07l-2.54-2.51.03-.03A17.52 17.52 0 0014.07 6H17V4h-7V2H8v2H1v2h11.17C11.5 7.92 10.44 9.75 9 11.35 8.07 10.32 7.3 9.19 6.69 8h-2c.73 1.63 1.73 3.17 2.98 4.56l-5.09 5.02L4 19l5-5 3.11 3.11.76-2.04z"/>
        <path d="M18.5 10l-4.5 12h2.1l1.12-3h4.56l1.12 3H25l-4.5-12h-2zm-1.68 7l1.68-4.68L20.18 17h-3.36z"/>
      </svg>
      <span>Translate All</span>
    `;
    btn.type = 'button';
    return btn;
  }

  // Create global status indicator (top-right corner)
  function createGlobalStatus() {
    const status = document.createElement('div');
    status.className = 'trp-ai-global-status';
    status.id = 'trp-ai-global-status';
    return status;
  }

  // Create blocking overlay
  function createBlockingOverlay() {
    const overlay = document.createElement('div');
    overlay.id = 'trp-ai-blocking-overlay';
    overlay.style.cssText = `
      position: fixed;
      top: 0;
      left: 0;
      width: 100%;
      height: 100%;
      background: rgba(255, 255, 255, 0.3);
      z-index: 999999;
      display: none;
    `;
    return overlay;
  }

  // Toggle blocking overlay
  function toggleOverlay(show) {
    let overlay = document.getElementById('trp-ai-blocking-overlay');
    if (!overlay) {
      overlay = createBlockingOverlay();
      document.body.appendChild(overlay);
    }
    overlay.style.display = show ? 'block' : 'none';
  }

  // Create individual language translate button
  function createSingleTranslateButton(langCode, langName) {
    const btn = document.createElement('button');
    btn.className = 'trp-ai-translate-single-btn';
    btn.innerHTML = `Translate`;
    btn.type = 'button';
    btn.title = `Translate to ${langName}`;
    btn.dataset.lang = langCode;
    return btn;
  }

  // Add status indicator
  function createStatusIndicator() {
    const status = document.createElement('div');
    status.className = 'trp-ai-status';
    status.style.display = 'none';
    return status;
  }

  // Inject translate icon into trp-span (parent of hover container)
  function injectHoverIcon(hoverContainer) {
    // Get the parent trp-span element
    const trpSpan = hoverContainer.closest('trp-span');
    if (!trpSpan || trpSpan.querySelector('.trp-ai-translate-icon')) return;

    const icon = document.createElement('div');
    icon.className = 'trp-ai-translate-icon';
    icon.title = 'AI Translate All Languages';

    // Insert after the hover container
    trpSpan.appendChild(icon);

    icon.addEventListener('click', async (e) => {
      e.preventDefault();
      e.stopPropagation();

      const globalStatus = document.getElementById('trp-ai-global-status');
      const currentSourceText = getCurrentSourceText();

      if (!currentSourceText) {
        if (globalStatus) {
          globalStatus.textContent = 'No source text found';
          globalStatus.className = 'trp-ai-global-status error';
          globalStatus.style.display = 'block';
          setTimeout(() => { globalStatus.style.display = 'none'; }, 3000);
        }
        return;
      }

      const targetLanguages = getTargetLanguages();
      if (globalStatus) {
        await handleTranslation(currentSourceText, targetLanguages, globalStatus);
      } else {
        console.error('Global status element not found');
      }
    });
  }

  // Main translation handler - PARALLEL version
  async function handleTranslation(sourceText, targetLanguages, statusEl, singleLang = null, autoSave = true) {
    console.log('TRP AI: handleTranslation started');

    // Check if we're translating a slug
    const translatingSlug = isPostSlug();

    // Show blocking overlay
    toggleOverlay(true);

    let settings;
    try {
      settings = await new Promise((resolve, reject) => {
        try {
          chrome.storage.sync.get(['openaiApiKey', 'geminiApiKey', 'openaiModel', 'openaiReasoning', 'openaiVerbosity'], (items) => {
            if (chrome.runtime.lastError) {
              reject(chrome.runtime.lastError);
            } else {
              resolve(items);
            }
          });
        } catch (e) {
          reject(e);
        }
      });
    } catch (error) {
      console.error('Extension context invalidated or storage error:', error);
      statusEl.textContent = 'Extension updated. Please refresh the page.';
      statusEl.className = 'trp-ai-status error';
      statusEl.style.display = 'block';
      toggleOverlay(false); // Hide overlay on error
      return;
    }

    const model = settings.openaiModel || 'gpt-5.2';
    const isGemini = isGeminiFamily(model);
    const apiKey = isGemini ? settings.geminiApiKey : settings.openaiApiKey;

    if (!apiKey) {
      statusEl.textContent = `Please set your ${isGemini ? 'Gemini' : 'OpenAI'} API key in the extension popup`;
      statusEl.className = 'trp-ai-status error';
      statusEl.style.display = 'block';
      toggleOverlay(false); // Hide overlay on error
      return;
    }
    const reasoning = settings.openaiReasoning || 'none';
    const verbosity = settings.openaiVerbosity || 'medium';
    const pageTitle = getPageTitle();
    console.log('TranslatePress AI Context - Page Title:', pageTitle);
    const langsToTranslate = singleLang
      ? targetLanguages.filter(l => l.code === singleLang)
      : targetLanguages;

    // Determine if using global or local status
    const isGlobal = statusEl.id === 'trp-ai-global-status';
    const baseClass = isGlobal ? 'trp-ai-global-status' : 'trp-ai-status';

    statusEl.style.display = 'block';
    statusEl.textContent = `Translating to ${langsToTranslate.length} language(s)...`;
    statusEl.className = `${baseClass} loading`;

    let completedCount = 0;
    let errorCount = 0;

    // Create translation tasks (functions that return promises)
    const translationTasks = langsToTranslate.map((lang) => async () => {
      try {
        // Call the appropriate translation API
        const translateFn = isGemini ? translateWithGemini : translateWithOpenAI;
        const translation = await translateFn(
          sourceText,
          lang.name,
          apiKey,
          model,
          reasoning,
          verbosity,
          pageTitle,
          translatingSlug,
          lang.code
        );

        setTextareaValue(lang.textarea, translation);

        // Visual feedback
        lang.textarea.style.backgroundColor = '#d4edda';
        setTimeout(() => {
          lang.textarea.style.backgroundColor = '';
        }, 1000);

        completedCount++;
        statusEl.textContent = `Translating... (${completedCount}/${langsToTranslate.length} done)`;

        return { success: true, lang: lang.name };
      } catch (error) {
        console.error(`Translation error for ${lang.name}:`, error);
        errorCount++;

        // Visual feedback for error
        lang.textarea.style.backgroundColor = '#f8d7da';
        setTimeout(() => {
          lang.textarea.style.backgroundColor = '';
        }, 2000);

        return { success: false, lang: lang.name, error: error.message };
      }
    });

    // Run translations with concurrency limit (10 at a time to balance speed vs reliability)
    const results = await runWithConcurrency(translationTasks, 10);

    // Show final status
    if (errorCount > 0) {
      statusEl.textContent = `Done! ${completedCount} translated, ${errorCount} failed`;
      statusEl.className = `${baseClass} error`;
      toggleOverlay(false); // Hide overlay if there were errors
    } else {
      if (autoSave) {
        statusEl.textContent = `Done! Translated to ${langsToTranslate.length} language(s). Saving...`;
        statusEl.className = `${baseClass} success`;

        // Auto-trigger Save button
        const saveBtn = document.getElementById('trp-save');
        if (saveBtn) {
          setTimeout(() => {
            saveBtn.click();
            statusEl.textContent = `Done! Translated & saved ${langsToTranslate.length} language(s)`;
            toggleOverlay(false); // Hide overlay after save click
          }, 500);
        } else {
          toggleOverlay(false); // Hide overlay if no save button
        }
      } else {
        statusEl.textContent = `Done! Translated to ${langsToTranslate.length} language(s).`;
        statusEl.className = `${baseClass} success`;
        toggleOverlay(false); // Hide overlay if no auto-save
      }
    }

    setTimeout(() => {
      statusEl.style.display = 'none';
    }, 3000);
  }

  // Check if current item is a Post Slug
  function isPostSlug() {
    const categoryContainer = document.getElementById('select2-trp-string-categories-container');
    return categoryContainer && categoryContainer.title === 'Post Slug';
  }

  // Initialize the extension
  function init() {
    if (!isTranslatePressEditor()) {
      return;
    }

    console.log('TranslatePress AI Translator: Editor detected');

    // Watch for DOM changes (TranslatePress loads content dynamically)
    const observer = new MutationObserver((mutations) => {
      injectButtons();
    });

    observer.observe(document.body, {
      childList: true,
      subtree: true
    });

    // Initial injection
    setTimeout(injectButtons, 1000);

    // Listen for clicks on hover container buttons to inject our translate button after panel loads
    document.addEventListener('click', (e) => {
      const clickedIcon = e.target.closest('.trp-icon');
      if (clickedIcon && clickedIcon.closest('.trp-editor-action-hover-container')) {
        console.log('Hover icon clicked, waiting for panel to load...');
        // Wait for the translation panel to load, then inject button
        setTimeout(() => {
          injectButtons();
        }, 500);
      }

      // Global listener for Merge/Split buttons (Capture phase)
      const mergeOrSplitBtn = e.target.closest('.trp-merge, .trp-split');
      if (mergeOrSplitBtn) {
        console.log('TRP AI: Merge/Split detected via global listener');
        
        // Check if it's a slug - if so, don't auto-save
        if (isPostSlug()) {
          return;
        }

        // Mark that we need a save before next translation
        needsSaveBeforeTranslate = true;
        console.log('TRP AI: Flagged for save before next translation');
      }
    }, true);

    // Continuously check for hover containers and inject translate button
    setInterval(() => {
      const hoverContainers = document.querySelectorAll('.trp-editor-action-hover-container');
      hoverContainers.forEach(hoverContainer => {
        if (!hoverContainer.querySelector('.trp-ai-translate-icon')) {
          const icon = document.createElement('div');
          icon.className = 'trp-ai-translate-icon';
          icon.title = 'AI Translate All Languages';
          icon.setAttribute('style', 'display: none !important;');
          icon.innerHTML = `<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="25" height="25" fill="#111111" style="display:block;margin:auto;padding:2px;margin-top:3px;"><path d="M12.65 15.67c.14-.36.05-.77-.23-1.05l-2.09-2.06.03-.03A17.52 17.52 0 0014.07 6h1.94c.54 0 .99-.45.99-.99v-.02c0-.54-.45-.99-.99-.99H10V3c0-.55-.45-1-1-1s-1 .45-1 1v1H1.99c-.54 0-.99.45-.99.99 0 .55.45.99.99.99h10.18C11.5 7.92 10.44 9.75 9 11.35c-.81-.89-1.49-1.86-2.06-2.88-.16-.29-.45-.47-.78-.47-.69 0-1.13.75-.79 1.35.63 1.13 1.4 2.21 2.3 3.21L3.3 16.87c-.4.39-.4 1.03 0 1.42.39.39 1.02.39 1.42 0L9 14l2.02 2.02c.51.51 1.38.32 1.63-.35zM17.5 10c-.6 0-1.14.37-1.35.94l-3.67 9.8c-.24.61.22 1.26.87 1.26.39 0 .74-.24.88-.61l.89-2.39h4.75l.9 2.39c.14.36.49.61.88.61.65 0 1.11-.65.88-1.26l-3.67-9.8c-.22-.57-.76-.94-1.36-.94zm-1.69 7l1.69-4.5 1.69 4.5h-3.38z"/></svg>`;

          hoverContainer.appendChild(icon);

          // Show icon only when Edit or Split is clicked
          const editBtn = hoverContainer.querySelector('.trp-edit-translation');
          if (editBtn) {
            editBtn.addEventListener('click', () => {
              icon.style.setProperty('display', 'inline-block', 'important');
            });
          }

          const splitBtn = hoverContainer.querySelector('.trp-split');
          if (splitBtn) {
            splitBtn.addEventListener('click', () => {
              icon.style.setProperty('display', 'inline-block', 'important');
            });
          }

          const mergeBtn = hoverContainer.querySelector('.trp-merge');
          if (mergeBtn) {
            mergeBtn.addEventListener('click', () => {
              icon.style.setProperty('display', 'inline-block', 'important');
            });
          }

          icon.addEventListener('click', async (e) => {
            e.preventDefault();
            e.stopPropagation();

            // Check local pending save flag (Iframe scope)
            if (needsSaveBeforeTranslate) {
              console.log('TRP AI: Hover icon clicked, need save first.');
              // We let the parent button handle the saving logic
            }

            // Trigger the main Translate All button in parent frame
            try {
              let mainBtn = document.getElementById('trp-ai-translate-all');
              if (!mainBtn && window.parent && window.parent !== window) {
                mainBtn = window.parent.document.getElementById('trp-ai-translate-all');
              }
              if (mainBtn) {
                mainBtn.click();
              }
            } catch (err) {
              console.log('Could not access parent frame:', err);
            }
          });
        }

        // Add auto-save to merge button
        const mergeBtn = hoverContainer.querySelector('.trp-merge');
        if (mergeBtn && !mergeBtn.dataset.autosaveAttached) {
          mergeBtn.dataset.autosaveAttached = 'true';
          // Listener removed - handled by global listener
        }

        // Add auto-save to split button
        const splitBtn = hoverContainer.querySelector('.trp-split');
        if (splitBtn && !splitBtn.dataset.autosaveAttached) {
          splitBtn.dataset.autosaveAttached = 'true';
          // Listener removed - handled by global listener
        }
      });
    }, 100);
  }

  // Inject buttons into the TranslatePress interface
  function injectButtons() {
    // Inject global status indicator if not present
    if (!document.getElementById('trp-ai-global-status')) {
      const globalStatus = createGlobalStatus();
      document.body.appendChild(globalStatus);
    }

    // Inject Translate All button above #trp-translation-section
    const translationSection = document.getElementById('trp-translation-section');
    if (translationSection && !document.getElementById('trp-ai-translate-all')) {
      const btn = createTranslateAllButton();
      const globalStatus = document.getElementById('trp-ai-global-status');

      // Insert button before the translation section
      translationSection.parentNode.insertBefore(btn, translationSection);

      btn.addEventListener('click', async (e) => {
        e.preventDefault();
        e.stopPropagation();

        // Check for Post Slug safety
        let autoSave = true;
        if (isPostSlug()) {
          if (!confirm('Warning: You are translating a Post Slug. This can break links. Auto-save will be disabled. Continue?')) {
            return;
          }
          autoSave = false;
        }

        // Check if we need to save first (due to previous merge/split)
        if (needsSaveBeforeTranslate) {
          console.log('TRP AI: Pending merge/split detected. Saving first...');
          btn.innerHTML = `<span class="trp-ai-spinner"></span> Saving changes...`;
          btn.disabled = true;
          toggleOverlay(true); // Block UI

          // Trigger Save
          let saveBtn = document.getElementById('trp-save');
          if (!saveBtn && window.parent && window.parent !== window) {
            try {
              saveBtn = window.parent.document.getElementById('trp-save');
            } catch (e) {}
          }

          if (saveBtn) {
            saveBtn.click();
            // Wait for save to complete (approx 2.5s)
            await new Promise(r => setTimeout(r, 2500));
            needsSaveBeforeTranslate = false;
          } else {
            console.error('TRP AI: Could not find save button');
            toggleOverlay(false);
            btn.disabled = false;
            return;
          }
        }

        // Get source text dynamically at click time
        const currentSourceText = getCurrentSourceText();

        if (!currentSourceText) {
          globalStatus.textContent = 'No source text found';
          globalStatus.className = 'trp-ai-global-status error';
          globalStatus.style.display = 'block';
          setTimeout(() => { globalStatus.style.display = 'none'; }, 3000);
          return;
        }

        btn.disabled = true;
        btn.innerHTML = `<span class="trp-ai-spinner"></span> Translating...`;

        const targetLanguages = getTargetLanguages();
        if (globalStatus) {
          await handleTranslation(currentSourceText, targetLanguages, globalStatus, null, autoSave);
        } else {
          console.error('Global status element not found');
        }

        btn.disabled = false;
        btn.innerHTML = `
          <svg width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
            <path d="M12.87 15.07l-2.54-2.51.03-.03A17.52 17.52 0 0014.07 6H17V4h-7V2H8v2H1v2h11.17C11.5 7.92 10.44 9.75 9 11.35 8.07 10.32 7.3 9.19 6.69 8h-2c.73 1.63 1.73 3.17 2.98 4.56l-5.09 5.02L4 19l5-5 3.11 3.11.76-2.04z"/>
            <path d="M18.5 10l-4.5 12h2.1l1.12-3h4.56l1.12 3H25l-4.5-12h-2zm-1.68 7l1.68-4.68L20.18 17h-3.36z"/>
          </svg>
          <span>Translate All</span>
        `;
      });
    }

    // Add individual translate buttons to each target language
    const targetContainers = document.querySelectorAll('[id^="trp-language-"]');
    targetContainers.forEach(container => {
      const langCode = container.id.replace('trp-language-', '');

      // Skip the source language (dynamically detected)
      if (isSourceLanguage(langCode)) return;

      const header = container.querySelector('.trp-language-name');
      if (!header || header.querySelector('.trp-ai-translate-single-btn')) return;

      const textarea = container.querySelector('textarea.trp-translation-input:not([readonly])');
      if (!textarea) return;

      const langName = LANGUAGE_NAMES[langCode] || langCode;
      const btn = createSingleTranslateButton(langCode, langName);
      const status = createStatusIndicator();

      const wrapper = document.createElement('div');
      wrapper.className = 'trp-ai-wrapper';
      wrapper.appendChild(btn);
      wrapper.appendChild(status);

      header.appendChild(wrapper);

      btn.addEventListener('click', async (e) => {
        e.preventDefault();
        e.stopPropagation();

        // Check for Post Slug safety
        if (isPostSlug()) {
          if (!confirm('Warning: You are translating a Post Slug. This can break links. Continue?')) {
            return;
          }
        }

        // Check if we need to save first (due to previous merge/split)
        if (needsSaveBeforeTranslate) {
          console.log('TRP AI: Pending merge/split detected. Saving first...');
          btn.textContent = 'Saving...';
          btn.disabled = true;
          toggleOverlay(true); // Block UI

          // Trigger Save
          let saveBtn = document.getElementById('trp-save');
          if (!saveBtn && window.parent && window.parent !== window) {
            try {
              saveBtn = window.parent.document.getElementById('trp-save');
            } catch (e) {}
          }

          if (saveBtn) {
            saveBtn.click();
            // Wait for save to complete (approx 2.5s)
            await new Promise(r => setTimeout(r, 2500));
            needsSaveBeforeTranslate = false;
          } else {
            console.error('TRP AI: Could not find save button');
            toggleOverlay(false);
            btn.disabled = false;
            return;
          }
        }

        // Get source text dynamically at click time
        const sourceText = getCurrentSourceText();

        if (!sourceText) {
          status.textContent = 'Could not find source text';
          status.className = 'trp-ai-status error';
          status.style.display = 'block';
          setTimeout(() => { status.style.display = 'none'; }, 3000);
          return;
        }

        btn.disabled = true;
        btn.textContent = '...';

        const targetLanguages = [{
          code: langCode,
          name: langName,
          textarea: textarea,
          container: container
        }];

        await handleTranslation(sourceText, targetLanguages, status, langCode, false);

        btn.disabled = false;
        btn.textContent = 'Translate';
      });
    });
  }

  // Start when DOM is ready
  console.log('TRP AI: Script loaded');
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
