<?php
/**
 * WooCommerce Product Content Extractor
 *
 * Extracts content from WooCommerce 'product' post type for AI embeddings.
 *
 * @package Listeo_AI_Search
 * @since 2.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Content_Extractor_Product {

    /**
     * Extract content from WooCommerce product for embedding generation
     *
     * @param int $post_id Product ID
     * @return string Structured content for embedding
     */
    public function extract_content($post_id) {
        $post = get_post($post_id);

        if (!$post || $post->post_type !== 'product') {
            return '';
        }

        // Check if WooCommerce is active
        if (!function_exists('wc_get_product')) {
            return $this->extract_basic_content($post);
        }

        $product = wc_get_product($post_id);
        if (!$product) {
            return $this->extract_basic_content($post);
        }

        $structured_content = "";

        // Product name
        $structured_content .= "PRODUCT_NAME: " . $product->get_name() . ". ";

        // Product type
        $structured_content .= "PRODUCT_TYPE: " . $product->get_type() . ". ";

        // Description (full)
        $description = $product->get_description();
        if (!empty($description)) {
            $description = strip_tags($description);
            $description = preg_replace('/\s+/', ' ', $description);
            $structured_content .= "DESCRIPTION: " . trim($description) . ". ";
        }

        // Short description
        $short_description = $product->get_short_description();
        if (!empty($short_description)) {
            $short_description = strip_tags($short_description);
            $structured_content .= "SHORT_DESCRIPTION: " . trim($short_description) . ". ";
        }

        // Price information
        if ($product->get_price()) {
            $regular_price = $product->get_regular_price();
            $sale_price = $product->get_sale_price();

            if ($sale_price) {
                $structured_content .= "PRICE: Regular {$regular_price}, Sale {$sale_price}. ";
            } else {
                $structured_content .= "PRICE: {$regular_price}. ";
            }
        }

        // SKU
        $sku = $product->get_sku();
        if ($sku) {
            $structured_content .= "SKU: " . $sku . ". ";
        }

        // Categories
        $categories = wp_get_post_terms($post_id, 'product_cat', array('fields' => 'names'));
        if (!is_wp_error($categories) && !empty($categories)) {
            $structured_content .= "CATEGORIES: " . implode(', ', $categories) . ". ";
        }

        // Tags
        $tags = wp_get_post_terms($post_id, 'product_tag', array('fields' => 'names'));
        if (!is_wp_error($tags) && !empty($tags)) {
            $structured_content .= "TAGS: " . implode(', ', $tags) . ". ";
        }

        // Attributes
        $attributes = $product->get_attributes();
        if (!empty($attributes)) {
            $attr_strings = array();
            foreach ($attributes as $attribute) {
                if (is_a($attribute, 'WC_Product_Attribute')) {
                    $name = wc_attribute_label($attribute->get_name());
                    $options = $attribute->get_options();
                    if (!empty($options)) {
                        if (is_array($options)) {
                            $attr_strings[] = $name . ': ' . implode(', ', $options);
                        } else {
                            $attr_strings[] = $name . ': ' . $options;
                        }
                    }
                }
            }
            if (!empty($attr_strings)) {
                $structured_content .= "ATTRIBUTES: " . implode('; ', $attr_strings) . ". ";
            }
        }

        // Stock status
        if ($product->is_in_stock()) {
            $structured_content .= "AVAILABILITY: In Stock. ";
        } else {
            $structured_content .= "AVAILABILITY: Out of Stock. ";
        }

        // Featured image alt text
        $thumbnail_id = get_post_thumbnail_id($post_id);
        if ($thumbnail_id) {
            $alt_text = get_post_meta($thumbnail_id, '_wp_attachment_image_alt', true);
            if (!empty($alt_text)) {
                $structured_content .= "IMAGE_CONTEXT: " . $alt_text . ". ";
            }
        }

        // Limit total length
        if (strlen($structured_content) > 8000) {
            $structured_content = substr($structured_content, 0, 8000);
        }

        return trim($structured_content);
    }

    /**
     * Extract basic content if WooCommerce functions aren't available
     *
     * @param WP_Post $post Post object
     * @return string Basic structured content
     */
    private function extract_basic_content($post) {
        $structured_content = "";

        $structured_content .= "TITLE: " . get_the_title($post->ID) . ". ";

        if (!empty($post->post_content)) {
            $content = strip_tags($post->post_content);
            $content = preg_replace('/\s+/', ' ', $content);
            $structured_content .= "CONTENT: " . trim($content) . ". ";
        }

        if (!empty($post->post_excerpt)) {
            $excerpt = strip_tags($post->post_excerpt);
            $structured_content .= "EXCERPT: " . trim($excerpt) . ". ";
        }

        return trim($structured_content);
    }
}
