<?php
/**
 * Content Extractor Factory
 *
 * Routes post types to their appropriate content extractors.
 *
 * @package Listeo_AI_Search
 * @since 2.0.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Content_Extractor_Factory {

    /**
     * Get the appropriate extractor for a post type
     *
     * @param string $post_type Post type slug
     * @return object Content extractor instance
     */
    public static function get_extractor($post_type) {
        switch ($post_type) {
            case 'listing':
                return new Listeo_AI_Content_Extractor_Listing();

            case 'post':
                return new Listeo_AI_Content_Extractor_Post();

            case 'page':
                return new Listeo_AI_Content_Extractor_Page();

            case 'product':
                return new Listeo_AI_Content_Extractor_Product();

            default:
                return new Listeo_AI_Content_Extractor_Default();
        }
    }

    /**
     * Get extractor for a specific post ID
     *
     * @param int $post_id Post ID
     * @return object|false Content extractor instance or false if post not found
     */
    public static function get_extractor_for_post($post_id) {
        $post = get_post($post_id);
        if (!$post) {
            return false;
        }

        return self::get_extractor($post->post_type);
    }

    /**
     * Extract content using the appropriate extractor
     *
     * @param int $post_id Post ID
     * @return string Extracted content or empty string on failure
     */
    public static function extract_content($post_id) {
        $extractor = self::get_extractor_for_post($post_id);

        if (!$extractor) {
            return '';
        }

        return $extractor->extract_content($post_id);
    }
}
