<?php
/**
 * Pro Features Manager
 *
 * Handles detection and limitations for Pro features
 * Provides hooks for Pro version to override
 *
 * ============================================
 * 🔗 CENTRALIZED PRO URLS - CHANGE HERE ONLY!
 * ============================================
 * All Pro upgrade links throughout the plugin use these constants.
 * Change the URLs below to update all links across the plugin.
 *
 * @package AI_Chat_Search
 * @since 1.1.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class AI_Chat_Search_Pro_Manager {

    /**
     * 🔗 Pro Upgrade URL
     *
     * Change this URL to update ALL "Upgrade to Pro" links throughout the plugin.
     * This URL is used everywhere a locked feature shows an upgrade link.
     */
    const PRO_UPGRADE_URL = 'https://purethemes.net/ai-chat-search-pro/';

    /**
     * 🔗 Pro Features URL
     *
     * Change this URL to update ALL "Learn More" and feature documentation links.
     * Currently not used in UI but available for future use.
     */
    const PRO_FEATURES_URL = 'https://purethemes.net/ai-chat-search-pro/';

    /**
     * Check if Pro version is active
     *
     * @return bool True if Pro plugin is active
     */
    public static function is_pro_active() {
        return apply_filters('ai_chat_search_pro_active', false);
    }

    /**
     * Check if post type is locked in free version
     *
     * @param string $post_type Post type slug
     * @return bool True if locked, false if available
     */
    public static function is_post_type_locked($post_type) {
        // Pro version can override to unlock all post types
        $is_locked = self::get_default_locked_status($post_type);

        return apply_filters('ai_chat_search_post_type_locked', $is_locked, $post_type);
    }

    /**
     * Can user access full conversation logs?
     *
     * @return bool True if allowed, false if locked
     */
    public static function can_access_conversation_logs() {
        // Pro can override to return true
        return apply_filters('ai_chat_search_can_access_conversation_logs', false);
    }

    /**
     * Get upgrade URL for specific feature
     *
     * @param string $feature Feature slug for tracking
     * @return string Upgrade URL
     */
    public static function get_upgrade_url($feature = '') {
        $url = apply_filters('ai_chat_search_upgrade_url', self::PRO_UPGRADE_URL);

        if ($feature) {
            $url = add_query_arg('feature', $feature, $url);
        }

        return $url;
    }

    /**
     * Get "Learn More" URL for Pro features
     *
     * @return string Learn more URL
     */
    public static function get_learn_more_url() {
        return apply_filters('ai_chat_search_learn_more_url', self::PRO_FEATURES_URL);
    }

    /**
     * Get default locked status for post type in free version
     *
     * FREE VERSION RULES:
     * - Listeo theme: ONLY listing = unlimited | post, page, products, custom = locked
     * - Generic theme: post, page = unlimited | products, custom = locked
     *
     * @param string $post_type Post type slug
     * @return bool True if locked, false if available
     */
    private static function get_default_locked_status($post_type) {
        $is_listeo = self::is_listeo_theme();

        // Listeo theme: ONLY listings are free, everything else is locked
        if ($is_listeo) {
            if ($post_type === 'listing') {
                return false; // Unlimited - not locked
            }
            // Everything else is locked in Listeo (including posts and pages)
            return true;
        }

        // Generic theme: posts and pages are free
        if (in_array($post_type, ['post', 'page'])) {
            return false; // Unlimited - not locked
        }

        // Lock products and all custom post types in generic theme
        return true;
    }

    /**
     * Get available post types for free version
     *
     * @return array Array of available post type slugs
     */
    public static function get_free_available_post_types() {
        $is_listeo = self::is_listeo_theme();

        if ($is_listeo) {
            // Listeo: Only listings are free
            return ['listing'];
        }

        // Generic: Posts and pages are free
        return ['post', 'page'];
    }

    /**
     * Get locked post types in free version
     *
     * @return array Array of locked post type slugs
     */
    public static function get_locked_post_types() {
        $is_listeo = self::is_listeo_theme();

        $locked = ['product'];

        // Listeo theme: Also lock posts and pages
        if ($is_listeo) {
            $locked[] = 'post';
            $locked[] = 'page';
        }

        // Add all detected custom post types
        $custom_types = AI_Chat_Search_Database_Manager::get_detected_custom_post_types();
        foreach ($custom_types as $type_data) {
            $locked[] = $type_data['name'];
        }

        return apply_filters('ai_chat_search_locked_post_types', $locked);
    }

    /**
     * Detect if Listeo theme is active
     *
     * @return bool True if Listeo or Listeo child theme is active
     */
    private static function is_listeo_theme() {
        if (class_exists('Listeo_AI_Detection')) {
            return Listeo_AI_Detection::is_listeo_available();
        }

        // Fallback: Check theme name
        $current_theme = wp_get_theme();
        $theme_name = $current_theme->get('Name');
        $parent_theme = $current_theme->get('Template');

        return (stripos($theme_name, 'listeo') !== false ||
                stripos($parent_theme, 'listeo') !== false);
    }

    /**
     * Get Pro badge HTML
     *
     * @return string HTML for Pro badge
     */
    public static function get_pro_badge() {
        return '<span class="ai-chat-pro-badge" title="' .
               esc_attr__('Available in Pro version', 'listeo-ai-search') .
               '">PRO</span>';
    }

    /**
     * Get lock icon HTML
     *
     * @return string HTML for lock icon
     */
    public static function get_lock_icon() {
        return '<span class="dashicons dashicons-lock"></span>';
    }

    /**
     * Check if feature should show upgrade prompt
     *
     * @param string $feature Feature identifier
     * @return bool True if should show upgrade prompt
     */
    public static function should_show_upgrade_prompt($feature) {
        if (self::is_pro_active()) {
            return false;
        }

        // Don't show if user dismissed this feature's prompt
        $dismissed = get_user_meta(get_current_user_id(), 'ai_chat_dismissed_upgrade_prompts', true);
        if (!is_array($dismissed)) {
            $dismissed = [];
        }

        return !in_array($feature, $dismissed);
    }

    /**
     * Render upgrade prompt box
     *
     * @param array $args Arguments for rendering prompt
     */
    public static function render_upgrade_prompt($args = []) {
        $defaults = [
            'title' => __('Upgrade to Pro', 'listeo-ai-search'),
            'description' => '',
            'features' => [],
            'feature_id' => '',
            'show_learn_more' => true,
        ];

        $args = wp_parse_args($args, $defaults);

        $upgrade_url = self::get_upgrade_url($args['feature_id']);
        $learn_more_url = self::get_learn_more_url();

        ?>
        <div class="ai-chat-upgrade-prompt">
            <div class="upgrade-prompt-icon">
                <?php echo self::get_lock_icon(); ?>
            </div>

            <div class="upgrade-prompt-content">
                <h3><?php echo esc_html($args['title']); ?></h3>

                <?php if ($args['description']): ?>
                    <p class="description"><?php echo esc_html($args['description']); ?></p>
                <?php endif; ?>

                <?php if (!empty($args['features'])): ?>
                    <ul class="upgrade-features-list">
                        <?php foreach ($args['features'] as $feature): ?>
                            <li>
                                <span class="dashicons dashicons-yes-alt"></span>
                                <?php echo esc_html($feature); ?>
                            </li>
                        <?php endforeach; ?>
                    </ul>
                <?php endif; ?>

                <div class="upgrade-prompt-actions">
                    <a href="<?php echo esc_url($upgrade_url); ?>"
                       class="button button-primary button-large"
                       target="_blank">
                        <?php _e('Upgrade to Pro', 'listeo-ai-search'); ?>
                    </a>

                    <?php if ($args['show_learn_more']): ?>
                        <a href="<?php echo esc_url($learn_more_url); ?>"
                           class="button button-secondary"
                           target="_blank">
                            <?php _e('Learn More', 'listeo-ai-search'); ?>
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php
    }
}
