<?php
/**
 * Manual Batch Processor Class
 * 
 * Handles manual batch processing for large catalogs (>300 listings)
 * to avoid OpenAI API rate limiting issues.
 * 
 * @package Listeo_AI_Search
 * @since 1.2.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Search_Manual_Batch_Processor {
    
    /**
     * Threshold for switching to manual mode
     */
    const MANUAL_MODE_THRESHOLD = 1000;
    
    /**
     * Default batch size
     */
    const DEFAULT_BATCH_SIZE = 250;
    
    /**
     * Default delay between requests (milliseconds)
     */
    const DEFAULT_DELAY = 0;
    
    /**
     * Maximum retry attempts for failed batches
     */
    const MAX_RETRY_ATTEMPTS = 5;
    
    /**
     * Get batch tracking table name
     * 
     * @return string Table name
     */
    public static function get_batch_table_name() {
        global $wpdb;
        return $wpdb->prefix . 'listeo_ai_batch_tracking';
    }
    
    /**
     * Create batch tracking table
     */
    public static function create_batch_table() {
        global $wpdb;
        
        $table_name = self::get_batch_table_name();
        $charset_collate = $wpdb->get_charset_collate();
        
        $sql = "CREATE TABLE $table_name (
            batch_id varchar(32) NOT NULL,
            start_index int(11) NOT NULL,
            end_index int(11) NOT NULL,
            status enum('ready','queued','processing','complete','failed','partial') DEFAULT 'ready',
            processed_count int(11) DEFAULT 0,
            total_count int(11) DEFAULT 0,
            error_message text DEFAULT NULL,
            retry_count int(11) DEFAULT 0,
            last_attempt datetime DEFAULT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            completed_at datetime DEFAULT NULL,
            PRIMARY KEY (batch_id),
            KEY status (status),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
    
    /**
     * Check if manual mode should be enabled
     *
     * @return bool True if manual mode should be enabled
     */
    public static function should_use_manual_mode() {
        global $wpdb;

        // Get whitelisted enabled post types
        $enabled_post_types = Listeo_AI_Search_Database_Manager::get_enabled_post_types();

        // If no post types are enabled, manual mode is not needed
        if (empty($enabled_post_types)) {
            return false;
        }

        // Build post type condition
        $post_types_placeholders = implode(',', array_fill(0, count($enabled_post_types), '%s'));

        // Exclude listeo-booking products from count
        $exclude_booking_products = in_array('product', $enabled_post_types);

        if ($exclude_booking_products) {
            $count = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->posts} p
                WHERE p.post_type IN ($post_types_placeholders)
                AND p.post_status = 'publish'
                AND NOT EXISTS (
                    SELECT 1
                    FROM {$wpdb->term_relationships} tr
                    INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                    INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                    WHERE tr.object_id = p.ID
                    AND tt.taxonomy = 'product_cat'
                    AND t.slug = 'listeo-booking'
                )
            ", ...$enabled_post_types));
        } else {
            $count = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->posts}
                WHERE post_type IN ($post_types_placeholders)
                AND post_status = 'publish'
            ", ...$enabled_post_types));
        }

        return intval($count) > self::MANUAL_MODE_THRESHOLD;
    }

    /**
     * Get total published posts count (across all enabled post types)
     *
     * @return int Total count
     */
    public static function get_total_listings_count() {
        global $wpdb;

        // Get whitelisted enabled post types
        $enabled_post_types = Listeo_AI_Search_Database_Manager::get_enabled_post_types();

        // If no post types are enabled, return 0
        if (empty($enabled_post_types)) {
            return 0;
        }

        // Build post type condition
        $post_types_placeholders = implode(',', array_fill(0, count($enabled_post_types), '%s'));

        // Exclude listeo-booking products from count
        $exclude_booking_products = in_array('product', $enabled_post_types);

        if ($exclude_booking_products) {
            return intval($wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->posts} p
                WHERE p.post_type IN ($post_types_placeholders)
                AND p.post_status = 'publish'
                AND NOT EXISTS (
                    SELECT 1
                    FROM {$wpdb->term_relationships} tr
                    INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                    INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                    WHERE tr.object_id = p.ID
                    AND tt.taxonomy = 'product_cat'
                    AND t.slug = 'listeo-booking'
                )
            ", ...$enabled_post_types)));
        } else {
            return intval($wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->posts}
                WHERE post_type IN ($post_types_placeholders)
                AND post_status = 'publish'
            ", ...$enabled_post_types)));
        }
    }

    /**
     * Get total processed posts count (across all enabled post types)
     *
     * @return int Processed count
     */
    public static function get_processed_listings_count() {
        global $wpdb;

        $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();

        // Get whitelisted enabled post types
        $enabled_post_types = Listeo_AI_Search_Database_Manager::get_enabled_post_types();

        // If no post types are enabled, return 0
        if (empty($enabled_post_types)) {
            return 0;
        }

        // Build post type condition
        $post_types_placeholders = implode(',', array_fill(0, count($enabled_post_types), '%s'));

        // Exclude listeo-booking products from count
        $exclude_booking_products = in_array('product', $enabled_post_types);

        if ($exclude_booking_products) {
            return intval($wpdb->get_var($wpdb->prepare("
                SELECT COUNT(DISTINCT e.listing_id)
                FROM {$embeddings_table} e
                INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID
                WHERE p.post_type IN ($post_types_placeholders)
                AND p.post_status = 'publish'
                AND NOT EXISTS (
                    SELECT 1
                    FROM {$wpdb->term_relationships} tr
                    INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                    INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                    WHERE tr.object_id = p.ID
                    AND tt.taxonomy = 'product_cat'
                    AND t.slug = 'listeo-booking'
                )
            ", ...$enabled_post_types)));
        } else {
            return intval($wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$embeddings_table} e
                INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID
                WHERE p.post_type IN ($post_types_placeholders)
                AND p.post_status = 'publish'
            ", ...$enabled_post_types)));
        }
    }
    
    /**
     * Initialize batches based on current configuration
     *
     * @param int $batch_size Batch size
     * @return array Result with success status and message
     */
    public static function initialize_batches($batch_size = null) {
        global $wpdb;

        if ($batch_size === null) {
            $batch_size = self::DEFAULT_BATCH_SIZE;
        }

        $batch_size = max(10, min(250, intval($batch_size))); // Clamp between 10-250 (10 for debug purposes)

        // Get whitelisted enabled post types
        $enabled_post_types = Listeo_AI_Search_Database_Manager::get_enabled_post_types();

        // If no post types are enabled, return error
        if (empty($enabled_post_types)) {
            return array(
                'success' => false,
                'message' => 'No post types enabled for indexing'
            );
        }

        // Build post type condition
        $post_types_placeholders = implode(',', array_fill(0, count($enabled_post_types), '%s'));

        try {
            // Clear existing batches
            $batch_table = self::get_batch_table_name();
            $wpdb->query("DELETE FROM {$batch_table}");

            // Check for manual selections
            $manual_selections = get_option('listeo_ai_search_manual_selections', array());
            $has_manual_selection = !empty($manual_selections);

            if ($has_manual_selection) {
                // Build list of all manually selected post IDs across all post types
                $selected_ids = array();
                foreach ($manual_selections as $post_type => $ids) {
                    if (in_array($post_type, $enabled_post_types)) {
                        $selected_ids = array_merge($selected_ids, $ids);
                    }
                }

                if (empty($selected_ids)) {
                    return array(
                        'success' => false,
                        'message' => 'No posts in manual selection'
                    );
                }

                // Get only manually selected posts
                $placeholders = implode(',', array_fill(0, count($selected_ids), '%d'));
                $listings = $wpdb->get_results($wpdb->prepare("
                    SELECT ID
                    FROM {$wpdb->posts}
                    WHERE ID IN ($placeholders)
                    AND post_status = 'publish'
                    ORDER BY ID ASC
                ", ...$selected_ids));
            } else {
                // No manual selection - get all published posts across enabled post types
                // Exclude listeo-booking category products (hidden booking products)
                $exclude_booking_products = in_array('product', $enabled_post_types);

                if ($exclude_booking_products) {
                    $listings = $wpdb->get_results($wpdb->prepare("
                        SELECT p.ID
                        FROM {$wpdb->posts} p
                        WHERE p.post_type IN ($post_types_placeholders)
                        AND p.post_status = 'publish'
                        AND NOT EXISTS (
                            SELECT 1
                            FROM {$wpdb->term_relationships} tr
                            INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                            INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                            WHERE tr.object_id = p.ID
                            AND tt.taxonomy = 'product_cat'
                            AND t.slug = 'listeo-booking'
                        )
                        ORDER BY p.ID ASC
                    ", ...$enabled_post_types));
                } else {
                    $listings = $wpdb->get_results($wpdb->prepare("
                        SELECT ID
                        FROM {$wpdb->posts}
                        WHERE post_type IN ($post_types_placeholders)
                        AND post_status = 'publish'
                        ORDER BY ID ASC
                    ", ...$enabled_post_types));
                }
            }

            if (empty($listings)) {
                return array(
                    'success' => false,
                    'message' => $has_manual_selection
                        ? 'No published posts found in manual selection'
                        : 'No published content found for enabled post types'
                );
            }
            
            $total_listings = count($listings);
            $batch_count = ceil($total_listings / $batch_size);
            
            // Create batches
            for ($i = 0; $i < $batch_count; $i++) {
                $start_index = $i * $batch_size;
                $end_index = min(($i + 1) * $batch_size - 1, $total_listings - 1);
                $batch_id = 'batch_' . str_pad($i + 1, 3, '0', STR_PAD_LEFT);
                
                $batch_listings = array_slice($listings, $start_index, $batch_size);
                $batch_listing_count = count($batch_listings);
                
                $wpdb->insert(
                    $batch_table,
                    array(
                        'batch_id' => $batch_id,
                        'start_index' => $start_index,
                        'end_index' => $end_index,
                        'status' => 'ready',
                        'processed_count' => 0,
                        'total_count' => $batch_listing_count,
                        'created_at' => current_time('mysql')
                    ),
                    array('%s', '%d', '%d', '%s', '%d', '%d', '%s')
                );
            }
            
            // Update batch size setting
            update_option('listeo_ai_search_manual_batch_size', $batch_size);
            
            return array(
                'success' => true,
                'message' => sprintf('Created %d batches of %d listings each', $batch_count, $batch_size),
                'batch_count' => $batch_count,
                'batch_size' => $batch_size
            );
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => 'Error initializing batches: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Get all batches with their status
     * 
     * @return array Batches data
     */
    public static function get_all_batches() {
        global $wpdb;
        
        $batch_table = self::get_batch_table_name();
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$batch_table}'") === $batch_table;
        
        if (!$table_exists) {
            self::create_batch_table();
            return array();
        }
        
        $batches = $wpdb->get_results("
            SELECT * 
            FROM {$batch_table} 
            ORDER BY batch_id ASC
        ", ARRAY_A);
        
        return $batches ?: array();
    }
    
    /**
     * Get batch status summary
     * 
     * @return array Status summary
     */
    public static function get_batch_summary() {
        global $wpdb;
        
        $batch_table = self::get_batch_table_name();
        
        // Check if table exists
        $table_exists = $wpdb->get_var("SHOW TABLES LIKE '{$batch_table}'") === $batch_table;
        
        if (!$table_exists) {
            return array(
                'total_batches' => 0,
                'ready' => 0,
                'queued' => 0,
                'processing' => 0,
                'complete' => 0,
                'failed' => 0,
                'partial' => 0
            );
        }
        
        $summary = $wpdb->get_results("
            SELECT 
                status,
                COUNT(*) as count
            FROM {$batch_table}
            GROUP BY status
        ", ARRAY_A);
        
        $result = array(
            'ready' => 0,
            'queued' => 0,
            'processing' => 0,
            'complete' => 0,
            'failed' => 0,
            'partial' => 0
        );
        
        foreach ($summary as $row) {
            $result[$row['status']] = intval($row['count']);
        }
        
        $result['total_batches'] = array_sum($result);
        
        return $result;
    }
    
    /**
     * Process selected batches
     * 
     * @param array $batch_ids Array of batch IDs to process
     * @param int $delay Delay between requests in milliseconds
     * @return array Processing result
     */
    public static function process_selected_batches($batch_ids, $delay = null) {
        if (empty($batch_ids) || !is_array($batch_ids)) {
            return array(
                'success' => false,
                'message' => 'No batches selected for processing'
            );
        }
        
        if ($delay === null) {
            $delay = self::DEFAULT_DELAY;
        }
        
        $delay = max(0, min(10000, intval($delay))); // Clamp between 0-10 seconds
        
        $results = array();
        $total_processed = 0;
        $total_errors = 0;
        
        foreach ($batch_ids as $batch_id) {
            $result = self::process_single_batch($batch_id, $delay);
            $results[$batch_id] = $result;
            
            if ($result['success']) {
                $total_processed += $result['processed_count'];
            } else {
                $total_errors++;
            }
        }
        
        return array(
            'success' => $total_errors === 0,
            'message' => sprintf(
                'Processed %d batches. %d listings processed, %d batches failed.',
                count($batch_ids),
                $total_processed,
                $total_errors
            ),
            'batch_results' => $results,
            'total_processed' => $total_processed,
            'total_errors' => $total_errors
        );
    }
    
    /**
     * Process a single batch
     * 
     * @param string $batch_id Batch ID
     * @param int $delay Delay between requests
     * @return array Processing result
     */
    public static function process_single_batch($batch_id, $delay = null) {
        global $wpdb;
        
        if ($delay === null) {
            $delay = self::DEFAULT_DELAY;
        }
        
        $batch_table = self::get_batch_table_name();
        
        // Get batch details
        $batch = $wpdb->get_row($wpdb->prepare("
            SELECT * FROM {$batch_table} WHERE batch_id = %s
        ", $batch_id), ARRAY_A);
        
        if (!$batch) {
            return array(
                'success' => false,
                'message' => 'Batch not found: ' . $batch_id
            );
        }
        
        // Update batch status to processing
        $wpdb->update(
            $batch_table,
            array(
                'status' => 'processing',
                'last_attempt' => current_time('mysql')
            ),
            array('batch_id' => $batch_id),
            array('%s', '%s'),
            array('%s')
        );
        
        // Get whitelisted enabled post types
        $enabled_post_types = Listeo_AI_Search_Database_Manager::get_enabled_post_types();

        // If no post types are enabled, fail immediately
        if (empty($enabled_post_types)) {
            throw new Exception('No post types enabled for processing');
        }

        // Build post type condition
        $post_types_placeholders = implode(',', array_fill(0, count($enabled_post_types), '%s'));

        try {
            // Check for manual selections
            $manual_selections = get_option('listeo_ai_search_manual_selections', array());
            $has_manual_selection = !empty($manual_selections);

            if ($has_manual_selection) {
                // Build list of all manually selected post IDs across all post types
                $selected_ids = array();
                foreach ($manual_selections as $post_type => $ids) {
                    if (in_array($post_type, $enabled_post_types)) {
                        $selected_ids = array_merge($selected_ids, $ids);
                    }
                }

                if (empty($selected_ids)) {
                    throw new Exception('No posts in manual selection');
                }

                // Get only manually selected posts for this batch
                $placeholders = implode(',', array_fill(0, count($selected_ids), '%d'));
                $listings = $wpdb->get_results($wpdb->prepare("
                    SELECT ID
                    FROM {$wpdb->posts}
                    WHERE ID IN ($placeholders)
                    AND post_status = 'publish'
                    ORDER BY ID ASC
                    LIMIT %d OFFSET %d
                ", ...array_merge($selected_ids, array($batch['total_count'], $batch['start_index']))));
            } else {
                // No manual selection - get all posts for this batch (across all enabled post types)
                // Exclude listeo-booking category products (hidden booking products)
                $exclude_booking_products = in_array('product', $enabled_post_types);

                if ($exclude_booking_products) {
                    $listings = $wpdb->get_results($wpdb->prepare("
                        SELECT p.ID
                        FROM {$wpdb->posts} p
                        WHERE p.post_type IN ($post_types_placeholders)
                        AND p.post_status = 'publish'
                        AND NOT EXISTS (
                            SELECT 1
                            FROM {$wpdb->term_relationships} tr
                            INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                            INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                            WHERE tr.object_id = p.ID
                            AND tt.taxonomy = 'product_cat'
                            AND t.slug = 'listeo-booking'
                        )
                        ORDER BY p.ID ASC
                        LIMIT %d OFFSET %d
                    ", ...array_merge($enabled_post_types, array($batch['total_count'], $batch['start_index']))));
                } else {
                    $listings = $wpdb->get_results($wpdb->prepare("
                        SELECT ID
                        FROM {$wpdb->posts}
                        WHERE post_type IN ($post_types_placeholders)
                        AND post_status = 'publish'
                        ORDER BY ID ASC
                        LIMIT %d OFFSET %d
                    ", ...array_merge($enabled_post_types, array($batch['total_count'], $batch['start_index']))));
                }
            }

            if (empty($listings)) {
                throw new Exception('No listings found for batch');
            }
            
            $processed_count = 0;
            $error_count = 0;
            $errors = array();
            
            $embedding_manager = new Listeo_AI_Search_Embedding_Manager();
            
            foreach ($listings as $listing) {
                try {
                    // Check if embedding already exists and is current
                    $content = $embedding_manager->get_listing_content_for_embedding($listing->ID);
                    $content_hash = md5($content);
                    
                    $existing = Listeo_AI_Search_Database_Manager::get_embedding_by_listing_id($listing->ID);
                    
                    if ($existing && $existing['content_hash'] === $content_hash) {
                        // Skip if already processed
                        $processed_count++;
                        continue;
                    }
                    
                    // Generate embedding with rate limit handling
                    $embedding = self::generate_embedding_with_retry($content, $embedding_manager);
                    
                    if (!$embedding) {
                        throw new Exception('Failed to generate embedding after retries');
                    }
                    
                    // Store embedding
                    $success = Listeo_AI_Search_Database_Manager::store_embedding(
                        $listing->ID, 
                        $embedding, 
                        $content_hash
                    );
                    
                    if ($success) {
                        $processed_count++;
                    } else {
                        throw new Exception('Failed to store embedding in database');
                    }
                    
                    // Update batch progress
                    $wpdb->update(
                        $batch_table,
                        array('processed_count' => $processed_count),
                        array('batch_id' => $batch_id),
                        array('%d'),
                        array('%s')
                    );
                    
                    // Add delay to avoid rate limits (convert to seconds)
                    if ($delay > 0) {
                        usleep($delay * 1000); // Convert milliseconds to microseconds
                    }
                    
                } catch (Exception $e) {
                    $error_count++;
                    $errors[] = sprintf('Listing %d: %s', $listing->ID, $e->getMessage());
                    
                    Listeo_AI_Search_Utility_Helper::debug_log(
                        'Batch processing error for listing ' . $listing->ID . ': ' . $e->getMessage(),
                        'error'
                    );
                }
            }
            
            // Determine final status
            $final_status = 'complete';
            if ($error_count > 0) {
                $final_status = ($processed_count > 0) ? 'partial' : 'failed';
            }
            
            // Update batch final status
            $wpdb->update(
                $batch_table,
                array(
                    'status' => $final_status,
                    'processed_count' => $processed_count,
                    'error_message' => !empty($errors) ? implode('; ', $errors) : null,
                    'completed_at' => current_time('mysql')
                ),
                array('batch_id' => $batch_id),
                array('%s', '%d', '%s', '%s'),
                array('%s')
            );
            
            return array(
                'success' => $final_status !== 'failed',
                'message' => sprintf(
                    'Batch %s: %d/%d processed, %d errors',
                    $batch_id,
                    $processed_count,
                    count($listings),
                    $error_count
                ),
                'processed_count' => $processed_count,
                'error_count' => $error_count,
                'status' => $final_status,
                'errors' => $errors
            );
            
        } catch (Exception $e) {
            // Mark batch as failed
            $wpdb->update(
                $batch_table,
                array(
                    'status' => 'failed',
                    'error_message' => $e->getMessage(),
                    'retry_count' => $batch['retry_count'] + 1
                ),
                array('batch_id' => $batch_id),
                array('%s', '%s', '%d'),
                array('%s')
            );
            
            return array(
                'success' => false,
                'message' => 'Batch failed: ' . $e->getMessage(),
                'error' => $e->getMessage()
            );
        }
    }
    
    /**
     * Generate embedding with exponential backoff retry logic
     * 
     * @param string $content Content to embed
     * @param Listeo_AI_Search_Embedding_Manager $embedding_manager Embedding manager instance
     * @return array|false Embedding vector or false on failure
     */
    private static function generate_embedding_with_retry($content, $embedding_manager) {
        $max_retries = self::MAX_RETRY_ATTEMPTS;
        $base_delay = 2; // Base delay in seconds
        
        for ($attempt = 1; $attempt <= $max_retries; $attempt++) {
            try {
                $embedding = $embedding_manager->generate_embedding($content, true);
                
                if ($embedding) {
                    return $embedding;
                }
                
                throw new Exception('Embedding generation returned null');
                
            } catch (Exception $e) {
                $error_message = $e->getMessage();
                
                // Check if it's a rate limit error
                if (strpos($error_message, '429') !== false || 
                    strpos(strtolower($error_message), 'rate limit') !== false) {
                    
                    if ($attempt < $max_retries) {
                        // Exponential backoff: 2s, 4s, 8s, 16s, 32s
                        $delay = $base_delay * pow(2, $attempt - 1);
                        
                        Listeo_AI_Search_Utility_Helper::debug_log(
                            sprintf('Rate limit hit, retrying in %d seconds (attempt %d/%d)', $delay, $attempt, $max_retries),
                            'warning'
                        );
                        
                        sleep($delay);
                        continue;
                    }
                }
                
                // Non-rate-limit error or max retries reached
                Listeo_AI_Search_Utility_Helper::debug_log(
                    sprintf('Embedding generation failed after %d attempts: %s', $attempt, $error_message),
                    'error'
                );
                
                break;
            }
        }
        
        return false;
    }
    
    /**
     * Retry a failed batch
     * 
     * @param string $batch_id Batch ID
     * @param int $delay Delay between requests
     * @return array Retry result
     */
    public static function retry_failed_batch($batch_id, $delay = null) {
        global $wpdb;
        
        $batch_table = self::get_batch_table_name();
        
        // Get batch details
        $batch = $wpdb->get_row($wpdb->prepare("
            SELECT * FROM {$batch_table} WHERE batch_id = %s
        ", $batch_id), ARRAY_A);
        
        if (!$batch) {
            return array(
                'success' => false,
                'message' => 'Batch not found: ' . $batch_id
            );
        }
        
        if ($batch['status'] !== 'failed' && $batch['status'] !== 'partial') {
            return array(
                'success' => false,
                'message' => 'Only failed or partial batches can be retried'
            );
        }
        
        if ($batch['retry_count'] >= self::MAX_RETRY_ATTEMPTS) {
            return array(
                'success' => false,
                'message' => 'Maximum retry attempts reached for this batch'
            );
        }
        
        // Reset batch status
        $wpdb->update(
            $batch_table,
            array(
                'status' => 'ready',
                'error_message' => null
            ),
            array('batch_id' => $batch_id),
            array('%s', '%s'),
            array('%s')
        );
        
        // Process the batch
        return self::process_single_batch($batch_id, $delay);
    }
    
    /**
     * Reset all batches and start fresh
     * 
     * @return array Reset result
     */
    public static function reset_all_batches() {
        global $wpdb;
        
        try {
            // Clear all embeddings
            $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();
            $wpdb->query("TRUNCATE TABLE {$embeddings_table}");
            
            // Clear all batches
            $batch_table = self::get_batch_table_name();
            $wpdb->query("TRUNCATE TABLE {$batch_table}");
            
            return array(
                'success' => true,
                'message' => 'All embeddings and batches have been reset'
            );
            
        } catch (Exception $e) {
            return array(
                'success' => false,
                'message' => 'Error resetting batches: ' . $e->getMessage()
            );
        }
    }
    
    /**
     * Get processing progress for real-time updates
     * 
     * @return array Progress data
     */
    public static function get_processing_progress() {
        $total_listings = self::get_total_listings_count();
        $processed_listings = self::get_processed_listings_count();
        $batch_summary = self::get_batch_summary();
        
        $progress_percentage = $total_listings > 0 ? round(($processed_listings / $total_listings) * 100, 1) : 0;
        $remaining_listings = $total_listings - $processed_listings;
        
        return array(
            'total_listings' => $total_listings,
            'processed_listings' => $processed_listings,
            'remaining_listings' => $remaining_listings,
            'progress_percentage' => $progress_percentage,
            'batch_summary' => $batch_summary,
            'manual_mode_enabled' => self::should_use_manual_mode()
        );
    }
    
    /**
     * Queue batches for background processing
     * 
     * @param array $batch_ids Array of batch IDs to queue
     * @return array Result array
     */
    public static function queue_batches($batch_ids) {
        global $wpdb;
        $batch_table = self::get_batch_table_name();
        
        if (empty($batch_ids)) {
            return array(
                'success' => false,
                'message' => __('No batches selected', 'listeo-ai-search')
            );
        }
        
        // Update selected batches to queued status
        $placeholders = implode(',', array_fill(0, count($batch_ids), '%s'));
        $wpdb->query($wpdb->prepare(
            "UPDATE {$batch_table} SET status = 'queued' WHERE batch_id IN ({$placeholders}) AND status IN ('ready', 'failed')",
            ...$batch_ids
        ));
        
        // Schedule the background processing
        if (!wp_next_scheduled('listeo_ai_process_background_batches')) {
            wp_schedule_single_event(time() + 10, 'listeo_ai_process_background_batches');
        }
        
        return array(
            'success' => true,
            'message' => sprintf(__('Queued %d batches for background processing', 'listeo-ai-search'), count($batch_ids))
        );
    }
}
