<?php
/**
 * Listeo AI Floating Chat Widget
 *
 * Floating chat button and popup that appears on all pages
 *
 * @package Listeo_AI_Search
 * @since 1.0.0
 */

if (!defined('ABSPATH')) exit;

class Listeo_AI_Search_Floating_Chat_Widget {

    /**
     * Constructor
     */
    public function __construct() {
        add_action('wp_footer', array($this, 'render_floating_widget'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_widget_assets'));
    }

    /**
     * Enqueue widget assets
     */
    public function enqueue_widget_assets() {
        // Only load if widget is enabled
        if (!get_option('listeo_ai_floating_chat_enabled', 0)) {
            return;
        }

        // Check if login is required and user is not logged in
        if (get_option('listeo_ai_chat_require_login', 0) && !is_user_logged_in()) {
            return;
        }

        // Enqueue chat styles (reuse from shortcode)
        wp_enqueue_style(
            'listeo-ai-chat',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/chat.css',
            array(),
            LISTEO_AI_SEARCH_VERSION . '-' . time()
        );

        // Enqueue floating widget styles
        wp_enqueue_style(
            'listeo-ai-floating-chat',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/floating-chat.css',
            array('listeo-ai-chat'),
            LISTEO_AI_SEARCH_VERSION . '-' . time()
        );

        // Enqueue chat script (reuse from shortcode)
        wp_enqueue_script(
            'listeo-ai-chat',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/chat.js',
            array('jquery'),
            LISTEO_AI_SEARCH_VERSION . '-' . time(),
            true
        );

        // Enqueue floating widget script
        wp_enqueue_script(
            'listeo-ai-floating-chat',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/floating-chat.js',
            array('jquery', 'listeo-ai-chat'),
            LISTEO_AI_SEARCH_VERSION . '-' . time(),
            true
        );

        // Get placeholder image
        $placeholder_url = $this->get_placeholder_image();

        // Get widget settings
        $chat_name = get_option('listeo_ai_chat_name', __('AI Assistant', 'listeo-ai-search'));
        $welcome_message = get_option('listeo_ai_chat_welcome_message', __('Hello! How can I help you today?', 'listeo-ai-search'));
        $welcome_bubble_message = get_option('listeo_ai_floating_welcome_bubble', __('Hi! How can I help you?', 'listeo-ai-search'));

        // Localize script with config (for chat functionality)
        wp_localize_script('listeo-ai-chat', 'listeoAiChatConfig', array(
            'apiBase' => esc_url(rest_url('listeo/v1')),
            'nonce' => wp_create_nonce('wp_rest'),
            'debugMode' => (bool) get_option('listeo_ai_search_debug_mode', false),
            'placeholderImage' => $placeholder_url,
            'chatName' => $chat_name,
            'hideImages' => get_option('listeo_ai_chat_hide_images', 1),
            'rateLimits' => array(
                'tier1' => intval(get_option('listeo_ai_chat_rate_limit_tier1', 10)),
                'tier2' => intval(get_option('listeo_ai_chat_rate_limit_tier2', 30)),
                'tier3' => intval(get_option('listeo_ai_chat_rate_limit_tier3', 100))
            ),
            'strings' => listeo_ai_get_chat_strings($welcome_message) // Use shared function
        ));

        // Localize script for floating widget
        wp_localize_script('listeo-ai-floating-chat', 'listeoAiFloatingChatConfig', array(
            'welcomeBubbleMessage' => $welcome_bubble_message,
            'buttonIcon' => get_option('listeo_ai_floating_button_icon', 'fa-robot'),
            'strings' => array(
                'openChat' => __('Open chat', 'listeo-ai-search'),
                'closeChat' => __('Close chat', 'listeo-ai-search')
            )
        ));
    }

    /**
     * Get placeholder image URL
     */
    private function get_placeholder_image() {
        $placeholder_url = '';

        // Try listeo-core function
        if (function_exists('get_listeo_core_placeholder_image')) {
            $placeholder = get_listeo_core_placeholder_image();
            if (is_numeric($placeholder)) {
                $placeholder_img = wp_get_attachment_image_src($placeholder, 'medium');
                if ($placeholder_img && isset($placeholder_img[0])) {
                    $placeholder_url = $placeholder_img[0];
                }
            } else {
                $placeholder_url = $placeholder;
            }
        }

        // Fallback to theme customizer
        if (empty($placeholder_url)) {
            $placeholder_id = get_theme_mod('listeo_placeholder_id');
            if ($placeholder_id) {
                $placeholder_img = wp_get_attachment_image_src($placeholder_id, 'medium');
                if ($placeholder_img && isset($placeholder_img[0])) {
                    $placeholder_url = $placeholder_img[0];
                }
            }
        }

        return $placeholder_url;
    }

    /**
     * Render floating widget HTML
     */
    public function render_floating_widget() {
        // Only render if widget is enabled
        if (!get_option('listeo_ai_floating_chat_enabled', 0)) {
            return;
        }

        // Check if chat is enabled
        if (!get_option('listeo_ai_chat_enabled', 0)) {
            return;
        }

        // Check if login is required and user is not logged in
        if (get_option('listeo_ai_chat_require_login', 0) && !is_user_logged_in()) {
            return;
        }

        // Get settings
        $chat_title = get_option('listeo_ai_chat_name', __('AI Assistant', 'listeo-ai-search'));
        $placeholder = __('Type a message', 'listeo-ai-search');
        $custom_icon_id = intval(get_option('listeo_ai_floating_custom_icon', 0));
        $welcome_bubble = get_option('listeo_ai_floating_welcome_bubble', __('Hi! How can I help you?', 'listeo-ai-search'));
        $popup_width = intval(get_option('listeo_ai_floating_popup_width', 390));
        $popup_height = intval(get_option('listeo_ai_floating_popup_height', 600));
        $hide_images = intval(get_option('listeo_ai_chat_hide_images', 1));
        $button_color = sanitize_hex_color(get_option('listeo_ai_floating_button_color', '#222222'));
        if (empty($button_color)) {
            $button_color = '#222222'; // Fallback
        }

        // Validate dimensions
        $popup_width = max(320, min(800, $popup_width));
        $popup_height = max(400, min(900, $popup_height));

        // Get custom icon URL if set
        $custom_icon_url = $custom_icon_id ? wp_get_attachment_image_url($custom_icon_id, 'thumbnail') : '';
        $use_custom_icon = !empty($custom_icon_url);

        // Get primary color from settings
        $primary_color = sanitize_hex_color(get_option('listeo_ai_primary_color', '#0073ee'));
        if (empty($primary_color)) {
            $primary_color = '#0073ee'; // Fallback
        }

        // Convert hex to RGB for light variant
        $primary_rgb = sscanf($primary_color, "#%02x%02x%02x");
        $primary_color_light = sprintf('rgba(%d, %d, %d, 0.1)', $primary_rgb[0], $primary_rgb[1], $primary_rgb[2]);

        ?>
        <!-- Custom Button Color Styles -->
        <style>
            .listeo-floating-chat-button,
            .listeo-ai-chat-send-btn,
            .listeo-ai-load-listing-btn {
                background: <?php echo esc_attr($button_color); ?> !important;
            }

            /* AI Chat Primary Color Variables */
            :root {
                --ai-chat-primary-color: <?php echo esc_attr($primary_color); ?>;
                --ai-chat-primary-color-light: <?php echo esc_attr($primary_color_light); ?>;
            }
        </style>

        <!-- Floating Chat Widget -->
        <div class="listeo-floating-chat-widget" id="listeo-floating-chat-widget">

            <!-- Welcome Bubble (shows on first visit only) -->
            <div class="listeo-floating-welcome-bubble hidden" id="listeo-floating-welcome-bubble">
                <div class="listeo-floating-welcome-bubble-content">
                    <?php echo wp_kses_post($welcome_bubble); ?>
                </div>
                <div class="listeo-floating-welcome-bubble-arrow"></div>
            </div>
            <!-- Check localStorage immediately to prevent flash -->
            <script>
                (function() {
                    var bubble = document.getElementById('listeo-floating-welcome-bubble');
                    var dismissed = localStorage.getItem('listeo_floating_chat_bubble_dismissed');
                    if (dismissed !== 'true' && bubble) {
                        bubble.classList.remove('hidden');
                    }
                })();
            </script>

            <!-- Floating Button -->
            <button
                class="listeo-floating-chat-button <?php echo $use_custom_icon ? 'has-custom-icon' : ''; ?>"
                id="listeo-floating-chat-button"
                aria-label="<?php esc_attr_e('Open chat', 'listeo-ai-search'); ?>"
            >
                <?php if ($use_custom_icon) : ?>
                    <img src="<?php echo esc_url($custom_icon_url); ?>" alt="Chat" class="listeo-floating-custom-icon listeo-floating-icon-open" />
                <?php else : ?>
                    <img src="<?php echo esc_url(LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/icons/chat.svg'); ?>" alt="Chat" class="listeo-floating-icon-open" width="28" height="28" />
                <?php endif; ?>
                <img src="<?php echo esc_url(LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/icons/close.svg'); ?>" alt="Close" class="listeo-floating-icon-close" style="display: none;" width="18" height="18" />
            </button>

            <!-- Chat Popup (reuses exact shortcode HTML structure) -->
            <div class="listeo-floating-chat-popup" id="listeo-floating-chat-popup" style="display: none; width: <?php echo esc_attr($popup_width); ?>px; height: <?php echo esc_attr($popup_height); ?>px;">
                <div class="listeo-ai-chat-wrapper" id="listeo-floating-chat-instance" data-hide-images="<?php echo esc_attr($hide_images); ?>">
                    <div class="listeo-ai-chat-container">
                        <div class="listeo-ai-chat-header">
                            <div class="listeo-ai-chat-title"><?php echo esc_html($chat_title); ?></div>
                            <button
                                class="listeo-ai-chat-clear-btn"
                                title="<?php esc_attr_e('Clear conversation', 'listeo-ai-search'); ?>"
                            >
                                <img src="<?php echo esc_url(LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/icons/refresh.svg'); ?>" alt="Refresh" width="16" height="16" />
                            </button>
                        </div>
                        <div class="listeo-ai-chat-messages" id="listeo-floating-chat-instance-messages">
                            <!-- Welcome message added by JavaScript -->
                        </div>

                        <div class="listeo-ai-chat-input-wrapper">
                            <textarea
                                id="listeo-floating-chat-instance-input"
                                class="listeo-ai-chat-input"
                                placeholder="<?php echo esc_attr($placeholder); ?>"
                                rows="2"
                            ></textarea>
                            <button
                                id="listeo-floating-chat-instance-send"
                                class="listeo-ai-chat-send-btn"
                            >
                                <img src="<?php echo esc_url(LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/icons/arrow-up.svg'); ?>" alt="Send" width="16" height="16" />
                            </button>
                        </div>

                        <?php if (get_option('listeo_ai_chat_terms_notice_enabled', 0)): ?>
                            <div class="listeo-ai-chat-terms-notice">
                                <?php echo wp_kses_post(get_option('listeo_ai_chat_terms_notice_text', 'By using this chat, you agree to our <a href="/terms-of-use" target="_blank">Terms of Use</a> and <a href="/privacy-policy" target="_blank">Privacy Policy</a>')); ?>
                            </div>
                        <?php endif; ?>
                    </div>
                </div>
            </div>

        </div>
        <?php
    }
}

// Initialize floating widget
new Listeo_AI_Search_Floating_Chat_Widget();
