<?php
/**
 * Listeo AI Chat Shortcode
 *
 * Provides shortcode for embedding AI chat interface
 *
 * @package Listeo_AI_Search
 * @since 1.0.0
 */

if (!defined('ABSPATH')) exit;

class Listeo_AI_Search_Chat_Shortcode {

    /**
     * Constructor
     */
    public function __construct() {
        // Register shortcodes (both old and new for backward compatibility)
        add_shortcode('listeo_ai_chat', array($this, 'render_chat')); // Legacy shortcode
        add_shortcode('ai_chat', array($this, 'render_chat')); // New shortcode
        add_action('wp_enqueue_scripts', array($this, 'enqueue_chat_assets'));
    }

    /**
     * Enqueue chat assets
     */
    public function enqueue_chat_assets() {
        // Enqueue chat styles
        wp_enqueue_style(
            'listeo-ai-chat',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/chat.css',
            array(),
            LISTEO_AI_SEARCH_VERSION . '-' . time() // Cache busting for testing
        );

        // Add inline CSS for primary color variables
        $primary_color = sanitize_hex_color(get_option('listeo_ai_primary_color', '#0073ee'));
        if (empty($primary_color)) {
            $primary_color = '#0073ee'; // Fallback
        }

        // Convert hex to RGB for light variant
        $primary_rgb = sscanf($primary_color, "#%02x%02x%02x");
        $primary_color_light = sprintf('rgba(%d, %d, %d, 0.1)', $primary_rgb[0], $primary_rgb[1], $primary_rgb[2]);

        $custom_css = sprintf(
            ':root { --ai-chat-primary-color: %s; --ai-chat-primary-color-light: %s; }',
            esc_attr($primary_color),
            esc_attr($primary_color_light)
        );
        wp_add_inline_style('listeo-ai-chat', $custom_css);

        // Enqueue chat script
        wp_enqueue_script(
            'listeo-ai-chat',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/chat.js',
            array('jquery'),
            LISTEO_AI_SEARCH_VERSION . '-' . time(), // Cache busting for testing
            true
        );

        // Get placeholder image from theme/core
        $placeholder_url = '';
        if (function_exists('get_listeo_core_placeholder_image')) {
            $placeholder = get_listeo_core_placeholder_image();
            if (is_numeric($placeholder)) {
                $placeholder_img = wp_get_attachment_image_src($placeholder, 'medium');
                if ($placeholder_img && isset($placeholder_img[0])) {
                    $placeholder_url = $placeholder_img[0];
                }
            } else {
                $placeholder_url = $placeholder;
            }
        }

        // Fallback to theme customizer
        if (empty($placeholder_url)) {
            $placeholder_id = get_theme_mod('listeo_placeholder_id');
            if ($placeholder_id) {
                $placeholder_img = wp_get_attachment_image_src($placeholder_id, 'medium');
                if ($placeholder_img && isset($placeholder_img[0])) {
                    $placeholder_url = $placeholder_img[0];
                }
            }
        }

        // Get chatbot name and welcome message
        $chat_name = get_option('listeo_ai_chat_name', __('AI Assistant', 'listeo-ai-search'));
        $welcome_message = get_option('listeo_ai_chat_welcome_message', __('Hello! How can I help you today?', 'listeo-ai-search'));

        // Localize script with config
        wp_localize_script('listeo-ai-chat', 'listeoAiChatConfig', array(
            'apiBase' => esc_url(rest_url('listeo/v1')),
            'nonce' => wp_create_nonce('wp_rest'),
            'debugMode' => (bool) get_option('listeo_ai_search_debug_mode', false), // Debug mode from settings
            'placeholderImage' => $placeholder_url, // Theme placeholder image
            'chatName' => $chat_name, // Configurable chat name
            'hideImages' => get_option('listeo_ai_chat_hide_images', 1), // Hide images setting
            'rateLimits' => array(
                'tier1' => intval(get_option('listeo_ai_chat_rate_limit_tier1', 10)),  // Messages per minute
                'tier2' => intval(get_option('listeo_ai_chat_rate_limit_tier2', 30)),  // Messages per 15 minutes
                'tier3' => intval(get_option('listeo_ai_chat_rate_limit_tier3', 100))  // Messages per day
            ),
            'strings' => listeo_ai_get_chat_strings($welcome_message) // Use shared function
        ));
    }

    /**
     * Render chat shortcode
     *
     * @param array $atts Shortcode attributes
     * @return string
     */
    public function render_chat($atts) {
        // Parse attributes (use 'ai_chat' as the tag name for shortcode_atts)
        $atts = shortcode_atts(array(
            'height' => '600px',
            'pictures' => '', // 'enabled', 'disabled', or empty (use global setting)
            'show_popular_searches' => 'no', // 'yes' or 'no'
            'popular_searches_limit' => 5, // Number of popular searches to display
            'popular_searches_title' => '' // Custom title text (empty = use default translation)
        ), $atts, 'ai_chat');

        // Get title, placeholder, and welcome message from settings (not shortcode attributes)
        $chat_title = get_option('listeo_ai_chat_name', __('AI Assistant', 'listeo-ai-search'));
        $placeholder = __('Type a message', 'listeo-ai-search');
        $welcome_message = get_option('listeo_ai_chat_welcome_message', __('Hello! How can I help you today?', 'listeo-ai-search'));

        // Determine hideImages setting for this instance
        $global_hide_images = get_option('listeo_ai_chat_hide_images', 1);
        if ($atts['pictures'] === 'enabled') {
            $hide_images = 0; // Show images
        } elseif ($atts['pictures'] === 'disabled') {
            $hide_images = 1; // Hide images
        } else {
            $hide_images = $global_hide_images; // Use global setting
        }

        // Check if chat is enabled
        if (!get_option('listeo_ai_chat_enabled', 0)) {
            return '<div class="listeo-ai-chat-disabled">' .
                   '<p>' . __('AI Chat is currently disabled.', 'listeo-ai-search') . '</p>' .
                   '</div>';
        }

        // Check if login is required
        if (get_option('listeo_ai_chat_require_login', 0) && !is_user_logged_in()) {
            return '<div class="listeo-ai-chat-disabled">' .
                   '<p>' . __('Please log in to use AI Chat.', 'listeo-ai-search') . '</p>' .
                   '</div>';
        }

        // Generate consistent ID for this chat instance (based on page ID for localStorage persistence)
        // Use post/page ID if available, otherwise use a hash of the URL
        global $post;
        if ($post && $post->ID) {
            $chat_id = 'listeo-ai-chat-' . $post->ID;
        } else {
            // Fallback: use hash of current URL for consistent ID across page loads
            $chat_id = 'listeo-ai-chat-' . md5($_SERVER['REQUEST_URI']);
        }

        ob_start();
        ?>
        <div class="listeo-ai-chat-wrapper" id="<?php echo esc_attr($chat_id); ?>" style="height: <?php echo esc_attr($atts['height']); ?>" data-hide-images="<?php echo esc_attr($hide_images); ?>">
            <div class="listeo-ai-chat-container">
                <div class="listeo-ai-chat-header">
                    <div class="listeo-ai-chat-title"><?php echo esc_html($chat_title); ?></div>
                    <button
                        class="listeo-ai-chat-clear-btn"
                        title="<?php esc_attr_e('Clear conversation', 'listeo-ai-search'); ?>"
                    >
                        <img src="<?php echo esc_url(LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/icons/refresh.svg'); ?>" alt="Refresh" width="16" height="16" />
                    </button>
                </div>
                <div class="listeo-ai-chat-messages" id="<?php echo esc_attr($chat_id); ?>-messages">
                    <!-- Welcome message added by JavaScript to avoid flash of wrong content -->
                </div>

                <div class="listeo-ai-chat-input-wrapper">
                    <textarea
                        id="<?php echo esc_attr($chat_id); ?>-input"
                        class="listeo-ai-chat-input"
                        placeholder="<?php echo esc_attr($placeholder); ?>"
                        rows="2"
                    ></textarea>
                    <button
                        id="<?php echo esc_attr($chat_id); ?>-send"
                        class="listeo-ai-chat-send-btn"
                    >
                        <img src="<?php echo esc_url(LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/icons/arrow-up.svg'); ?>" alt="Send" width="16" height="16" />
                    </button>
                </div>

                <?php if (get_option('listeo_ai_chat_terms_notice_enabled', 0)): ?>
                    <div class="listeo-ai-chat-terms-notice">
                        <?php echo wp_kses_post(get_option('listeo_ai_chat_terms_notice_text', 'By using this chat, you agree to our <a href="/terms-of-use" target="_blank">Terms of Use</a> and <a href="/privacy-policy" target="_blank">Privacy Policy</a>')); ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php

        // Add popular searches section if enabled
        if ($atts['show_popular_searches'] === 'yes') {
            echo $this->render_popular_searches($atts['popular_searches_limit'], $chat_id, $atts['popular_searches_title']);
        }

        return ob_get_clean();
    }

    /**
     * Render popular searches tags
     *
     * @param int $limit Number of popular searches to display
     * @param string $chat_id Chat container ID for JavaScript targeting
     * @param string $title Custom title text (empty = use default translation)
     * @return string
     */
    private function render_popular_searches($limit, $chat_id, $title = '') {
        // Check if search suggestions are enabled globally
        if (!get_option('listeo_ai_search_suggestions_enabled', true)) {
            return '';
        }

        // Get the suggestions source setting from plugin
        $suggestions_source = get_option('listeo_ai_search_suggestions_source', 'top_searches');
        $suggestions = array();

        if ($suggestions_source === 'custom') {
            // Use custom suggestions from plugin settings
            $custom_suggestions = get_option('listeo_ai_search_custom_suggestions', '');
            if (!empty($custom_suggestions)) {
                $suggestions_array = array_map('trim', explode(',', $custom_suggestions));
                $suggestions_array = array_filter($suggestions_array); // Remove empty items

                // Limit to the requested number
                $suggestions = array_slice($suggestions_array, 0, intval($limit));
            }
        } elseif ($suggestions_source === 'top_searches_10') {
            // Get top 10 searches from analytics
            if (class_exists('Listeo_AI_Search_Analytics')) {
                $analytics = Listeo_AI_Search_Analytics::get_analytics(30);
                if (!empty($analytics['popular_queries']) && is_array($analytics['popular_queries'])) {
                    // Get top 10, but limit to the requested limit
                    $top_queries = array_slice($analytics['popular_queries'], 0, min(10, intval($limit)), true);
                    $suggestions = array_keys($top_queries);
                }
            }
        } else {
            // Default: top 5 searches from analytics
            if (class_exists('Listeo_AI_Search_Analytics')) {
                $analytics = Listeo_AI_Search_Analytics::get_analytics(30);
                if (!empty($analytics['popular_queries']) && is_array($analytics['popular_queries'])) {
                    // Get top 5, but limit to the requested limit
                    $top_queries = array_slice($analytics['popular_queries'], 0, min(5, intval($limit)), true);
                    $suggestions = array_keys($top_queries);
                }
            }
        }

        // If no suggestions found, return empty
        if (empty($suggestions)) {
            return '';
        }

        // Use custom title if provided, otherwise use default translation
        $header_text = !empty($title) ? $title : __('Popular Searches:', 'listeo-ai-search');

        ob_start();
        ?>
        <div class="listeo-ai-popular-searches" data-chat-id="<?php echo esc_attr($chat_id); ?>">
            <div class="popular-searches-header">
                <?php echo esc_html($header_text); ?>
            </div>
            <div class="popular-searches-tags">
                <?php foreach ($suggestions as $suggestion) : ?>
                    <button class="popular-search-tag" data-query="<?php echo esc_attr($suggestion); ?>">
                        <?php echo esc_html(ucfirst($suggestion)); ?>
                    </button>
                <?php endforeach; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }
}

// Initialize shortcode
new Listeo_AI_Search_Chat_Shortcode();
