<?php
/**
 * Listeo AI Search Chat History Manager
 *
 * Handles storage and retrieval of chat conversations
 * Tracks user questions and AI responses for analytics and training
 *
 * @package Listeo_AI_Search
 * @since 1.3
 */

if (!defined('ABSPATH')) exit;

// Only declare if not already loaded by pro version
if (!class_exists('Listeo_AI_Search_Chat_History')) :

class Listeo_AI_Search_Chat_History {

    /**
     * Database table name
     */
    private static $table_name = 'listeo_ai_chat_history';

    /**
     * Get full table name with WordPress prefix
     *
     * @return string
     */
    public static function get_table_name() {
        global $wpdb;
        return $wpdb->prefix . self::$table_name;
    }

    /**
     * Create chat history database table
     * Called on first enable or plugin activation
     */
    public static function create_table() {
        global $wpdb;

        $table_name = self::get_table_name();
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE IF NOT EXISTS {$table_name} (
            id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
            conversation_id varchar(50) NOT NULL,
            session_id varchar(100) NOT NULL,
            user_id bigint(20) unsigned DEFAULT NULL,
            user_message text NOT NULL,
            assistant_message text NOT NULL,
            model_used varchar(50) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY conversation_id (conversation_id),
            KEY session_id (session_id),
            KEY user_id (user_id),
            KEY created_at (created_at)
        ) {$charset_collate};";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);

        // Verify table was created
        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") != $table_name) {
            error_log('Listeo AI Search: Failed to create chat history table');
            return false;
        }

        return true;
    }

    /**
     * Save a chat exchange (user question + AI answer)
     *
     * @param string $session_id Browser session identifier
     * @param string $user_message User's question
     * @param string $assistant_message AI's text answer
     * @param string $model_used OpenAI model name
     * @param int|null $user_id WordPress user ID (NULL for guests)
     * @return int|false Insert ID on success, false on failure
     */
    public static function save_exchange($session_id, $user_message, $assistant_message, $model_used, $user_id = null) {
        global $wpdb;

        // PRO FEATURE: Chat history logging requires Pro version
        if (!apply_filters('ai_chat_search_can_access_conversation_logs', false)) {
            return false;
        }

        // Check if chat history tracking is enabled
        if (!get_option('listeo_ai_chat_history_enabled', 0)) {
            return false;
        }

        // Validate required parameters
        if (empty($session_id) || empty($user_message) || empty($assistant_message) || empty($model_used)) {
            error_log('Listeo AI Chat History: Missing required parameters');
            return false;
        }

        $table_name = self::get_table_name();

        // Use session_id as conversation_id for grouping
        $conversation_id = $session_id;

        // Get current user ID if not provided
        if ($user_id === null && is_user_logged_in()) {
            $user_id = get_current_user_id();
        }

        $result = $wpdb->insert(
            $table_name,
            array(
                'conversation_id' => $conversation_id,
                'session_id' => $session_id,
                'user_id' => $user_id,
                'user_message' => wp_kses_post($user_message), // Sanitize but preserve HTML
                'assistant_message' => wp_kses_post($assistant_message),
                'model_used' => sanitize_text_field($model_used),
                'created_at' => current_time('mysql')
            ),
            array('%s', '%s', '%d', '%s', '%s', '%s', '%s')
        );

        if ($result === false) {
            error_log('Listeo AI Chat History: Database insert failed - ' . $wpdb->last_error);
            return false;
        }

        return $wpdb->insert_id;
    }

    /**
     * Get chat history statistics
     *
     * @param int $days Number of days to include (default: 30)
     * @return array Statistics data
     */
    public static function get_stats($days = 30) {
        global $wpdb;

        $table_name = self::get_table_name();
        $date_from = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        // Total conversations (unique conversation_ids)
        $total_conversations = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT conversation_id) FROM {$table_name} WHERE created_at >= %s",
            $date_from
        ));

        // Total messages exchanged
        $total_messages = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} WHERE created_at >= %s",
            $date_from
        ));

        // Average messages per conversation
        $avg_per_conversation = $total_conversations > 0 ? round($total_messages / $total_conversations, 1) : 0;

        // Registered users vs guests
        $registered_users = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT conversation_id) FROM {$table_name} WHERE user_id IS NOT NULL AND created_at >= %s",
            $date_from
        ));

        $guest_users = $total_conversations - $registered_users;

        return array(
            'total_conversations' => intval($total_conversations),
            'total_messages' => intval($total_messages),
            'avg_per_conversation' => floatval($avg_per_conversation),
            'registered_users' => intval($registered_users),
            'guest_users' => intval($guest_users)
        );
    }

    /**
     * Get recent conversations
     *
     * @param int $limit Number of conversations to retrieve
     * @param int $offset Offset for pagination
     * @return array Array of conversation data
     */
    public static function get_recent_conversations($limit = 20, $offset = 0) {
        global $wpdb;

        $table_name = self::get_table_name();

        $conversations = $wpdb->get_results($wpdb->prepare(
            "SELECT
                conversation_id,
                MIN(created_at) as first_message_at,
                MAX(created_at) as last_message_at,
                COUNT(*) as message_count,
                user_id
            FROM {$table_name}
            GROUP BY conversation_id
            ORDER BY last_message_at DESC
            LIMIT %d OFFSET %d",
            $limit,
            $offset
        ), ARRAY_A);

        return $conversations;
    }

    /**
     * Get all messages in a specific conversation
     *
     * @param string $conversation_id Conversation identifier
     * @return array Array of messages
     */
    public static function get_conversation($conversation_id) {
        global $wpdb;

        $table_name = self::get_table_name();

        $messages = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE conversation_id = %s ORDER BY created_at ASC",
            $conversation_id
        ), ARRAY_A);

        return $messages;
    }

    /**
     * Delete old chat history records
     *
     * @param int $days Delete records older than this many days
     * @return int|false Number of rows deleted, or false on error
     */
    public static function cleanup_old_records($days = 90) {
        global $wpdb;

        $table_name = self::get_table_name();
        $date_threshold = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        $deleted = $wpdb->query($wpdb->prepare(
            "DELETE FROM {$table_name} WHERE created_at < %s",
            $date_threshold
        ));

        return $deleted;
    }

    /**
     * Get popular user questions (for insights)
     *
     * @param int $limit Number of questions to return
     * @param int $days Days to look back
     * @return array Array of popular questions with counts
     */
    public static function get_popular_questions($limit = 10, $days = 30) {
        global $wpdb;

        $table_name = self::get_table_name();
        $date_from = date('Y-m-d H:i:s', strtotime("-{$days} days"));

        $questions = $wpdb->get_results($wpdb->prepare(
            "SELECT user_message, COUNT(*) as count
            FROM {$table_name}
            WHERE created_at >= %s
            GROUP BY user_message
            ORDER BY count DESC
            LIMIT %d",
            $date_from,
            $limit
        ), ARRAY_A);

        return $questions;
    }

    /**
     * Drop the chat history table
     * Used for cleanup or uninstall
     */
    public static function drop_table() {
        global $wpdb;
        $table_name = self::get_table_name();
        $wpdb->query("DROP TABLE IF EXISTS {$table_name}");
    }
}

endif; // class_exists check
