<?php
/**
 * Background Processing for Listeo AI Search
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Background processing class
 */
class Listeo_AI_Background_Processor {
    
    /**
     * Hook into WordPress
     */
    public static function init() {
        add_action('listeo_ai_process_listing', array(__CLASS__, 'process_single_listing'));
        add_action('listeo_ai_bulk_process_listings', array(__CLASS__, 'process_all_listings'));
    }
    
    /**
     * Process a single post (any post type)
     */
    public static function process_single_listing($listing_id) {
        global $wpdb;

        $post = get_post($listing_id);

        if (!$post || $post->post_status !== 'publish') {
            return false;
        }

        try {
            // Get API key
            $api_key = get_option('listeo_ai_search_api_key', '');
            if (empty($api_key)) {
                Listeo_AI_Search_Utility_Helper::debug_log('No API key configured', 'error');
                return false;
            }

            // Collect post content using factory pattern
            $content = self::collect_content($listing_id);
            $content_hash = md5($content);
            
            // Check if we already have current embedding
            $table_name = $wpdb->prefix . 'listeo_ai_embeddings';
            $existing = $wpdb->get_row($wpdb->prepare(
                "SELECT content_hash FROM $table_name WHERE listing_id = %d",
                $listing_id
            ));
            
            if ($existing && $existing->content_hash === $content_hash) {
                // Content hasn't changed, skip
                return true;
            }
            
            // Generate embedding
            $embedding = self::generate_embedding($content, $api_key);
            
            if (!$embedding) {
                Listeo_AI_Search_Utility_Helper::debug_log("Failed to generate embedding for listing $listing_id", 'error');
                return false;
            }
            
            // Store embedding
            $result = $wpdb->replace($table_name, array(
                'listing_id' => $listing_id,
                'embedding' => Listeo_AI_Search_Database_Manager::compress_embedding_for_storage($embedding),
                'content_hash' => $content_hash,
                'updated_at' => current_time('mysql')
            ));
            
            if ($result === false) {
                Listeo_AI_Search_Utility_Helper::debug_log("Failed to store embedding for listing $listing_id", 'error');
                return false;
            }
            
            Listeo_AI_Search_Utility_Helper::debug_log("Successfully processed listing $listing_id", 'info');
            return true;
            
        } catch (Exception $e) {
            if (get_option('listeo_ai_search_debug_mode', false)) {
                error_log("Listeo AI Search: Error processing listing $listing_id - " . $e->getMessage());
            }
            return false;
        }
    }
    
    /**
     * Process all existing posts (supports multiple post types)
     */
    public static function process_all_listings() {
        // Get configured post types or use defaults
        $post_types = get_option('listeo_ai_search_post_types', array('listing', 'post', 'page', 'product'));

        $args = array(
            'post_type' => $post_types,
            'post_status' => 'publish',
            'posts_per_page' => -1,
            'fields' => 'ids'
        );

        $listings = get_posts($args);

        if (get_option('listeo_ai_search_debug_mode', false)) {
            error_log("Listeo AI Search: Starting bulk processing of " . count($listings) . " listings");
        }

        $processed = 0;
        $failed = 0;

        foreach ($listings as $listing_id) {
            // Add delay to avoid rate limiting
            if ($processed > 0 && $processed % 3 === 0) {
                sleep(1); // 1 second delay every 3 requests
            }

            if (self::process_single_listing($listing_id)) {
                $processed++;
            } else {
                $failed++;
            }

            // Break if too many failures
            if ($failed > 10) {
                if (get_option('listeo_ai_search_debug_mode', false)) {
                    error_log("Listeo AI Search: Too many failures ($failed), stopping bulk process");
                }
                break;
            }
        }

        if (get_option('listeo_ai_search_debug_mode', false)) {
            error_log("Listeo AI Search: Bulk processing complete. Processed: $processed, Failed: $failed");
        }
        
        // Schedule cleanup of old embeddings
        wp_schedule_single_event(time() + 300, 'listeo_ai_cleanup_embeddings');
    }
    
    /**
     * Collect content from any post type using factory pattern
     */
    public static function collect_content($post_id) {
        return Listeo_AI_Content_Extractor_Factory::extract_content($post_id);
    }

    /**
     * Legacy method - backward compatibility
     * @deprecated 2.0.0 Use collect_content() instead
     */
    public static function collect_listing_content($listing_id) {
        return self::collect_content($listing_id);
    }
    
    /**
     * Generate embedding using OpenAI
     */
    private static function generate_embedding($text, $api_key) {
        $start_time = microtime(true);
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("Making OpenAI API call for " . strlen($text) . " characters", 'info');
        }
        
        $response = wp_remote_post('https://api.openai.com/v1/embeddings', array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $api_key,
                'Content-Type' => 'application/json',
            ),
            'body' => json_encode(array(
                'model' => 'text-embedding-3-small',
                'input' => $text,
            )),
            'timeout' => 60,
        ));
        
        $duration = microtime(true) - $start_time;
        
        if (is_wp_error($response)) {
            $error_msg = 'OpenAI API request failed: ' . $response->get_error_message();
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API Error after {$duration}s: {$error_msg}", 'error');
            }
            throw new Exception($error_msg);
        }
        
        $http_code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            Listeo_AI_Search_Utility_Helper::debug_log("API Response after {$duration}s: HTTP {$http_code}", 'info');
        }
        
        if ($http_code !== 200) {
            $error_msg = "OpenAI API HTTP error: {$http_code}";
            if (isset($body['error']['message'])) {
                $error_msg .= " - " . $body['error']['message'];
            }
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API HTTP Error: {$error_msg}", 'error');
            }
            throw new Exception($error_msg);
        }
        
        if (isset($body['error'])) {
            $error_msg = 'OpenAI API error: ' . $body['error']['message'];
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API Response Error: {$error_msg}", 'error');
            }
            throw new Exception($error_msg);
        }
        
        if (!isset($body['data'][0]['embedding'])) {
            $error_msg = 'OpenAI API response missing embedding data';
            if (get_option('listeo_ai_search_debug_mode', false)) {
                Listeo_AI_Search_Utility_Helper::debug_log("API Response Missing Data: " . json_encode($body), 'error');
            }
            throw new Exception($error_msg);
        }
        
        if (get_option('listeo_ai_search_debug_mode', false)) {
            $embedding_count = count($body['data'][0]['embedding']);
            Listeo_AI_Search_Utility_Helper::debug_log("Successfully received embedding with {$embedding_count} dimensions in {$duration}s", 'info');
        }
        
        return $body['data'][0]['embedding'];
    }
    
    /**
     * Cleanup old embeddings for deleted posts
     */
    public static function cleanup_embeddings() {
        global $wpdb;

        $table_name = $wpdb->prefix . 'listeo_ai_embeddings';

        // Delete embeddings for non-existent or non-published posts
        $wpdb->query("
            DELETE e FROM $table_name e
            LEFT JOIN {$wpdb->posts} p ON e.listing_id = p.ID
            WHERE p.ID IS NULL OR p.post_status != 'publish'
        ");

        if (get_option('listeo_ai_search_debug_mode', false)) {
            error_log("Listeo AI Search: Embedding cleanup completed");
        }
    }
}

// Initialize background processor
add_action('listeo_ai_cleanup_embeddings', array('Listeo_AI_Background_Processor', 'cleanup_embeddings'));
Listeo_AI_Background_Processor::init();
