/**
 * AI Chat Script
 *
 * Handles chat functionality for shortcode
 */

(function($) {
    'use strict';

    /**
     * Debug logging helper - only logs when debug mode is enabled
     */
    const debugLog = function(...args) {
        if (typeof listeoAiChatConfig !== 'undefined' && listeoAiChatConfig.debugMode) {
            console.log('[AI Chat]', ...args);
        }
    };

    const debugError = function(...args) {
        if (typeof listeoAiChatConfig !== 'undefined' && listeoAiChatConfig.debugMode) {
            console.error('[AI Chat ERROR]', ...args);
        }
    };

    // Initialize all chat instances
    $(document).ready(function() {
        debugLog('Initializing...');
        debugLog('jQuery loaded:', typeof $ !== 'undefined');
        debugLog('Found chat wrappers:', $('.listeo-ai-chat-wrapper').length);

        // Check if config is loaded
        if (typeof listeoAiChatConfig === 'undefined') {
            console.error('ERROR: listeoAiChatConfig is not defined! Script may not be enqueued properly.');
            return;
        }

        debugLog('Config:', listeoAiChatConfig);

        $('.listeo-ai-chat-wrapper').each(function() {
            debugLog('Initializing chat instance:', $(this).attr('id'));
            try {
                new ListeoAIChat($(this));
            } catch (error) {
                console.error('Failed to initialize chat:', error);
            }
        });
    });

    /**
     * Chat class
     */
    function ListeoAIChat($wrapper) {
        this.$wrapper = $wrapper;
        this.chatId = $wrapper.attr('id');
        this.$messages = $wrapper.find('.listeo-ai-chat-messages');
        this.$input = $wrapper.find('.listeo-ai-chat-input');
        this.$sendBtn = $wrapper.find('.listeo-ai-chat-send-btn');
        this.$clearBtn = $wrapper.find('.listeo-ai-chat-clear-btn');

        this.conversationHistory = [];
        this.chatConfig = null;
        this.isProcessing = false;
        this.configLoaded = false;
        this.storageKey = 'listeo_ai_chat_' + this.chatId;
        this.rateLimitStorageKey = 'listeo_chat_rate_limit';

        // Read per-instance hideImages setting from data attribute (overrides global config)
        var dataHideImages = $wrapper.data('hide-images');
        this.hideImages = (dataHideImages !== undefined) ? parseInt(dataHideImages) : listeoAiChatConfig.hideImages;

        debugLog('[Chat Init] Hide Images Config:', {
            dataAttribute: dataHideImages,
            globalConfig: listeoAiChatConfig.hideImages,
            finalValue: this.hideImages
        });

        // Rate limiting configuration (from backend or defaults)
        this.rateLimits = {
            tier1: { limit: listeoAiChatConfig.rateLimits?.tier1 || 10, window: 60 },      // 10/min
            tier2: { limit: listeoAiChatConfig.rateLimits?.tier2 || 30, window: 900 },    // 30/15min
            tier3: { limit: listeoAiChatConfig.rateLimits?.tier3 || 100, window: 86400 }  // 100/day
        };

        // Loaded listing context (for single listing pages)
        this.loadedListing = null;
        this.loadedListingStorageKey = 'listeo_chat_loaded_listing_' + this.chatId;

        // Track if chat has been expanded (for style 2)
        this.isExpanded = false;

        this.init();
    }

    ListeoAIChat.prototype = {
        /**
         * Initialize chat
         */
        init: function() {
            var self = this;

            debugLog('Chat init:', {
                chatId: this.chatId,
                hasMessages: this.$messages.length,
                hasInput: this.$input.length,
                hasSendBtn: this.$sendBtn.length
            });

            // Disable send button until config loads
            this.$sendBtn.prop('disabled', true);

            // Load config
            this.loadConfig();

            // Load conversation from localStorage
            this.loadConversation();

            // Load previously loaded listing from localStorage (if any)
            this.loadPersistedListingContext();

            // Detect if we're on a single listing page and add "Talk about X" button
            this.detectAndAddListingButton();

            // Event listeners
            this.$sendBtn.on('click', function() {
                debugLog('Send button clicked');
                self.sendMessage();
            });

            this.$input.on('keydown', function(e) {
                if (e.key === 'Enter' && !e.shiftKey) {
                    e.preventDefault();
                    self.sendMessage();
                }
            });

            // Clear conversation button
            this.$clearBtn.on('click', function() {
                debugLog('Clear button clicked');
                self.clearConversation();
            });

            // Show more button (event delegation for dynamically added buttons)
            this.$messages.on('click', '.listeo-ai-show-more-btn', function() {
                var $btn = $(this);
                var $list = $btn.prev('.listeo-ai-results-list');

                // Show all hidden listings
                $list.find('.listeo-ai-listing-hidden').removeClass('listeo-ai-listing-hidden');

                // Hide the button
                $btn.remove();
            });

            // Popular search tag click handler (event delegation)
            $(document).on('click', '.popular-search-tag', function() {
                var $tag = $(this);
                var query = $tag.data('query');
                var $popularSearches = $tag.closest('.listeo-ai-popular-searches');
                var targetChatId = $popularSearches.data('chat-id');

                debugLog('Popular search tag clicked:', { query: query, targetChatId: targetChatId, currentChatId: self.chatId });

                // Only handle if this tag is for the current chat instance
                if (targetChatId === self.chatId) {
                    debugLog('Inserting popular search into input:', query);

                    // Insert query into input field
                    self.$input.val(query);

                    // Focus on input
                    self.$input.focus();

                    // Optionally auto-send the message
                    self.sendMessage();
                }
            });
        },

        /**
         * Load chat configuration
         */
        loadConfig: function() {
            var self = this;

            debugLog('[CONFIG] Loading chat config from WordPress (no OpenAI call):', listeoAiChatConfig.apiBase + '/chat-config');

            $.ajax({
                url: listeoAiChatConfig.apiBase + '/chat-config',
                method: 'GET',
                success: function(response) {
                    debugLog('[CONFIG] ✅ Loaded from WordPress database (no OpenAI API call made)');
                    if (response.success) {
                        self.chatConfig = response.config;
                        self.configLoaded = true;

                        // Enable send button now that config is ready
                        self.$sendBtn.prop('disabled', false);

                        // Enable listing context button if present
                        $('.listeo-ai-load-listing-btn').prop('disabled', false);
                    }
                },
                error: function(xhr) {
                    debugError('Failed to load chat config:', {
                        status: xhr.status,
                        statusText: xhr.statusText,
                        responseText: xhr.responseText,
                        fullError: xhr
                    });

                    var errorMsg = '⚠️ Chat is currently unavailable. ';
                    if (xhr.status === 404) {
                        errorMsg += 'Configuration endpoint not found. Try flushing permalinks: Settings → Permalinks → Save Changes.';
                    } else if (xhr.status === 0) {
                        errorMsg += 'Cannot connect to server. Please check your connection and refresh the page.';
                    } else {
                        errorMsg += 'Please refresh the page or contact support if the problem persists.';
                    }

                    self.addMessage('system', errorMsg);

                    // Don't enable chat if config can't load - it won't work anyway
                    self.configLoaded = false;
                    // Keep send button disabled (already disabled in init)
                }
            });
        },

        /**
         * Check if chat is in Style 2 (elementor-chat-style wrapper)
         */
        isStyle2: function() {
            return this.$wrapper.parent().hasClass('elementor-chat-style');
        },

        /**
         * Expand chat to 80vh (Style 2 only)
         * Called on first message or when loading past conversation
         */
        expandChat: function() {
            if (!this.isStyle2() || this.isExpanded) {
                return; // Already expanded or not Style 2
            }

            debugLog('Expanding chat to 80vh (Style 2 animation)');

            this.$wrapper.addClass('expanded');
            this.isExpanded = true;

            // Scroll to bottom after expansion animation completes
            var self = this;
            setTimeout(function() {
                self.$messages.scrollTop(self.$messages[0].scrollHeight);
            }, 600); // Match CSS transition duration
        },

        /**
         * Send message - Dual Mode Architecture
         *
         * Mode 1 (Listeo Available): Function Calling with Listeo Tools
         *   Flow: User Question → OpenAI with tools → Tool calls → Execute → AI Answer
         *
         * Mode 2 (No Listeo): RAG-First Architecture
         *   Flow: User Question → Universal Search (top 3) → Get Content → AI Answer
         */
        sendMessage: function() {
            var self = this;
            var message = this.$input.val().trim();

            if (!message || this.isProcessing) {
                return;
            }

            // Check if config is still loading
            if (!this.configLoaded) {
                this.addMessage('system', listeoAiChatConfig.strings.loadingConfig);
                debugLog('Config not loaded yet, waiting...');
                return;
            }

            // Check if config loaded
            if (!this.chatConfig) {
                this.addMessage('system', listeoAiChatConfig.strings.errorConfig);
                return;
            }

            // Check if enabled
            if (!this.chatConfig.enabled) {
                this.addMessage('system', listeoAiChatConfig.strings.chatDisabled);
                return;
            }

            // Check rate limits (client-side)
            var rateLimitCheck = this.checkRateLimit();
            if (!rateLimitCheck.allowed) {
                this.addMessage('system', rateLimitCheck.message);
                debugLog('[Rate Limit] Blocked:', rateLimitCheck);
                return;
            }

            // Record message timestamp for rate limiting
            this.recordMessage();

            // Check if this is the first real message (expand chat for Style 2)
            var isFirstMessage = this.conversationHistory.length === 0;

            // Add user message
            this.addMessage('user', message);
            this.$input.val('');
            this.isProcessing = true;
            this.$sendBtn.prop('disabled', true);

            // Expand chat on first message (Style 2 only)
            if (isFirstMessage) {
                this.expandChat();
            }

            // Add loading indicator
            var loadingId = 'loading-' + Date.now();
            this.addMessage('assistant', '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.searchingDatabase, loadingId);

            // ALWAYS use function calling mode - LLM decides whether to search or answer from context
            debugLog('===== FUNCTION CALLING MODE =====');
            debugLog('Tools available:', this.chatConfig.tools ? this.chatConfig.tools.length : 0);
            debugLog('LLM will decide whether to call tools or answer directly from conversation context');
            this.sendMessageWithFunctionCalling(message, loadingId);
        },

        /**
         * Send message with function calling (Listeo tools mode)
         * OpenAI decides which tool to call, we execute it, and send results back
         */
        sendMessageWithFunctionCalling: function(userMessage, loadingId) {
            var self = this;

            // Get valid history slice
            var recentHistory = self.getValidHistorySlice(12);

            // Build system prompt with loaded listing context (if available)
            var systemPrompt = self.chatConfig.system_prompt;
            if (self.loadedListing && self.loadedListing.content) {
                systemPrompt += '\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
                systemPrompt += 'CURRENT LISTING CONTEXT (User is viewing this listing):\n';
                systemPrompt += '━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n';
                systemPrompt += self.loadedListing.content;
                systemPrompt += '\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
                systemPrompt += 'Use this listing information to answer questions about it. Do not search for this listing again.\n';
                systemPrompt += '━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';

                debugLog('[Listing Context] Injected into system prompt:', self.loadedListing.title);
            }

            // Build messages array
            var messages = [
                {
                    role: 'system',
                    content: systemPrompt
                }
            ].concat(recentHistory).concat([
                { role: 'user', content: userMessage }
            ]);

            // Build payload with tools
            var payload = {
                model: self.chatConfig.model,
                messages: messages,
                tools: self.chatConfig.tools,
                tool_choice: 'auto'
            };

            // Check if GPT-5 model
            if (self.chatConfig.model.startsWith('gpt-5')) {
                payload.max_completion_tokens = self.chatConfig.max_tokens;
                payload.verbosity = "medium";
            } else {
                payload.max_tokens = self.chatConfig.max_tokens;
                payload.temperature = self.chatConfig.temperature;
            }

            debugLog('===== SENDING TO OPENAI WITH TOOLS =====');
            debugLog('Messages:', messages.length);
            debugLog('Tools available:', self.chatConfig.tools.length);
            debugLog('===== FULL SYSTEM PROMPT =====');
            debugLog(self.chatConfig.system_prompt);
            debugLog('===== END SYSTEM PROMPT =====');

            // Send to OpenAI
            $.ajax({
                url: listeoAiChatConfig.apiBase + '/chat-proxy',
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': listeoAiChatConfig.nonce
                },
                data: JSON.stringify(payload),
                success: function(data) {
                    var assistantMessage = data.choices[0].message;

                    debugLog('===== OPENAI RESPONSE =====');
                    debugLog('Has tool_calls:', !!assistantMessage.tool_calls);

                    // Check if AI wants to call tools
                    if (assistantMessage.tool_calls && assistantMessage.tool_calls.length > 0) {
                        debugLog('Tool calls requested:', assistantMessage.tool_calls.length);

                        // Update loading message
                        self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                            '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.searchingDatabase
                        );

                        // Execute tool calls
                        self.executeToolCalls(userMessage, assistantMessage, loadingId);
                    } else {
                        // No tools called - AI responded directly (greetings, clarifications, simple questions)
                        debugLog('No tool calls, displaying direct response');
                        debugLog('This handles: greetings, thanks, clarifications, simple questions');

                        // Remove loading
                        self.$messages.find('#' + loadingId).remove();

                        // Display AI's direct response
                        var content = assistantMessage.content || 'No response received.';
                        self.addMessage('assistant', content);

                        // Update conversation history
                        self.conversationHistory.push(
                            { role: 'user', content: userMessage },
                            { role: 'assistant', content: content }
                        );

                        // Save conversation
                        self.saveConversation();

                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                },
                error: function(xhr) {
                    debugError('OpenAI API error:', xhr);
                    self.$messages.find('#' + loadingId).remove();
                    var errorMsg = listeoAiChatConfig.strings.errorGeneral;
                    if (xhr.responseJSON && xhr.responseJSON.error) {
                        errorMsg += ' ' + xhr.responseJSON.error.message;
                    }
                    self.addMessage('system', errorMsg);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Execute tool calls (search_listings, get_listing_details, or search_universal_content)
         */
        executeToolCalls: function(userMessage, assistantMessage, loadingId) {
            var self = this;
            var toolCall = assistantMessage.tool_calls[0]; // Handle first tool call
            var functionName = toolCall.function.name;
            var functionArgs = JSON.parse(toolCall.function.arguments);

            debugLog('===== EXECUTING TOOL CALL =====');
            debugLog('Function:', functionName);
            debugLog('Arguments:', functionArgs);

            if (functionName === 'search_listings') {
                // Call Listeo hybrid search
                self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                    '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.searchingListings
                );

                $.ajax({
                    url: listeoAiChatConfig.apiBase + '/listeo-hybrid-search',
                    method: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify(functionArgs),
                    success: function(response) {
                        if (response.success && response.results && response.results.length > 0) {
                            debugLog('Search results:', response.results.length);

                            // Show results grid
                            var gridHTML = self.formatListingsGrid(response.results);
                            self.$messages.find('#' + loadingId).remove();
                            self.addMessage('assistant', gridHTML);

                            // Update loading for AI response
                            var responseLoadingId = 'loading-response-' + Date.now();
                            self.addMessage('assistant', '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.analyzingResults, responseLoadingId);

                            // Send results back to OpenAI for natural language response
                            self.getFinalResponse(userMessage, assistantMessage, toolCall, response, responseLoadingId);
                        } else {
                            // No results
                            debugLog('No results found');
                            self.$messages.find('#' + loadingId).remove();

                            var noResultsMsg = {
                                total: 0,
                                results: []
                            };

                            // Still send to OpenAI so it can respond naturally
                            self.getFinalResponse(userMessage, assistantMessage, toolCall, noResultsMsg, loadingId);
                        }
                    },
                    error: function(xhr) {
                        debugError('Search error:', xhr);
                        self.$messages.find('#' + loadingId).remove();
                        self.addMessage('system', listeoAiChatConfig.strings.searchFailed);
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                });
            } else if (functionName === 'get_listing_details') {
                // Get listing details
                self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                    '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.analyzingListing
                );

                self.getListingDetails(functionArgs.listing_id, userMessage, assistantMessage, toolCall, loadingId);
            } else if (functionName === 'search_universal_content') {
                // Search universal content (posts/pages/products/docs)
                debugLog('Calling universal content search for:', functionArgs.query);

                self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                    '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.searchingSiteContent
                );

                // Call RAG endpoint directly - it handles everything server-side
                // IMPORTANT: Don't pass chat_history here because:
                // 1. RAG endpoint doesn't use function calling
                // 2. Chat history may contain tool messages that will break the request
                // 3. This is being called AS a tool, so history isn't needed
                $.ajax({
                    url: listeoAiChatConfig.apiBase + '/rag-chat',
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': listeoAiChatConfig.nonce
                    },
                    data: JSON.stringify({
                        query: functionArgs.query,
                        chat_history: [], // Empty history - this is a tool call, not a conversation
                        top_results: functionArgs.top_results || 5
                    }),
                    success: function(response) {
                        if (response.success) {
                            debugLog('RAG response received:', response.sources?.length || 0, 'sources');

                            // Remove loading
                            self.$messages.find('#' + loadingId).remove();

                            // Display answer
                            self.addMessage('assistant', response.answer);

                            // Display source attribution (if sources exist)
                            if (response.sources && response.sources.length > 0) {
                                var sourcesHTML = self.formatSourceAttribution(response.sources);
                                self.addMessage('assistant', sourcesHTML);
                            }

                            // Update conversation history - MUST include complete tool calling sequence
                            self.conversationHistory.push(
                                { role: 'user', content: userMessage },
                                assistantMessage,  // Assistant message WITH tool_calls
                                {
                                    role: 'tool',
                                    tool_call_id: toolCall.id,
                                    content: JSON.stringify({
                                        answer: response.answer,
                                        sources: response.sources || []
                                    })
                                },
                                { role: 'assistant', content: response.answer }  // Final response
                            );

                            // Save conversation
                            self.saveConversation();

                            self.isProcessing = false;
                            self.$sendBtn.prop('disabled', false);
                        } else {
                            debugError('RAG endpoint error:', response.error);
                            self.$messages.find('#' + loadingId).remove();
                            self.addMessage('system', listeoAiChatConfig.strings.errorGeneral + ' ' + (response.error?.message || ''));
                            self.isProcessing = false;
                            self.$sendBtn.prop('disabled', false);
                        }
                    },
                    error: function(xhr) {
                        debugError('RAG AJAX error:', xhr);
                        self.$messages.find('#' + loadingId).remove();
                        var errorMsg = listeoAiChatConfig.strings.errorGeneral;
                        if (xhr.responseJSON && xhr.responseJSON.error) {
                            errorMsg += ' ' + xhr.responseJSON.error.message;
                        }
                        self.addMessage('system', errorMsg);
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                });
            } else if (functionName === 'search_products') {
                // Search WooCommerce products
                self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                    '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.searchingProducts
                );

                $.ajax({
                    url: listeoAiChatConfig.apiBase + '/woocommerce-product-search',
                    method: 'POST',
                    contentType: 'application/json',
                    data: JSON.stringify(functionArgs),
                    success: function(response) {
                        if (response.success && response.results && response.results.length > 0) {
                            debugLog('Product search results:', response.results.length);

                            // Show products grid (same container as listings, but with price instead of location)
                            var gridHTML = self.formatProductsGrid(response.results);
                            self.$messages.find('#' + loadingId).remove();
                            self.addMessage('assistant', gridHTML);

                            // Update loading for AI response
                            var responseLoadingId = 'loading-response-' + Date.now();
                            self.addMessage('assistant', '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.analyzingProducts, responseLoadingId);

                            // Send results back to OpenAI for natural language response
                            self.getFinalResponse(userMessage, assistantMessage, toolCall, response, responseLoadingId);
                        } else {
                            // No results
                            debugLog('No products found');
                            self.$messages.find('#' + loadingId).remove();

                            var noResultsMsg = {
                                total: 0,
                                results: []
                            };

                            // Still send to OpenAI so it can respond naturally
                            self.getFinalResponse(userMessage, assistantMessage, toolCall, noResultsMsg, loadingId);
                        }
                    },
                    error: function(xhr) {
                        debugError('Product search error:', xhr);
                        self.$messages.find('#' + loadingId).remove();
                        self.addMessage('system', listeoAiChatConfig.strings.productSearchFailed);
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                });
            } else if (functionName === 'get_product_details') {
                // Get WooCommerce product details
                self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                    '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.gettingProductDetails
                );

                self.getProductDetails(functionArgs.product_id, userMessage, assistantMessage, toolCall, loadingId);
            } else {
                debugError('Unknown function:', functionName);
                self.$messages.find('#' + loadingId).remove();
                self.addMessage('system', listeoAiChatConfig.strings.unknownFunction);
                self.isProcessing = false;
                self.$sendBtn.prop('disabled', false);
            }
        },

        /**
         * Build enriched context from retrieved content
         */
        buildEnrichedContext: function(contentResponses, sourceResults) {
            var context = "RELEVANT CONTENT FROM SITE:\n\n";

            contentResponses.forEach(function(response, index) {
                // Handle both direct response and array response from $.when
                var data = response[0] || response;

                if (data.success && data.structured_content) {
                    var source = sourceResults[index];
                    context += "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
                    context += "SOURCE " + (index + 1) + ": " + data.title + "\n";
                    context += "Type: " + data.post_type.charAt(0).toUpperCase() + data.post_type.slice(1) + "\n";
                    context += "URL: " + data.url + "\n";
                    context += "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
                    context += data.structured_content + "\n\n\n";
                }
            });

            return context;
        },

        /**
         * Answer with RAG (server-side search + context + OpenAI in ONE call)
         * Simple non-streaming AJAX approach
         */
        answerWithContext: function(userMessage, enrichedContext, sourceResults, loadingId) {
            var self = this;

            // Build conversation history (last 4 exchanges = 8 messages)
            var recentHistory = [];
            var historySlice = self.conversationHistory.slice(-8);
            historySlice.forEach(function(msg) {
                recentHistory.push({ role: msg.role, content: msg.content });
            });

            var payload = {
                query: userMessage,
                chat_history: recentHistory,
                top_results: 3
            };

            debugLog('===== CALLING RAG ENDPOINT (SERVER-SIDE RAG) =====');
            debugLog('🔍 Query:', userMessage);
            debugLog('📚 Chat history length:', recentHistory.length, 'messages');
            debugLog('🎯 Top results to retrieve:', payload.top_results);
            debugLog('🌐 Endpoint:', listeoAiChatConfig.apiBase + '/rag-chat');
            debugLog('📦 Full payload:', payload);
            debugLog('⚙️  Listeo available:', self.chatConfig.listeo_available);
            debugLog('🔧 Post types will be filtered server-side based on Listeo availability');
            debugLog('===== FULL SYSTEM PROMPT =====');
            debugLog(self.chatConfig.system_prompt);
            debugLog('===== END SYSTEM PROMPT =====');

            $.ajax({
                url: listeoAiChatConfig.apiBase + '/rag-chat',
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': listeoAiChatConfig.nonce
                },
                data: JSON.stringify(payload),
                success: function(response) {
                    debugLog('===== RAG ENDPOINT RESPONSE =====');
                    debugLog('✅ Response received');
                    debugLog('📄 Full response:', response);

                    if (!response.success) {
                        debugError('❌ RAG endpoint returned success:false');
                        debugError('Error object:', response.error);
                        self.$messages.find('#' + loadingId).remove();
                        self.addMessage('system', listeoAiChatConfig.strings.errorGeneral + ' ' + (response.error?.message || ''));
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                        return;
                    }

                    var answer = response.answer;
                    var sources = response.sources || [];

                    debugLog('===== RAG RESPONSE DETAILS =====');
                    if (response.query_info) {
                        debugLog('📝 Query Transformation:');
                        debugLog('  Original query:', response.query_info.original_query);
                        debugLog('  Optimized query:', response.query_info.optimized_query);
                    }
                    debugLog('💬 Answer length:', answer ? answer.length : 0, 'chars');
                    debugLog('📚 Sources found:', sources.length);
                    if (sources.length > 0) {
                        debugLog('📖 Source details:');
                        sources.forEach(function(source, idx) {
                            debugLog('  ' + (idx + 1) + '. ' + source.title + ' (' + source.type + ') - ' + source.url);
                        });
                    } else {
                        debugLog('⚠️  No sources returned from search');
                    }
                    debugLog('⏱️  Performance:', response.performance);
                    debugLog('🤖 Model used:', response.model);
                    if (response.usage) {
                        debugLog('💰 Token usage:', response.usage);
                    }

                    // Remove loading
                    self.$messages.find('#' + loadingId).remove();

                    // Display answer
                    if (answer) {
                        self.addMessage('assistant', answer);
                        debugLog('✅ Answer displayed to user');
                    } else {
                        debugError('⚠️  Empty answer received');
                    }

                    // Display source attribution (if sources exist)
                    if (sources && sources.length > 0) {
                        var sourcesHTML = self.formatSourceAttribution(sources);
                        self.addMessage('assistant', sourcesHTML);
                        debugLog('✅ Source attribution displayed');
                    }

                    // Update conversation history (simple, no tool calls)
                    self.conversationHistory.push(
                        { role: 'user', content: userMessage },
                        { role: 'assistant', content: answer }
                    );

                    // Save conversation
                    self.saveConversation();

                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                    debugLog('✅ RAG flow completed successfully');
                },
                error: function(xhr) {
                    debugError('===== RAG ENDPOINT ERROR =====');
                    debugError('❌ AJAX request failed');
                    debugError('Status:', xhr.status);
                    debugError('Status Text:', xhr.statusText);
                    debugError('Response Text:', xhr.responseText);
                    debugError('Response JSON:', xhr.responseJSON);
                    debugError('Full XHR object:', xhr);

                    self.$messages.find('#' + loadingId).remove();

                    var errorMsg = listeoAiChatConfig.strings.errorGeneral;
                    if (xhr.responseJSON && xhr.responseJSON.error) {
                        errorMsg += ' ' + xhr.responseJSON.error.message;
                        debugError('Parsed error message:', xhr.responseJSON.error.message);
                    } else if (xhr.status === 404) {
                        errorMsg += ' Endpoint not found. Check if permalinks need flushing.';
                    } else if (xhr.status === 0) {
                        errorMsg += ' Network error. Check your connection.';
                    }

                    self.addMessage('system', errorMsg);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Format source attribution cards
         */
        formatSourceAttribution: function(sourceResults) {
            // var self = this;
            // var html = '<div class="listeo-ai-sources">';
            // html += '<div class="listeo-ai-sources-label">📚 Sources used:</div>';
            // html += '<div class="listeo-ai-sources-list">';

            // sourceResults.forEach(function(source, index) {
            //     var postTypeLabel = source.post_type ? source.post_type.charAt(0).toUpperCase() + source.post_type.slice(1) : 'Content';
            //     var icon = self.getIconForPostType(source.post_type);

            //     html += '<a href="' + source.url + '" class="listeo-ai-source-card" target="_blank">';
            //     html += '  <span class="source-icon">' + icon + '</span>';
            //     html += '  <span class="source-info">';
            //     html += '    <span class="source-title">' + source.title + '</span>';
            //     html += '    <span class="source-type">' + postTypeLabel + '</span>';
            //     html += '  </span>';
            //     html += '</a>';
            // });

            // html += '</div></div>';
            // return html;
        },

        /**
         * Get icon for post type
         */
        getIconForPostType: function(postType) {
            var icons = {
                'post': '<i class="fa fa-file-text-o"></i>',
                'page': '<i class="fa fa-file-o"></i>',
                'listing': '<i class="fa fa-map-marker"></i>',
                'product': '<i class="fa fa-shopping-cart"></i>'
            };
            return icons[postType] || '<i class="fa fa-file-o"></i>';
        },

        /**
         * Search listings via API
         */
        searchListings: function(params, callback) {
            // Use Listeo-specific endpoint if available
            var endpoint = this.chatConfig.listeo_available ? '/listeo-hybrid-search' : '/universal-search';

            debugLog('[searchListings] Using endpoint:', endpoint, '(Listeo available:', this.chatConfig.listeo_available + ')');

            $.ajax({
                url: listeoAiChatConfig.apiBase + endpoint,
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify(params),
                success: function(response) {
                    callback(response);
                },
                error: function(xhr) {
                    callback({ success: false, error: xhr.responseJSON?.error || 'Search failed' });
                }
            });
        },

        /**
         * Universal content search via API (any post type)
         */
        searchContent: function(params, callback) {
            $.ajax({
                url: listeoAiChatConfig.apiBase + '/universal-search',
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify(params),
                success: function(response) {
                    callback(response);
                },
                error: function(xhr) {
                    callback({ success: false, error: xhr.responseJSON?.error || 'Universal search failed' });
                }
            });
        },

        /**
         * Get content details via API (any post type)
         */
        getContentDetails: function(postId, userMessage, assistantMessage, toolCall, loadingId) {
            var self = this;

            $.ajax({
                url: listeoAiChatConfig.apiBase + '/get-content',
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify({ post_id: postId }),
                success: function(response) {
                    if (response.success) {
                        debugLog('===== CONTENT DETAILS =====');
                        debugLog('Post ID:', response.post_id);
                        debugLog('Post Type:', response.post_type);
                        debugLog('Title:', response.title);
                        debugLog('Structured content length:', response.structured_content.length);

                        // Update loading message
                        self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                            '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.analyzingContent
                        );

                        // Send structured content to OpenAI for natural response
                        self.getDetailsResponse(userMessage, assistantMessage, toolCall, response, loadingId);
                    } else {
                        self.$messages.find('#' + loadingId).remove();
                        self.addMessage('system', listeoAiChatConfig.strings.contentNotFound);
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                },
                error: function(xhr) {
                    self.$messages.find('#' + loadingId).remove();
                    self.addMessage('system', listeoAiChatConfig.strings.errorGettingContent);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Format universal content grid (for posts, pages, products, etc.)
         */
        formatContentGrid: function(results) {
            var self = this;
            var html = '<div class="listeo-ai-results-list">';

            results.forEach(function(item, index) {
                var thumbnail = item.featured_image || listeoAiChatConfig.placeholderImage || '';
                var title = item.title || 'Untitled';
                var excerpt = item.excerpt || '';
                var postType = item.post_type || 'post';
                var url = item.url || '#';

                // Best Match badge for first result
                var bestMatchBadge = '';
                if (index === 0) {
                    bestMatchBadge = '<div class="match-badge best">' + (listeoAiChatConfig.strings.bestMatch || 'Best Match') + '</div>';
                }

                // Add hidden class to items after the first 3
                var hiddenClass = index >= 3 ? ' listeo-ai-listing-hidden' : '';

                // Format post type label
                var postTypeLabel = postType.charAt(0).toUpperCase() + postType.slice(1);

                html += '<a href="' + url + '" class="listeo-ai-listing-item' + hiddenClass + '">';

                // Only render thumbnail if hideImages is not enabled
                if (!self.hideImages && thumbnail) {
                    html += '  <div class="listeo-ai-listing-thumbnail">';
                    html += '    <img src="' + thumbnail + '" alt="' + title + '">';
                    html += '  </div>';
                }

                html += '  <div class="listeo-ai-listing-details">';
                html += '    <div class="listeo-ai-listing-main">';
                html += '      <h3 class="listeo-ai-listing-title">' + title + ' ' + bestMatchBadge + '</h3>';
                if (excerpt) {
                    html += '      <p class="listeo-ai-listing-excerpt">' + excerpt + '</p>';
                }
                html += '      <div class="listeo-ai-listing-meta">';
                html += '        <span class="content-type"><i class="fa fa-file-text-o"></i> ' + postTypeLabel + '</span>';
                html += '      </div>';
                html += '    </div>';
                html += '  </div>';
                html += '</a>';
            });

            html += '</div>';

            // Add "Show more" button if more than 3 results
            if (results.length > 3) {
                html += '<button class="listeo-ai-show-more-btn">' + listeoAiChatConfig.strings.showMore.replace('%d', results.length - 3) + '</button>';
            }

            return html;
        },

        /**
         * Safely slice conversation history without breaking tool calling sequences
         * OpenAI requires: assistant[tool_calls] → tool → assistant (complete sequence)
         */
        getValidHistorySlice: function(maxMessages) {
            if (this.conversationHistory.length <= maxMessages) {
                return this.conversationHistory;
            }

            var sliced = this.conversationHistory.slice(-maxMessages);

            // Remove any orphaned 'tool' messages at the start (tool without preceding assistant)
            while (sliced.length > 0 && sliced[0].role === 'tool') {
                debugLog('[History Validation] Removing orphaned tool message from start');
                sliced.shift();
            }

            // Comprehensive validation: scan entire array and remove incomplete sequences
            var validated = [];
            for (var i = 0; i < sliced.length; i++) {
                var msg = sliced[i];

                // If this is an assistant message with tool_calls
                if (msg.role === 'assistant' && msg.tool_calls) {
                    // Check if next message is a tool response
                    var nextMsg = sliced[i + 1];
                    if (!nextMsg || nextMsg.role !== 'tool') {
                        debugLog('[History Validation] Removing assistant+tool_calls at index ' + i + ' - no tool response follows');
                        // Skip this message and all remaining (incomplete sequence at end)
                        break;
                    }

                    // Valid sequence start - add assistant message
                    validated.push(msg);

                } else if (msg.role === 'tool') {
                    // Tool message should only appear after assistant+tool_calls (already validated above)
                    validated.push(msg);

                } else {
                    // Regular user or assistant message (no tool_calls)
                    validated.push(msg);
                }
            }

            debugLog('[History Validation] Original: ' + sliced.length + ' messages, Validated: ' + validated.length + ' messages');
            return validated;
        },

        /**
         * Get listing details via API
         */
        getListingDetails: function(listingId, userMessage, assistantMessage, toolCall, loadingId) {
            var self = this;

            // Use Listeo-specific endpoint if available
            var endpoint = this.chatConfig.listeo_available ? '/listeo-listing-details' : '/get-content';
            var param = this.chatConfig.listeo_available ? { listing_id: listingId } : { post_id: listingId };

            debugLog('[getListingDetails] Using endpoint:', endpoint, '(Listeo available:', this.chatConfig.listeo_available + ')');

            $.ajax({
                url: listeoAiChatConfig.apiBase + endpoint,
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify(param),
                success: function(response) {
                    if (response.success) {
                        debugLog('===== LISTING DETAILS =====');
                        debugLog('Listing/Post ID:', response.listing_id || response.post_id);
                        debugLog('Title:', response.title);
                        debugLog('Structured content length:', response.structured_content.length);

                        // Update loading message (don't remove - keep showing progress)
                        self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                            '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.analyzingListing
                        );

                        // Send structured content to OpenAI for natural response
                        self.getDetailsResponse(userMessage, assistantMessage, toolCall, response, loadingId);
                    } else {
                        self.$messages.find('#' + loadingId).remove();
                        self.addMessage('system', listeoAiChatConfig.strings.listingNotFound);
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                },
                error: function(xhr) {
                    self.$messages.find('#' + loadingId).remove();
                    self.addMessage('system', listeoAiChatConfig.strings.errorGettingDetails);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Get product details via API (WooCommerce)
         */
        getProductDetails: function(productId, userMessage, assistantMessage, toolCall, loadingId) {
            var self = this;

            $.ajax({
                url: listeoAiChatConfig.apiBase + '/woocommerce-product-details',
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify({ product_id: productId }),
                success: function(response) {
                    if (response.success) {
                        debugLog('===== PRODUCT DETAILS =====');
                        debugLog('Product ID:', response.product_id);
                        debugLog('Title:', response.title);
                        debugLog('Structured content length:', response.structured_content.length);

                        // Update loading message
                        self.$messages.find('#' + loadingId + ' .listeo-ai-chat-message-content').html(
                            '<span class="listeo-ai-chat-loading"></span> ' + listeoAiChatConfig.strings.analyzingProduct
                        );

                        // Send structured content to OpenAI for natural response
                        self.getDetailsResponse(userMessage, assistantMessage, toolCall, response, loadingId);
                    } else {
                        self.$messages.find('#' + loadingId).remove();
                        self.addMessage('system', listeoAiChatConfig.strings.productNotFound);
                        self.isProcessing = false;
                        self.$sendBtn.prop('disabled', false);
                    }
                },
                error: function(xhr) {
                    self.$messages.find('#' + loadingId).remove();
                    self.addMessage('system', listeoAiChatConfig.strings.errorGettingProduct);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Get AI response for listing details
         */
        getDetailsResponse: function(userMessage, assistantMessage, toolCall, detailsResponse, loadingId) {
            var self = this;

            // Get valid history slice without breaking tool calling sequences
            var recentHistory = self.getValidHistorySlice(12);

            // Build system prompt with loaded listing context (if available)
            var systemPrompt = self.chatConfig.system_prompt;
            if (self.loadedListing && self.loadedListing.content) {
                systemPrompt += '\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
                systemPrompt += 'CURRENT LISTING CONTEXT (User is viewing this listing):\n';
                systemPrompt += '━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n';
                systemPrompt += self.loadedListing.content;
                systemPrompt += '\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
                systemPrompt += 'Use this listing information to answer questions about it.\n';
                systemPrompt += '━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
            }

            // Build API payload
            var payload = {
                model: self.chatConfig.model,
                messages: [
                    {
                        role: 'system',
                        content: systemPrompt
                    }
                ].concat(recentHistory).concat([
                    { role: 'user', content: userMessage },
                    assistantMessage,
                    {
                        role: 'tool',
                        tool_call_id: toolCall.id,
                        content: detailsResponse.structured_content  // Semantic embedding content
                    }
                ])
            };

            // Check if GPT-5 model
            if (self.chatConfig.model.startsWith('gpt-5')) {
                payload.max_completion_tokens = self.chatConfig.max_tokens;
                payload.verbosity = "medium";
            } else {
                payload.max_tokens = self.chatConfig.max_tokens;
                payload.temperature = self.chatConfig.temperature;
            }

            debugLog('===== SENDING LISTING DETAILS TO OPENAI =====');
            debugLog('Payload messages count:', payload.messages.length);
            debugLog('Tool response content length:', detailsResponse.structured_content.length, 'chars');
            debugLog('Content preview (first 500 chars):', detailsResponse.structured_content.substring(0, 500));
            debugLog('Content preview (last 500 chars):', detailsResponse.structured_content.substring(detailsResponse.structured_content.length - 500));
            debugLog('Full payload:', payload);

            $.ajax({
                url: listeoAiChatConfig.apiBase + '/chat-proxy',
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': listeoAiChatConfig.nonce
                },
                data: JSON.stringify(payload),
                success: function(data) {
                    var finalMessage = data.choices[0].message.content;

                    // Remove loading indicator
                    self.$messages.find('#' + loadingId).remove();

                    // Add AI response
                    self.addMessage('assistant', finalMessage);

                    // Update history - MUST include complete tool calling sequence
                    self.conversationHistory.push(
                        { role: 'user', content: userMessage },
                        assistantMessage,  // Assistant message WITH tool_calls
                        {
                            role: 'tool',
                            tool_call_id: toolCall.id,
                            content: detailsResponse.structured_content
                        },
                        { role: 'assistant', content: finalMessage }  // Final response
                    );

                    // Save conversation
                    self.saveConversation();

                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                },
                error: function(xhr) {
                    // Remove loading indicator
                    self.$messages.find('#' + loadingId).remove();

                    // Enhanced error logging
                    debugError('===== OPENAI API ERROR (getDetailsResponse) =====');
                    debugError('Status:', xhr.status);
                    debugError('Response:', xhr.responseJSON);
                    debugError('Full XHR:', xhr);

                    var errorMsg = listeoAiChatConfig.strings.errorGeneral;
                    if (xhr.responseJSON && xhr.responseJSON.error) {
                        errorMsg += ' OpenAI Error: ' + xhr.responseJSON.error.message;
                        debugError('OpenAI Error Message:', xhr.responseJSON.error.message);
                        debugError('OpenAI Error Type:', xhr.responseJSON.error.type);
                        debugError('OpenAI Error Code:', xhr.responseJSON.error.code);
                    }

                    self.addMessage('system', errorMsg);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Get final response from OpenAI
         */
        getFinalResponse: function(userMessage, assistantMessage, toolCall, apiResult, loadingId) {
            var self = this;

            // Get valid history slice without breaking tool calling sequences
            var recentHistory = self.getValidHistorySlice(12);

            // Send listing IDs, titles, addresses, and URLs - AI can call get_listing_details() for more info
            var condensedResults = {
                total: apiResult.total,
                listings: apiResult.results ? apiResult.results.map(function(r) {
                    return {
                        id: r.id,
                        title: r.title,
                        address: r.location?.address || '',
                        url: r.url || ''
                    };
                }) : []
            };

            debugLog('===== CONDENSED RESULTS SENT TO AI =====');
            debugLog(JSON.stringify(condensedResults, null, 2));

            // Build system prompt with loaded listing context (if available)
            var systemPrompt = self.chatConfig.system_prompt;
            if (self.loadedListing && self.loadedListing.content) {
                systemPrompt += '\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
                systemPrompt += 'CURRENT LISTING CONTEXT (User is viewing this listing):\n';
                systemPrompt += '━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n';
                systemPrompt += self.loadedListing.content;
                systemPrompt += '\n\n━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
                systemPrompt += 'Use this listing information to answer questions about it.\n';
                systemPrompt += '━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n';
            }

            // Build API payload for final response
            var payload = {
                model: self.chatConfig.model,
                messages: [
                    {
                        role: 'system',
                        content: systemPrompt
                    }
                ].concat(recentHistory).concat([
                    { role: 'user', content: userMessage },
                    assistantMessage,
                    {
                        role: 'tool',
                        tool_call_id: toolCall.id,
                        content: JSON.stringify(condensedResults)
                    }
                ])
            };

            // Check if GPT-5 model (use new API parameters)
            if (self.chatConfig.model.startsWith('gpt-5')) {
                payload.max_completion_tokens = self.chatConfig.max_tokens;
                payload.verbosity = "medium";
                // GPT-5 doesn't support temperature parameter
            } else {
                // Traditional GPT-4 parameters
                payload.max_tokens = self.chatConfig.max_tokens;
                payload.temperature = self.chatConfig.temperature;
            }

            debugLog('===== SENDING MESSAGE TO OPENAI (SECOND CALL - SUMMARY) =====');
            debugLog('Conversation history length:', recentHistory.length, 'messages');
            debugLog('===== FULL SYSTEM PROMPT =====');
            debugLog(payload.messages[0].content);
            debugLog('===== END SYSTEM PROMPT =====');
            debugLog('Full messages array:', payload.messages);
            debugLog('Condensed results being sent:', condensedResults);
            debugLog('Complete payload:', payload);

            $.ajax({
                url: listeoAiChatConfig.apiBase + '/chat-proxy',
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': listeoAiChatConfig.nonce
                },
                data: JSON.stringify(payload),
                success: function(data) {
                    var finalMessage = data.choices[0].message.content;

                    // Remove loading indicator
                    self.$messages.find('#' + loadingId).remove();

                    // Add text response ONLY (grid already shown above)
                    self.addMessage('assistant', finalMessage);

                    // Update history - MUST include complete tool calling sequence
                    self.conversationHistory.push(
                        { role: 'user', content: userMessage },
                        assistantMessage,  // Assistant message WITH tool_calls
                        {
                            role: 'tool',
                            tool_call_id: toolCall.id,
                            content: JSON.stringify(condensedResults)
                        },
                        { role: 'assistant', content: finalMessage }  // Final response
                    );

                    // Save conversation to localStorage
                    self.saveConversation();

                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                },
                error: function(xhr) {
                    self.$messages.find('#' + loadingId).remove();
                    self.addMessage('system', listeoAiChatConfig.strings.errorGeneral);
                    self.isProcessing = false;
                    self.$sendBtn.prop('disabled', false);
                }
            });
        },

        /**
         * Format listings grid
         */
        formatListingsGrid: function(results) {
            var self = this; // Capture 'this' for use in forEach callback
            var html = '<div class="listeo-ai-results-list">';

            results.forEach(function(listing, index) {
                // Use theme placeholder (matches frontend)
                var thumbnail = listing.featured_image || listeoAiChatConfig.placeholderImage || '';
                var title = listing.title || 'Untitled';
                var excerpt = listing.excerpt || '';
                var location = listing.location?.address || '';
                var rating = listing.rating?.average || 0;
                var ratingCount = listing.rating?.count || 0;
                var url = listing.url || '#';

                // Best Match badge for first result (matches frontend)
                var bestMatchBadge = '';
                if (index === 0) {
                    bestMatchBadge = '<div class="match-badge best">' + (listeoAiChatConfig.strings.bestMatch || 'Best Match') + '</div>';
                }

                // Add hidden class to items after the first 3
                var hiddenClass = index >= 3 ? ' listeo-ai-listing-hidden' : '';

                html += '<a href="' + url + '" class="listeo-ai-listing-item' + hiddenClass + '">';

                // Only render thumbnail if hideImages is not enabled (use instance setting)
                if (!self.hideImages) {
                    html += '  <div class="listeo-ai-listing-thumbnail">';
                    html += '    <img src="' + thumbnail + '" alt="' + title + '">';
                    html += '  </div>';
                }

                html += '  <div class="listeo-ai-listing-details">';
                html += '    <div class="listeo-ai-listing-main">';
                html += '      <h3 class="listeo-ai-listing-title">' + title + ' ' + bestMatchBadge + '</h3>';
                if (excerpt) {
                    html += '      <p class="listeo-ai-listing-excerpt">' + excerpt + '</p>';
                }
                html += '      <div class="listeo-ai-listing-meta">';
                if (location) {
                    html += '        <span class="address"><i class="fa fa-map-marker"></i> ' + location + '</span>';
                }
                if (rating > 0) {
                    html += '        <span class="listeo-ai-listing-rating"><i class="fa fa-star"></i> ' + parseFloat(rating).toFixed(1) + '</span>';
                }
                html += '      </div>';
                html += '    </div>';
                html += '  </div>';
                html += '</a>';
            });

            html += '</div>';

            // Add "Show more" button if more than 3 results
            if (results.length > 3) {
                html += '<button class="listeo-ai-show-more-btn">' + listeoAiChatConfig.strings.showMore.replace('%d', results.length - 3) + '</button>';
            }

            return html;
        },

        /**
         * Format products grid (for WooCommerce products)
         */
        formatProductsGrid: function(results) {
            var self = this;
            var html = '<div class="listeo-ai-results-list">';

            results.forEach(function(product, index) {
                // Use theme placeholder (matches frontend)
                var thumbnail = product.featured_image || listeoAiChatConfig.placeholderImage || '';
                var title = product.title || 'Untitled';
                var excerpt = product.excerpt || '';
                var price = product.price?.formatted || '';
                var regularPrice = product.price?.regular || '';
                var salePrice = product.price?.sale || null;
                var onSale = product.on_sale || false;
                var stockStatus = product.stock_status || '';
                var rating = product.rating?.average || 0;
                var ratingCount = product.rating?.count || 0;
                var url = product.url || '#';

                // Best Match badge for first result
                var bestMatchBadge = '';
                if (index === 0) {
                    bestMatchBadge = '<div class="match-badge best">' + (listeoAiChatConfig.strings.bestMatch || 'Best Match') + '</div>';
                }

                // Add hidden class to items after the first 3
                var hiddenClass = index >= 3 ? ' listeo-ai-listing-hidden' : '';

                html += '<a href="' + url + '" class="listeo-ai-listing-item' + hiddenClass + '">';

                // Only render thumbnail if hideImages is not enabled
                if (!self.hideImages) {
                    html += '  <div class="listeo-ai-listing-thumbnail">';
                    html += '    <img src="' + thumbnail + '" alt="' + title + '">';
                    html += '  </div>';
                }

                html += '  <div class="listeo-ai-listing-details">';
                html += '    <div class="listeo-ai-listing-main">';
                html += '      <h3 class="listeo-ai-listing-title">' + title + ' ' + bestMatchBadge + '</h3>';
                if (excerpt) {
                    html += '      <p class="listeo-ai-listing-excerpt">' + excerpt + '</p>';
                }
                html += '      <div class="listeo-ai-listing-meta">';

                // Price display (with sale price handling)
                if (price) {
                    if (onSale && salePrice) {
                        html += '        <span class="product-price"><span class="regular-price">' + regularPrice + '</span> <span class="sale-price">' + salePrice + '</span></span>';
                    } else {
                        html += '        <span class="product-price">' + price + '</span>';
                    }
                }

                // Stock status
                if (stockStatus === 'instock') {
                    html += '        <span class="stock-status in-stock"><i class="fa fa-check-circle"></i> ' + listeoAiChatConfig.strings.inStock + '</span>';
                } else if (stockStatus === 'outofstock') {
                    html += '        <span class="stock-status out-of-stock"><i class="fa fa-times-circle"></i> ' + listeoAiChatConfig.strings.outOfStock + '</span>';
                }

                // Rating
                if (rating > 0) {
                    html += '        <span class="listeo-ai-listing-rating"><i class="fa fa-star"></i> ' + parseFloat(rating).toFixed(1) + '</span>';
                }

                html += '      </div>';
                html += '    </div>';
                html += '  </div>';
                html += '</a>';
            });

            html += '</div>';

            // Add "Show more" button if more than 3 results
            if (results.length > 3) {
                html += '<button class="listeo-ai-show-more-btn">' + listeoAiChatConfig.strings.showMore.replace('%d', results.length - 3) + '</button>';
            }

            return html;
        },

        /**
         * Add message to chat
         */
        addMessage: function(role, content, id) {
            // Skip if content is empty/undefined
            if (!content) {
                console.warn('Skipping message with empty content');
                return;
            }

            var $message = $('<div class="listeo-ai-chat-message listeo-ai-chat-message-' + role + '"></div>');
            if (id) {
                $message.attr('id', id);
            }

            // Check if content contains search results grid
            if (content.indexOf('listeo-ai-results-list') !== -1) {
                $message.addClass('chat-message-results');
            }

            var $content = $('<div class="listeo-ai-chat-message-content"></div>');

            // Check if content contains HTML
            if (content.indexOf('<') !== -1) {
                $content.html(content);
            } else {
                $content.text(content);
            }

            $message.append($content);
            this.$messages.append($message);
            this.$messages.scrollTop(this.$messages[0].scrollHeight);
        },

        /**
         * Save conversation to localStorage
         */
        saveConversation: function() {
            try {
                var data = {
                    history: this.conversationHistory,
                    messages: [],
                    timestamp: Date.now()
                };

                // Save message HTML for display (skip listing-action messages)
                this.$messages.find('.listeo-ai-chat-message').each(function() {
                    var $msg = $(this);

                    // Skip listing-action messages (don't persist them)
                    if ($msg.hasClass('listeo-ai-chat-message-listing-action')) {
                        return; // continue to next message
                    }

                    var role = 'assistant';
                    if ($msg.hasClass('listeo-ai-chat-message-user')) {
                        role = 'user';
                    } else if ($msg.hasClass('listeo-ai-chat-message-system')) {
                        role = 'system';
                    }

                    data.messages.push({
                        role: role,
                        content: $msg.find('.listeo-ai-chat-message-content').html()
                    });
                });

                localStorage.setItem(this.storageKey, JSON.stringify(data));
                debugLog('Conversation saved to localStorage');
            } catch (e) {
                debugError('Failed to save conversation:', e);
            }
        },

        /**
         * Load conversation from localStorage
         */
        loadConversation: function() {
            try {
                var data = localStorage.getItem(this.storageKey);
                if (!data) {
                    debugLog('No saved conversation found - showing fresh welcome message');
                    // Replace HTML welcome message with current setting from JS
                    this.$messages.empty();
                    this.addMessage('system', listeoAiChatConfig.strings.welcomeMessage);
                    return;
                }

                data = JSON.parse(data);

                // Check if conversation is less than 24 hours old
                var hoursSinceLastMessage = (Date.now() - data.timestamp) / (1000 * 60 * 60);
                if (hoursSinceLastMessage > 24) {
                    debugLog('Conversation expired (older than 24 hours)');
                    localStorage.removeItem(this.storageKey);
                    // Show fresh welcome message
                    this.$messages.empty();
                    this.addMessage('system', listeoAiChatConfig.strings.welcomeMessage);
                    return;
                }

                // Restore conversation history
                this.conversationHistory = data.history || [];

                // Clear welcome message and restore messages
                this.$messages.empty();

                var self = this;
                if (data.messages && data.messages.length > 0) {
                    data.messages.forEach(function(msg) {
                        // Skip messages with empty content
                        if (msg.content) {
                            self.addMessage(msg.role, msg.content);
                        }
                    });
                    debugLog('Loaded ' + data.messages.length + ' messages from localStorage');

                    // Expand chat when loading past conversation (Style 2 only)
                    this.expandChat();
                } else {
                    // Show welcome message if no saved messages
                    this.addMessage('system', listeoAiChatConfig.strings.welcomeMessage);
                }
            } catch (e) {
                debugError('Failed to load conversation:', e);
                // Clear and show fresh welcome message on error
                this.$messages.empty();
                this.addMessage('system', listeoAiChatConfig.strings.welcomeMessage);
            }
        },

        /**
         * Clear conversation
         */
        clearConversation: function() {
            // Clear localStorage
            localStorage.removeItem(this.storageKey);

            // Clear loaded listing context
            this.loadedListing = null;
            this.clearLoadedListingFromStorage();

            // Clear conversation history
            this.conversationHistory = [];

            // Clear messages and show welcome
            this.$messages.empty();
            this.addMessage('system', listeoAiChatConfig.strings.welcomeMessage);

            // Re-detect and add listing button if on listing page
            this.detectAndAddListingButton();

            debugLog('Conversation cleared');
        },

        /**
         * Check rate limits (client-side using localStorage)
         * Returns {allowed: boolean, message: string}
         */
        checkRateLimit: function() {
            var now = Date.now();
            var timestamps = this.getMessageTimestamps();

            // Clean up old timestamps (older than 24 hours)
            timestamps = timestamps.filter(function(ts) {
                return (now - ts) < 86400000; // 24 hours in ms
            });

            // Check Tier 1: X messages per minute
            var tier1Window = this.rateLimits.tier1.window * 1000; // Convert to ms
            var tier1Count = timestamps.filter(function(ts) {
                return (now - ts) < tier1Window;
            }).length;

            if (tier1Count >= this.rateLimits.tier1.limit) {
                var tier1Wait = Math.ceil((timestamps[timestamps.length - tier1Count] + tier1Window - now) / 1000);
                return {
                    allowed: false,
                    tier: 'tier1',
                    message: this.getRateLimitMessage('minute', this.rateLimits.tier1.limit, tier1Wait)
                };
            }

            // Check Tier 2: X messages per 15 minutes
            var tier2Window = this.rateLimits.tier2.window * 1000;
            var tier2Count = timestamps.filter(function(ts) {
                return (now - ts) < tier2Window;
            }).length;

            if (tier2Count >= this.rateLimits.tier2.limit) {
                var tier2Wait = Math.ceil((timestamps[timestamps.length - tier2Count] + tier2Window - now) / 1000);
                return {
                    allowed: false,
                    tier: 'tier2',
                    message: this.getRateLimitMessage('15 minutes', this.rateLimits.tier2.limit, tier2Wait)
                };
            }

            // Check Tier 3: X messages per day
            var tier3Window = this.rateLimits.tier3.window * 1000;
            var tier3Count = timestamps.filter(function(ts) {
                return (now - ts) < tier3Window;
            }).length;

            if (tier3Count >= this.rateLimits.tier3.limit) {
                var tier3Wait = Math.ceil((timestamps[timestamps.length - tier3Count] + tier3Window - now) / 1000);
                return {
                    allowed: false,
                    tier: 'tier3',
                    message: this.getRateLimitMessage('day', this.rateLimits.tier3.limit, tier3Wait)
                };
            }

            // Save cleaned timestamps
            this.saveMessageTimestamps(timestamps);

            return { allowed: true };
        },

        /**
         * Record message timestamp
         */
        recordMessage: function() {
            var timestamps = this.getMessageTimestamps();
            timestamps.push(Date.now());
            this.saveMessageTimestamps(timestamps);
        },

        /**
         * Get message timestamps from localStorage
         */
        getMessageTimestamps: function() {
            try {
                var data = localStorage.getItem(this.rateLimitStorageKey);
                if (!data) return [];

                var parsed = JSON.parse(data);
                return Array.isArray(parsed) ? parsed : [];
            } catch (e) {
                debugError('[Rate Limit] Failed to parse timestamps:', e);
                return [];
            }
        },

        /**
         * Save message timestamps to localStorage
         */
        saveMessageTimestamps: function(timestamps) {
            try {
                localStorage.setItem(this.rateLimitStorageKey, JSON.stringify(timestamps));
            } catch (e) {
                debugError('[Rate Limit] Failed to save timestamps:', e);
            }
        },

        /**
         * Get human-readable rate limit error message
         */
        getRateLimitMessage: function(period, limit, waitSeconds) {
            var waitText = '';
            if (waitSeconds >= 60) {
                var minutes = Math.ceil(waitSeconds / 60);
                var minuteText = minutes > 1 ? listeoAiChatConfig.strings.minutes : listeoAiChatConfig.strings.minute;
                waitText = minutes + ' ' + minuteText;
            } else {
                var secondText = waitSeconds > 1 ? listeoAiChatConfig.strings.seconds : listeoAiChatConfig.strings.second;
                waitText = waitSeconds + ' ' + secondText;
            }

            return '⏱️ ' + listeoAiChatConfig.strings.rateLimitPrefix + ' ' + limit + ' ' + listeoAiChatConfig.strings.rateLimitSuffix + ' ' + period + '. ' + listeoAiChatConfig.strings.rateLimitWait + ' ' + waitText + ' ' + listeoAiChatConfig.strings.rateLimitBeforeTrying;
        },

        /**
         * Detect if we're on a single listing page and add "Talk about X" button
         */
        detectAndAddListingButton: function() {
            var self = this;

            // Check if we're on a single listing page
            var listingId = this.getCurrentListingId();
            if (!listingId) {
                debugLog('[Listing Context] Not on a listing page, skipping button');
                return;
            }

            debugLog('[Listing Context] Detected listing page, ID:', listingId);

            // Check if this listing is already loaded
            if (this.loadedListing && this.loadedListing.id === listingId) {
                debugLog('[Listing Context] This listing is already loaded, no button needed');
                return; // Don't show button if already loaded
            }

            // Get listing title from page
            var listingTitle = this.getCurrentListingTitle();

            // Build button HTML (only add title if found)
            var buttonHtml = '';
            if (listingTitle) {
                buttonHtml += '<h4 class="listeo-ai-listing-context-title">' + listingTitle + '</h4>';
            }

            buttonHtml += '<button class="listeo-ai-load-listing-btn" data-listing-id="' + listingId + '" disabled>' +
                '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">' +
                '<path d="M19 3H5C3.89 3 3 3.89 3 5V19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.89 20.1 3 19 3ZM19 19H5V5H19V19Z" fill="currentColor"/>' +
                '<path d="M7 7H17V9H7V7ZM7 11H17V13H7V11ZM7 15H14V17H7V15Z" fill="currentColor"/>' +
                '</svg>' +
                '<span class="btn-text">' + listeoAiChatConfig.strings.talkAboutListing + '</span>' +
                '</button>';

            // Add as a special message
            this.addMessage('listing-action', buttonHtml, 'listing-context-btn');

            // Button click handler (use event delegation since button is inside messages)
            this.$messages.on('click', '.listeo-ai-load-listing-btn', function() {
                var $btn = $(this);
                // Load listing
                self.loadListingContext(listingId, self.getCurrentListingTitle());
            });
        },

        /**
         * Get current listing ID from page
         */
        getCurrentListingId: function() {
            var bodyClasses = $('body').attr('class') || '';

            // Check if it's a single listing page
            var isSingleListing = bodyClasses.indexOf('single-listing') !== -1;

            // Make sure we're NOT on admin or other pages
            var isAdminPage = bodyClasses.indexOf('wp-admin') !== -1 ||
                             bodyClasses.indexOf('admin-bar') !== -1 && window.location.href.indexOf('/wp-admin/') !== -1;

            if (!isSingleListing || isAdminPage) {
                return null; // Not a single listing page or is admin
            }

            // Method 1: Check body class (e.g., postid-1234)
            var postIdMatch = bodyClasses.match(/postid-(\d+)/);
            if (postIdMatch) {
                var postId = parseInt(postIdMatch[1]);
                return postId;
            }

            // Method 2: Check for data attribute on listing container
            var $listingContainer = $('.single-listing-wrapper, .listing-single-container, [data-listing-id]');
            if ($listingContainer.length) {
                var dataId = $listingContainer.data('listing-id');
                if (dataId) {
                    return parseInt(dataId);
                }
            }

            // Method 3: Check global WordPress JS object (if available)
            if (typeof listeo_core !== 'undefined' && listeo_core.post_id) {
                return parseInt(listeo_core.post_id);
            }

            return null;
        },

        /**
         * Get current listing title from page
         */
        getCurrentListingTitle: function() {
            // Get from listing titlebar only
            var $titlebar = $('.listing-titlebar-title h1');
            if ($titlebar.length) {
                return $titlebar.text().trim();
            }

            // No title found - return empty string
            return '';
        },

        /**
         * Load listing context into chat
         */
        loadListingContext: function(listingId, listingTitle) {
            var self = this;
            var $btn = $('.listeo-ai-load-listing-btn');

            // Show loading state
            $btn.addClass('loading').prop('disabled', true);
            $btn.find('.btn-text').text(listeoAiChatConfig.strings.loadingButton);

            debugLog('[Listing Context] Fetching listing details for ID:', listingId);

            // Use Listeo-specific endpoint if available (config is now guaranteed to be loaded)
            var endpoint = this.chatConfig.listeo_available ? '/listeo-listing-details' : '/get-content';
            var param = this.chatConfig.listeo_available ? { listing_id: listingId } : { post_id: listingId };

            debugLog('[loadListingContext] Using endpoint:', endpoint, '(Listeo available:', this.chatConfig.listeo_available + ')');

            // Fetch listing details via API
            $.ajax({
                url: listeoAiChatConfig.apiBase + endpoint,
                method: 'POST',
                contentType: 'application/json',
                data: JSON.stringify(param),
                success: function(response) {
                    if (response.success) {
                        // IMPORTANT: Wipe any previous listing context
                        // This prevents context pollution when loading multiple listings
                        self.loadedListing = {
                            id: listingId,
                            title: response.title,
                            url: response.url,
                            content: response.structured_content
                        };

                        // Persist to localStorage
                        self.saveLoadedListingToStorage();

                        debugLog('[Listing Context] Loaded listing:', self.loadedListing.title);

                        // Remove the button (hide it)
                        $btn.closest('.listeo-ai-chat-message').fadeOut(300, function() {
                            $(this).remove();
                        });

                        // Add confirmation message
                        self.addMessage('system',
                            '📄 <strong>' + listeoAiChatConfig.strings.listingContextLoaded + '</strong> <a href="' + response.url + '">' + response.title + '</a>'
                        );
                    } else {
                        self.handleListingLoadError(listeoAiChatConfig.strings.failedLoadDetails);
                        $btn.removeClass('loading').prop('disabled', false);
                        $btn.find('.btn-text').text(listeoAiChatConfig.strings.talkAboutListing);
                    }
                },
                error: function(xhr) {
                    debugError('[Listing Context] Error loading listing:', xhr);
                    self.handleListingLoadError(listeoAiChatConfig.strings.errorLoadingListing);
                    $btn.removeClass('loading').prop('disabled', false);
                    $btn.find('.btn-text').text(listeoAiChatConfig.strings.talkAboutListing);
                }
            });
        },


        /**
         * Handle listing load error
         */
        handleListingLoadError: function(message) {
            this.addMessage('system', '⚠️ ' + message);
        },

        /**
         * Save loaded listing to localStorage
         */
        saveLoadedListingToStorage: function() {
            try {
                if (this.loadedListing) {
                    localStorage.setItem(this.loadedListingStorageKey, JSON.stringify(this.loadedListing));
                    debugLog('[Listing Context] Saved to localStorage:', this.loadedListing.title);
                }
            } catch (e) {
                debugError('[Listing Context] Failed to save to localStorage:', e);
            }
        },

        /**
         * Load persisted listing context from localStorage
         */
        loadPersistedListingContext: function() {
            try {
                var data = localStorage.getItem(this.loadedListingStorageKey);
                if (data) {
                    this.loadedListing = JSON.parse(data);
                    debugLog('[Listing Context] Restored from localStorage:', this.loadedListing.title);
                }
            } catch (e) {
                debugError('[Listing Context] Failed to load from localStorage:', e);
            }
        },

        /**
         * Clear loaded listing from localStorage
         */
        clearLoadedListingFromStorage: function() {
            try {
                localStorage.removeItem(this.loadedListingStorageKey);
                debugLog('[Listing Context] Cleared from localStorage');
            } catch (e) {
                debugError('[Listing Context] Failed to clear from localStorage:', e);
            }
        }
    };

})(jQuery);
