<?php
/**
 * Universal Settings Admin Interface
 *
 * Clean admin interface for configuring content sources
 *
 * @package Listeo_AI_Search
 * @since 1.6.0
 */

// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

class Listeo_AI_Search_Universal_Settings {

    /**
     * Default post types that cannot be removed
     *
     * @var array
     */
    private $default_types = array();

    /**
     * Get whitelisted and enabled post types (delegates to Database Manager)
     *
     * @return array Filtered array of enabled post types
     */
    public static function get_enabled_post_types() {
        return Listeo_AI_Search_Database_Manager::get_enabled_post_types();
    }

    /**
     * Constructor
     */
    public function __construct() {
        add_action('admin_init', array($this, 'register_settings'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_assets'));

        // AJAX handlers
        add_action('wp_ajax_listeo_ai_get_post_type_stats', array($this, 'ajax_get_post_type_stats'));
        add_action('wp_ajax_listeo_ai_toggle_post_type', array($this, 'ajax_toggle_post_type'));
        add_action('wp_ajax_listeo_ai_get_posts_for_selection', array($this, 'ajax_get_posts_for_selection'));
        add_action('wp_ajax_listeo_ai_generate_selected_posts', array($this, 'ajax_generate_selected_posts'));
        add_action('wp_ajax_listeo_ai_get_total_count', array($this, 'ajax_get_total_count'));
        add_action('wp_ajax_listeo_ai_add_custom_post_types', array($this, 'ajax_add_custom_post_types'));
        add_action('wp_ajax_listeo_ai_remove_custom_post_type', array($this, 'ajax_remove_custom_post_type'));
    }

    /**
     * Register settings
     */
    public function register_settings() {
        // Register settings with explicit args for multisite compatibility
        $settings_args = array(
            'type' => 'array',
            'sanitize_callback' => null, // We handle sanitization in AJAX handlers
            'default' => array(),
        );

        register_setting('listeo_ai_content_sources', 'listeo_ai_search_enabled_post_types', $settings_args);
        register_setting('listeo_ai_content_sources', 'listeo_ai_search_custom_meta_fields', $settings_args);

        // Add allowed_options filter for multisite compatibility
        add_filter('allowed_options', array($this, 'add_allowed_options'));
    }

    /**
     * Add plugin options to allowed options list for multisite compatibility
     *
     * @param array $allowed_options Array of allowed options
     * @return array Modified array of allowed options
     */
    public function add_allowed_options($allowed_options) {
        $allowed_options['listeo_ai_content_sources'] = array(
            'listeo_ai_search_enabled_post_types',
            'listeo_ai_search_custom_meta_fields',
            'listeo_ai_search_custom_post_types',
            'listeo_ai_search_manual_selections',
        );

        return $allowed_options;
    }

    /**
     * Enqueue admin assets
     */
    public function enqueue_admin_assets($hook) {
        // Only load on Listeo AI Search admin page
        if ($hook !== 'toplevel_page_ai-chat-search') {
            return;
        }

        // Only load on database tab
        $active_tab = isset($_GET['tab']) ? sanitize_text_field(wp_unslash($_GET['tab'])) : 'settings';
        if ($active_tab !== 'database') {
            return;
        }

        wp_enqueue_style(
            'listeo-ai-universal-settings',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/css/universal-settings.css',
            array(),
            LISTEO_AI_SEARCH_VERSION
        );

        wp_enqueue_script(
            'listeo-ai-universal-settings',
            LISTEO_AI_SEARCH_PLUGIN_URL . 'assets/js/universal-settings.js',
            array('jquery'),
            LISTEO_AI_SEARCH_VERSION,
            true
        );

        wp_localize_script('listeo-ai-universal-settings', 'listeoAiUniversalSettings', array(
            'ajax_url' => get_admin_url(get_current_blog_id(), 'admin-ajax.php'),
            'nonce' => wp_create_nonce('listeo_ai_universal_settings'),
            'database_nonce' => wp_create_nonce('listeo_ai_search_nonce'),
            'strings' => array(
                'confirm_toggle' => __('Enable embeddings for this content type?', 'ai-chat-search'),
                'confirm_reindex' => __('This will regenerate all embeddings for this content type. Continue?', 'ai-chat-search'),
                'reindexing' => __('Reindexing...', 'ai-chat-search'),
                'success' => __('Success!', 'ai-chat-search'),
                'error' => __('Error occurred', 'ai-chat-search')
            )
        ));
    }

    /**
     * Render just the content sources cards (no wrapping)
     */
    public function render_content_sources_cards() {
        // Get enabled post types from option (raw, no defaults applied by helper)
        $enabled_post_types = get_option('listeo_ai_search_enabled_post_types', array('listing'));

        // Get custom types that have been added
        $custom_post_types_added = get_option('listeo_ai_search_custom_post_types', array());
        if (!is_array($custom_post_types_added)) {
            $custom_post_types_added = array();
        }

        // Default types + added custom types (exclude ai_pdf_document from loop, it's rendered separately)
        $default_types = array('listing', 'post', 'page', 'product');
        $allowed_post_types = array_merge($default_types, $custom_post_types_added);

        // Store default types for checking if a type is custom (include ai_pdf_document for validation)
        $this->default_types = array_merge($default_types, array('ai_pdf_document'));

        // Get only allowed post types
        $post_types = array();
        foreach ($allowed_post_types as $post_type_name) {
            $post_type_obj = get_post_type_object($post_type_name);
            if ($post_type_obj) {
                $post_types[$post_type_name] = $post_type_obj;
            }
        }

        // Get detected custom post types that haven't been added yet
        $detected_custom_types = Listeo_AI_Search_Database_Manager::get_detected_custom_post_types();
        $available_custom_types = array();
        global $wpdb;

        foreach ($detected_custom_types as $custom_type) {
            if (!in_array($custom_type['name'], $custom_post_types_added)) {
                // Get count of published posts for this type
                $count = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = %s AND post_status = 'publish'",
                    $custom_type['name']
                ));
                $custom_type['count'] = $count;
                $available_custom_types[] = $custom_type;
            }
        }

        // Sort by count (descending) - most populated first
        usort($available_custom_types, function($a, $b) {
            return $b['count'] - $a['count'];
        });

        ?>
        <div class="listeo-ai-universal-settings">

            <!-- Post Types Grid -->
            <div class="listeo-ai-post-types-grid" style="margin-top: 0;">
                <?php foreach ($post_types as $post_type):
                    $is_enabled = in_array($post_type->name, $enabled_post_types);
                    $is_custom = !in_array($post_type->name, $this->default_types);

                    // PRO FEATURE CHECK: Check if post type is locked
                    $is_locked = AI_Chat_Search_Pro_Manager::is_post_type_locked($post_type->name);

                    // Get stats
                    $stats = $this->get_post_type_stats($post_type->name);

                    // Determine badge class
                    $badge_class = $stats['total'] > 0 ? 'has-content' : 'empty';
                ?>
                    <div class="post-type-card <?php echo $is_enabled ? 'enabled' : ''; ?> <?php echo $is_custom ? 'is-custom' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>"
                         data-post-type="<?php echo esc_attr($post_type->name); ?>">

                        <!-- Card Header -->
                        <div class="card-header">
                            <div class="post-type-icon">
                                <?php echo $this->get_post_type_icon($post_type->name); ?>
                            </div>
                            <div class="post-type-info">
                                <h3>
                                    <?php if ($is_locked): ?>
                                        <?php echo AI_Chat_Search_Pro_Manager::get_lock_icon(); ?>
                                    <?php endif; ?>
                                    <?php if ($is_custom): ?>
                                    <span class="dashicons dashicons-no-alt delete-custom-type" data-post-type="<?php echo esc_attr($post_type->name); ?>" title="<?php _e('Remove this post type', 'ai-chat-search'); ?>"></span>
                                    <?php endif; ?>
                                    <?php echo esc_html($post_type->label); ?>
                                    <?php if ($is_locked): ?>
                                        <?php echo AI_Chat_Search_Pro_Manager::get_pro_badge(); ?>
                                    <?php else: ?>
                                        <span class="custom-type-badge <?php echo $badge_class; ?>">
                                            <?php echo number_format($stats['total']); ?>
                                        </span>
                                    <?php endif; ?>
                                </h3>
                                <code><?php echo esc_html($post_type->name); ?></code>

                                <?php if ($is_locked): ?>
                                    <!-- Locked state -->
                                    <div class="manual-selection-actions">
                                        <a href="<?php echo esc_url(AI_Chat_Search_Pro_Manager::get_upgrade_url('post_type_' . $post_type->name)); ?>"
                                           class="upgrade-link" target="_blank">
                                            <?php _e('Upgrade to Pro', 'ai-chat-search'); ?> →
                                        </a>
                                    </div>
                                <?php else: ?>
                                    <!-- Manual selection link -->
                                    <div class="manual-selection-actions">
                                        <?php if ($stats['has_manual_selection']): ?>
                                            <a href="#" class="manual-selection-link active" data-post-type="<?php echo esc_attr($post_type->name); ?>">
                                                <span class="dashicons dashicons-yes"></span>
                                                <?php _e('Manual selection active', 'ai-chat-search'); ?>
                                            </a>
                                            <a href="#" class="clear-selection-link" data-post-type="<?php echo esc_attr($post_type->name); ?>">
                                                <?php _e('Clear', 'ai-chat-search'); ?>
                                            </a>
                                        <?php else: ?>
                                            <a href="#" class="manual-selection-link" data-post-type="<?php echo esc_attr($post_type->name); ?>">
                                                <span class="dashicons dashicons-admin-generic"></span>
                                                <?php _e('Manual selection', 'ai-chat-search'); ?>
                                            </a>
                                        <?php endif; ?>
                                    </div>
                                <?php endif; ?>
                            </div>

                            <!-- Toggle Switch -->
                            <?php if (!$is_locked): ?>
                                <label class="toggle-switch">
                                    <input
                                        type="checkbox"
                                        class="post-type-toggle"
                                        value="<?php echo esc_attr($post_type->name); ?>"
                                        <?php checked($is_enabled); ?>
                                    >
                                    <span class="toggle-slider"></span>
                                </label>
                            <?php else: ?>
                                <div class="toggle-switch-locked" title="<?php _e('Pro version required', 'ai-chat-search'); ?>">
                                    <span class="dashicons dashicons-lock"></span>
                                </div>
                            <?php endif; ?>
                        </div>
                    </div>
                <?php endforeach; ?>

                <?php
                // Render Documents card (always, not dependent on listing post type)
                $this->render_pdf_documents_card();
                ?>

                <?php
                // Render External Pages card (Pro feature)
                $this->render_external_pages_card();
                ?>

                <?php
                // Allow PRO plugin to add additional custom cards
                do_action('listeo_ai_after_post_types_grid');
                ?>
            </div>

            <!-- Detected Custom Post Types Section -->
            <?php if (!empty($available_custom_types)):
                // Check if custom post types feature is locked (FREE version)
                $custom_types_locked = !AI_Chat_Search_Pro_Manager::is_pro_active();
            ?>
            <div class="listeo-ai-custom-types-section <?php echo $custom_types_locked ? 'locked' : ''; ?>">
                <h3 class="collapsible-header" data-toggle="custom-types-content">
                    <span class="collapsible-header-text">
                        <?php if ($custom_types_locked): ?>
                            <?php echo AI_Chat_Search_Pro_Manager::get_lock_icon(); ?>
                        <?php endif; ?>
                        <?php _e('Detected Custom Post Types', 'ai-chat-search'); ?>
                        <?php if ($custom_types_locked): ?>
                            <?php echo AI_Chat_Search_Pro_Manager::get_pro_badge(); ?>
                        <?php else: ?>
                            <span class="custom-type-badge has-content">
                                <?php echo count($available_custom_types); ?>
                            </span>
                        <?php endif; ?>
                    </span>
                    <span class="dashicons dashicons-arrow-down-alt2"></span>
                </h3>
                <div id="custom-types-content" class="collapsible-content" style="display: none;">
                    <p><?php _e('Select custom post types to add to your training content. Once added, they will appear above as cards.', 'ai-chat-search'); ?></p>

                    <div class="custom-types-checkboxes <?php echo $custom_types_locked ? 'disabled' : ''; ?>">
                    <?php foreach ($available_custom_types as $custom_type):
                        $badge_class = $custom_type['count'] > 0 ? 'has-content' : 'empty';
                    ?>
                        <label>
                            <input type="checkbox"
                                   class="custom-post-type-checkbox"
                                   value="<?php echo esc_attr($custom_type['name']); ?>"
                                   <?php disabled($custom_types_locked); ?>>
                            <span>
                                <strong><?php echo esc_html($custom_type['label']); ?></strong>
                                <span class="custom-type-badge <?php echo $badge_class; ?>">
                                    <?php echo number_format($custom_type['count']); ?>
                                </span>
                                <br>
                                <code><?php echo esc_html($custom_type['name']); ?></code>
                            </span>
                        </label>
                    <?php endforeach; ?>
                </div>

                    <button type="button"
                            id="add-custom-post-types-btn"
                            class="airs-button airs-button-primary"
                            <?php disabled($custom_types_locked); ?>>
                        <?php _e('Add Selected Types', 'ai-chat-search'); ?>
                    </button>
                </div> <!-- Close collapsible-content -->
            </div>
            <?php endif; ?>

            <!-- Manual Selection Modal -->
            <div id="manual-selection-modal" class="listeo-ai-modal" style="display: none;">
                <div class="listeo-ai-modal-overlay"></div>
                <div class="listeo-ai-modal-content">
                    <div class="listeo-ai-modal-header">
                        <h2 id="modal-title"><?php _e('Manual Selection', 'ai-chat-search'); ?></h2>
                        <button type="button" class="listeo-ai-modal-close">
                            <span class="dashicons dashicons-no-alt"></span>
                        </button>
                    </div>
                    <div class="listeo-ai-modal-body">
                        <div class="listeo-ai-modal-controls">
                            <input type="text" id="modal-search" placeholder="<?php _e('Search...', 'ai-chat-search'); ?>" class="widefat" />
                            <div class="button-group">
                                <button type="button" id="select-all-posts" class="button button-small"><?php _e('Select All', 'ai-chat-search'); ?></button>
                                <button type="button" id="deselect-all-posts" class="button button-small"><?php _e('Deselect All', 'ai-chat-search'); ?></button>
                                <button type="button" id="select-pending-posts" class="button button-small"><?php _e('Select Pending Only', 'ai-chat-search'); ?></button>
                                <button type="button" id="select-verified-posts" class="button button-small" style="display:none;"><?php _e('Select Verified Only', 'ai-chat-search'); ?></button>
                            </div>
                        </div>
                        <div id="modal-posts-list" class="listeo-ai-posts-list">
                            <p class="loading-message"><span class="airs-spinner" style="margin-right: 6px;"></span><?php _e('Loading posts...', 'ai-chat-search'); ?></p>
                        </div>
                    </div>
                    <div class="listeo-ai-modal-footer">
                        <span id="modal-selection-count"></span>
                        <div class="modal-footer-buttons">
                            <button type="button" class="button" id="modal-cancel"><?php _e('Cancel', 'ai-chat-search'); ?></button>
                            <button type="button" class="button button-secondary" id="modal-save"><?php _e('Save Selection', 'ai-chat-search'); ?></button>
                            <button type="button" class="button button-primary" id="modal-train-now">
                                <span class="dashicons dashicons-update"></span>
                                <?php _e('Train Now', 'ai-chat-search'); ?>
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Documents card (PDF, TXT, MD, XML, CSV)
     */
    private function render_pdf_documents_card() {
        $post_type = 'ai_pdf_document';

        // Check if documents are enabled
        $enabled_types = get_option('listeo_ai_search_enabled_post_types', array('listing'));
        $is_enabled = in_array($post_type, $enabled_types);

        // Check if locked (PRO feature)
        $is_locked = AI_Chat_Search_Pro_Manager::is_post_type_locked($post_type);

        // Get document count
        $pdf_count = wp_count_posts($post_type);
        $total = isset($pdf_count->publish) ? $pdf_count->publish : 0;

        // Get indexed count
        global $wpdb;
        $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();
        $indexed = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT e.listing_id)
             FROM {$embeddings_table} e
             INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID
             WHERE p.post_type = %s AND p.post_status = 'publish'",
            $post_type
        ));

        $badge_class = $total > 0 ? 'has-content' : 'empty';
        ?>
        <div class="post-type-card pdf-card <?php echo $is_enabled ? 'enabled' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>"
             data-post-type="<?php echo esc_attr($post_type); ?>">

            <!-- Card Header -->
            <div class="card-header">
                <div class="post-type-icon">
                    <span class="dashicons dashicons-media-document"></span>
                </div>
                <div class="post-type-info">
                    <h3>
                        <?php if ($is_locked): ?>
                            <?php echo AI_Chat_Search_Pro_Manager::get_lock_icon(); ?>
                        <?php endif; ?>
                        <?php _e('Documents', 'ai-chat-search'); ?>
                        <?php if ($is_locked): ?>
                            <?php echo AI_Chat_Search_Pro_Manager::get_pro_badge(); ?>
                        <?php else: ?>
                            <span class="custom-type-badge <?php echo $badge_class; ?>">
                                <?php echo number_format($total); ?>
                            </span>
                        <?php endif; ?>
                    </h3>
                    <code>ai_document</code>

                    <?php if ($is_locked): ?>
                        <!-- Locked state -->
                        <div class="manual-selection-actions">
                            <a href="<?php echo esc_url(AI_Chat_Search_Pro_Manager::get_upgrade_url('pdf_documents')); ?>"
                               class="upgrade-link" target="_blank">
                                <?php _e('Upgrade to Pro', 'ai-chat-search'); ?> →
                            </a>
                        </div>
                    <?php else: ?>
                        <!-- Upload documents button (PRO active) -->
                        <div class="manual-selection-actions">
                            <a href="#" class="pdf-upload-link" id="upload-pdf-btn">
                                <span class="dashicons dashicons-upload"></span>
                                <?php _e('Upload documents', 'ai-chat-search'); ?>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Toggle Switch -->
                <?php if (!$is_locked): ?>
                    <label class="toggle-switch">
                        <input
                            type="checkbox"
                            class="post-type-toggle"
                            value="<?php echo esc_attr($post_type); ?>"
                            <?php checked($is_enabled); ?>
                        >
                        <span class="toggle-slider"></span>
                    </label>
                <?php else: ?>
                    <div class="toggle-switch-locked" title="<?php _e('Pro version required', 'ai-chat-search'); ?>">
                        <span class="dashicons dashicons-lock"></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Render External Pages card (Pro feature)
     */
    private function render_external_pages_card() {
        $post_type = 'ai_external_page';

        // Check if external pages are enabled
        $enabled_types = get_option('listeo_ai_search_enabled_post_types', array());
        $is_enabled = in_array($post_type, $enabled_types);

        // Check if locked (Pro feature)
        $is_locked = AI_Chat_Search_Pro_Manager::is_post_type_locked($post_type);

        // Get total count
        global $wpdb;
        $total = (int) $wpdb->get_var(
            "SELECT COUNT(*) FROM {$wpdb->posts} WHERE post_type = 'ai_external_page' AND post_status = 'publish'"
        );

        // Get indexed count
        $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();
        $indexed = (int) $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(DISTINCT e.listing_id)
             FROM {$embeddings_table} e
             INNER JOIN {$wpdb->posts} p ON e.listing_id = p.ID
             WHERE p.post_type = %s AND p.post_status = 'publish'",
            $post_type
        ));

        $badge_class = $total > 0 ? 'has-content' : 'empty';
        ?>
        <div class="post-type-card external-pages-card <?php echo $is_enabled ? 'enabled' : ''; ?> <?php echo $is_locked ? 'locked' : ''; ?>"
             data-post-type="<?php echo esc_attr($post_type); ?>">

            <!-- Card Header -->
            <div class="card-header">
                <div class="post-type-icon">
                    <span class="dashicons dashicons-admin-site-alt3"></span>
                </div>
                <div class="post-type-info">
                    <h3>
                        <?php if ($is_locked): ?>
                            <?php echo AI_Chat_Search_Pro_Manager::get_lock_icon(); ?>
                        <?php endif; ?>
                        <?php _e('External Pages', 'ai-chat-search'); ?>
                        <?php if ($is_locked): ?>
                            <?php echo AI_Chat_Search_Pro_Manager::get_pro_badge(); ?>
                        <?php else: ?>
                            <span class="custom-type-badge <?php echo $badge_class; ?>">
                                <?php echo number_format($total); ?>
                            </span>
                        <?php endif; ?>
                    </h3>
                    <code>ai_external_page</code>

                    <?php if ($is_locked): ?>
                        <!-- Locked state -->
                        <div class="manual-selection-actions">
                            <a href="<?php echo esc_url(AI_Chat_Search_Pro_Manager::get_upgrade_url('external_pages')); ?>"
                               class="upgrade-link" target="_blank">
                                <?php _e('Upgrade to Pro', 'ai-chat-search'); ?> →
                            </a>
                        </div>
                    <?php else: ?>
                        <!-- Add external pages button (PRO active) -->
                        <div class="manual-selection-actions">
                            <a href="#" class="external-pages-link" id="manage-external-pages-btn">
                                <span class="dashicons dashicons-admin-site-alt3"></span>
                                <?php _e('Add external pages', 'ai-chat-search'); ?>
                            </a>
                        </div>
                    <?php endif; ?>
                </div>

                <!-- Toggle Switch -->
                <?php if (!$is_locked): ?>
                    <label class="toggle-switch">
                        <input
                            type="checkbox"
                            class="post-type-toggle"
                            value="<?php echo esc_attr($post_type); ?>"
                            <?php checked($is_enabled); ?>
                        >
                        <span class="toggle-slider"></span>
                    </label>
                <?php else: ?>
                    <div class="toggle-switch-locked" title="<?php _e('Pro version required', 'ai-chat-search'); ?>">
                        <span class="dashicons dashicons-lock"></span>
                    </div>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /**
     * Get post type icon
     */
    private function get_post_type_icon($post_type) {
        $icons = array(
            'listing' => 'dashicons-location-alt',
            'post' => 'dashicons-admin-post',
            'page' => 'dashicons-admin-page',
            'product' => 'dashicons-products',
            'attachment' => 'dashicons-admin-media',
        );

        $icon_class = isset($icons[$post_type]) ? $icons[$post_type] : 'dashicons-admin-generic';

        return '<span class="dashicons ' . $icon_class . '"></span>';
    }

    /**
     * Get post type statistics
     * Supports three states:
     * 1. No manual selection → count all posts
     * 2. Manual selection with 0 posts → count 0
     * 3. Manual selection with specific IDs → count those
     *
     * Posts are considered "indexed" if they have:
     * - A direct embedding, OR
     * - Content chunks (which have their own embeddings)
     */
    private function get_post_type_stats($post_type) {
        global $wpdb;

        $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();
        $chunk_post_type = Listeo_AI_Content_Chunker::CHUNK_POST_TYPE;

        // Check if there's a manual selection for this post type
        $manual_selections = get_option('listeo_ai_search_manual_selections', array());
        $has_manual_selection = array_key_exists($post_type, $manual_selections);

        if ($has_manual_selection) {
            // Manual selection active
            $selected_ids = is_array($manual_selections[$post_type])
                ? array_filter(array_map('intval', $manual_selections[$post_type]))
                : array();

            if (empty($selected_ids)) {
                // User explicitly selected 0 posts
                return array(
                    'total' => 0,
                    'indexed' => 0,
                    'pending' => 0,
                    'has_manual_selection' => true
                );
            }

            // Count only selected posts
            $placeholders = implode(',', array_fill(0, count($selected_ids), '%d'));

            $total = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$wpdb->posts}
                 WHERE ID IN ($placeholders) AND post_status = 'publish'",
                ...$selected_ids
            ));

            // Count posts with direct embedding OR chunks
            $indexed = (int) $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(DISTINCT p.ID)
                 FROM {$wpdb->posts} p
                 WHERE p.ID IN ($placeholders)
                 AND p.post_status = 'publish'
                 AND (
                     EXISTS (SELECT 1 FROM {$embeddings_table} e WHERE e.listing_id = p.ID)
                     OR EXISTS (
                         SELECT 1 FROM {$wpdb->posts} chunk
                         INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                         WHERE chunk.post_type = %s
                         AND chunk_parent.meta_key = '_chunk_parent_id'
                         AND chunk_parent.meta_value = p.ID
                     )
                 )",
                ...array_merge($selected_ids, array($chunk_post_type))
            ));
        } else {
            // No manual selection - count all published posts
            // Exclude listeo-booking products (hidden booking products)
            if ($post_type === 'product') {
                $total = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT p.ID)
                     FROM {$wpdb->posts} p
                     WHERE p.post_type = %s AND p.post_status = 'publish'
                     AND NOT EXISTS (
                         SELECT 1
                         FROM {$wpdb->term_relationships} tr
                         INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                         INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                         WHERE tr.object_id = p.ID
                         AND tt.taxonomy = 'product_cat'
                         AND t.slug = 'listeo-booking'
                     )",
                    $post_type
                ));

                // Count posts with direct embedding OR chunks
                $indexed = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT p.ID)
                     FROM {$wpdb->posts} p
                     WHERE p.post_type = %s AND p.post_status = 'publish'
                     AND NOT EXISTS (
                         SELECT 1
                         FROM {$wpdb->term_relationships} tr
                         INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                         INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                         WHERE tr.object_id = p.ID
                         AND tt.taxonomy = 'product_cat'
                         AND t.slug = 'listeo-booking'
                     )
                     AND (
                         EXISTS (SELECT 1 FROM {$embeddings_table} e WHERE e.listing_id = p.ID)
                         OR EXISTS (
                             SELECT 1 FROM {$wpdb->posts} chunk
                             INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                             WHERE chunk.post_type = %s
                             AND chunk_parent.meta_key = '_chunk_parent_id'
                             AND chunk_parent.meta_value = p.ID
                         )
                     )",
                    $post_type,
                    $chunk_post_type
                ));
            } else {
                $total = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->posts}
                     WHERE post_type = %s AND post_status = 'publish'",
                    $post_type
                ));

                // Count posts with direct embedding OR chunks
                $indexed = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT p.ID)
                     FROM {$wpdb->posts} p
                     WHERE p.post_type = %s AND p.post_status = 'publish'
                     AND (
                         EXISTS (SELECT 1 FROM {$embeddings_table} e WHERE e.listing_id = p.ID)
                         OR EXISTS (
                             SELECT 1 FROM {$wpdb->posts} chunk
                             INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                             WHERE chunk.post_type = %s
                             AND chunk_parent.meta_key = '_chunk_parent_id'
                             AND chunk_parent.meta_value = p.ID
                         )
                     )",
                    $post_type,
                    $chunk_post_type
                ));
            }
        }

        return array(
            'total' => $total,
            'indexed' => $indexed,
            'pending' => max(0, $total - $indexed),
            'has_manual_selection' => $has_manual_selection
        );
    }

    /**
     * AJAX: Get post type stats
     */
    public function ajax_get_post_type_stats() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $post_type = sanitize_text_field($_POST['post_type']);
        $stats = $this->get_post_type_stats($post_type);

        wp_send_json_success($stats);
    }

    /**
     * AJAX: Toggle post type
     */
    public function ajax_toggle_post_type() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $post_type = sanitize_text_field($_POST['post_type']);

        // Handle boolean conversion properly (AJAX sends 'true'/'false' as strings or 1/0)
        $enabled_raw = $_POST['enabled'] ?? false;
        $enabled = filter_var($enabled_raw, FILTER_VALIDATE_BOOLEAN);

        // Validate post type exists
        if (!post_type_exists($post_type)) {
            wp_send_json_error('Invalid post type');
        }

        // Get custom types that have been added
        $custom_post_types_added = get_option('listeo_ai_search_custom_post_types', array());
        if (!is_array($custom_post_types_added)) {
            $custom_post_types_added = array();
        }

        // Validate post type is either default or has been added as custom
        $default_types = array('listing', 'post', 'page', 'product', 'ai_pdf_document', 'ai_external_page');
        $allowed_post_types = array_merge($default_types, $custom_post_types_added);
        if (!in_array($post_type, $allowed_post_types)) {
            wp_send_json_error('Post type not available for training');
        }

        // Get current enabled post types (raw, no filtering)
        $enabled_post_types = get_option('listeo_ai_search_enabled_post_types', array('listing'));
        if (!is_array($enabled_post_types)) {
            $enabled_post_types = array('listing');
        }

        if ($enabled) {
            // Enable: Add if not already in array
            if (!in_array($post_type, $enabled_post_types)) {
                $enabled_post_types[] = $post_type;
            }
        } else {
            // Disable: Remove from array
            $enabled_post_types = array_diff($enabled_post_types, array($post_type));

            // Allow empty array - user can disable all post types to show 0 counts
        }

        // Save the option
        $enabled_post_types = array_values($enabled_post_types); // Re-index
        update_option('listeo_ai_search_enabled_post_types', $enabled_post_types);

        wp_send_json_success(array(
            'message' => $enabled ? __('Post type enabled', 'ai-chat-search') : __('Post type disabled', 'ai-chat-search'),
            'enabled_post_types' => $enabled_post_types
        ));
    }

    /**
     * AJAX: Get posts for manual selection
     */
    public function ajax_get_posts_for_selection() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $post_type = sanitize_text_field($_POST['post_type']);

        // Validate post type exists
        if (!post_type_exists($post_type)) {
            wp_send_json_error('Invalid post type');
        }

        // Get custom types that have been added
        $custom_post_types_added = get_option('listeo_ai_search_custom_post_types', array());
        if (!is_array($custom_post_types_added)) {
            $custom_post_types_added = array();
        }

        // Validate post type is either default or has been added as custom
        $default_types = array('listing', 'post', 'page', 'product', 'ai_pdf_document', 'ai_external_page');
        $allowed_post_types = array_merge($default_types, $custom_post_types_added);
        if (!in_array($post_type, $allowed_post_types)) {
            wp_send_json_error('Post type not available for training');
        }

        global $wpdb;
        $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();
        $chunk_post_type = Listeo_AI_Content_Chunker::CHUNK_POST_TYPE;

        // Get all published posts with their embedding status
        // has_embedding = 1 if post has direct embedding OR has chunks (which have embeddings)
        // For products, exclude listeo-booking category (hidden booking products)
        if ($post_type === 'product') {
            $posts = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID, p.post_title, p.post_modified,
                       CASE
                           WHEN e.listing_id IS NOT NULL THEN 1
                           WHEN EXISTS (
                               SELECT 1 FROM {$wpdb->posts} chunk
                               INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                               WHERE chunk.post_type = %s
                               AND chunk_parent.meta_key = '_chunk_parent_id'
                               AND chunk_parent.meta_value = p.ID
                           ) THEN 1
                           ELSE 0
                       END as has_embedding
                FROM {$wpdb->posts} p
                LEFT JOIN {$embeddings_table} e ON p.ID = e.listing_id
                WHERE p.post_type = %s
                AND p.post_status = 'publish'
                AND NOT EXISTS (
                    SELECT 1
                    FROM {$wpdb->term_relationships} tr
                    INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                    INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                    WHERE tr.object_id = p.ID
                    AND tt.taxonomy = 'product_cat'
                    AND t.slug = 'listeo-booking'
                )
                ORDER BY p.post_title ASC
            ", $chunk_post_type, $post_type), ARRAY_A);
        } elseif ($post_type === 'listing') {
            // For listings, include verified status
            $posts = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID, p.post_title, p.post_modified,
                       CASE
                           WHEN e.listing_id IS NOT NULL THEN 1
                           WHEN EXISTS (
                               SELECT 1 FROM {$wpdb->posts} chunk
                               INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                               WHERE chunk.post_type = %s
                               AND chunk_parent.meta_key = '_chunk_parent_id'
                               AND chunk_parent.meta_value = p.ID
                           ) THEN 1
                           ELSE 0
                       END as has_embedding,
                       CASE WHEN pm.meta_value = 'on' THEN 1 ELSE 0 END as is_verified
                FROM {$wpdb->posts} p
                LEFT JOIN {$embeddings_table} e ON p.ID = e.listing_id
                LEFT JOIN {$wpdb->postmeta} pm ON p.ID = pm.post_id AND pm.meta_key = '_verified'
                WHERE p.post_type = %s
                AND p.post_status = 'publish'
                ORDER BY p.post_title ASC
            ", $chunk_post_type, $post_type), ARRAY_A);
        } else {
            $posts = $wpdb->get_results($wpdb->prepare("
                SELECT p.ID, p.post_title, p.post_modified,
                       CASE
                           WHEN e.listing_id IS NOT NULL THEN 1
                           WHEN EXISTS (
                               SELECT 1 FROM {$wpdb->posts} chunk
                               INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                               WHERE chunk.post_type = %s
                               AND chunk_parent.meta_key = '_chunk_parent_id'
                               AND chunk_parent.meta_value = p.ID
                           ) THEN 1
                           ELSE 0
                       END as has_embedding
                FROM {$wpdb->posts} p
                LEFT JOIN {$embeddings_table} e ON p.ID = e.listing_id
                WHERE p.post_type = %s
                AND p.post_status = 'publish'
                ORDER BY p.post_title ASC
            ", $chunk_post_type, $post_type), ARRAY_A);
        }

        if (empty($posts)) {
            wp_send_json_error(sprintf(__('No published %s found', 'ai-chat-search'), $post_type));
        }

        // Get currently selected post IDs for this post type
        $manual_selections = get_option('listeo_ai_search_manual_selections', array());
        $selected_ids = isset($manual_selections[$post_type]) ? $manual_selections[$post_type] : array();

        wp_send_json_success(array(
            'posts' => $posts,
            'post_type' => $post_type,
            'post_type_label' => get_post_type_object($post_type)->label,
            'selected_ids' => $selected_ids
        ));
    }

    /**
     * AJAX: Save manual selection for a post type
     */
    public function ajax_generate_selected_posts() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $post_type = sanitize_text_field($_POST['post_type']);
        $post_ids = isset($_POST['post_ids']) ? array_map('intval', (array) $_POST['post_ids']) : array();

        // Validate post type exists
        if (!post_type_exists($post_type)) {
            wp_send_json_error('Invalid post type');
        }

        // Get custom types that have been added
        $custom_post_types_added = get_option('listeo_ai_search_custom_post_types', array());
        if (!is_array($custom_post_types_added)) {
            $custom_post_types_added = array();
        }

        // Validate post type is either default or has been added as custom
        $default_types = array('listing', 'post', 'page', 'product', 'ai_pdf_document', 'ai_external_page');
        $allowed_post_types = array_merge($default_types, $custom_post_types_added);
        if (!in_array($post_type, $allowed_post_types)) {
            wp_send_json_error('Post type not available for training');
        }

        // Get current manual selections
        $manual_selections = get_option('listeo_ai_search_manual_selections', array());

        // Distinguish between "clear selection" (button) vs "save with 0 selected" (modal)
        // If post_ids is empty but request came from modal save, treat as 0 posts selected
        // If post_ids is empty from clear button, remove manual selection entirely

        $is_clear_request = isset($_POST['clear']) && $_POST['clear'] === 'true';

        if (empty($post_ids)) {
            if ($is_clear_request) {
                // Clear button clicked - remove manual selection (revert to all)
                unset($manual_selections[$post_type]);
            } else {
                // Modal saved with 0 posts - treat as explicit empty selection
                $manual_selections[$post_type] = array();
            }
        } else {
            // Save manual selection for this post type
            $manual_selections[$post_type] = $post_ids;
        }

        // Update with autoload false to prevent caching issues
        update_option('listeo_ai_search_manual_selections', $manual_selections, false);

        // Clear WordPress cache to ensure option is immediately available
        wp_cache_delete('alloptions', 'options');
        wp_cache_delete('listeo_ai_search_manual_selections', 'options');

        // Debug log
        if (get_option('listeo_ai_search_debug_mode', false)) {
            error_log(sprintf(
                '[AI Chat] Manual selection saved for %s: %s (is_clear: %s)',
                $post_type,
                json_encode($manual_selections[$post_type] ?? 'REMOVED'),
                $is_clear_request ? 'true' : 'false'
            ));
        }

        // Get updated stats
        $stats = $this->get_post_type_stats($post_type);

        wp_send_json_success(array(
            'message' => empty($post_ids)
                ? __('Manual selection cleared - all posts will be processed', 'ai-chat-search')
                : sprintf(__('Saved selection of %d posts', 'ai-chat-search'), count($post_ids)),
            'total' => count($post_ids),
            'stats' => $stats
        ));
    }

    /**
     * AJAX: Get total count across all enabled post types
     * Respects manual selections for accurate counts
     *
     * Posts are considered "indexed" if they have:
     * - A direct embedding, OR
     * - Content chunks (which have their own embeddings)
     */
    public function ajax_get_total_count() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        global $wpdb;

        // Get whitelisted enabled post types
        $enabled_post_types = self::get_enabled_post_types();

        // If no post types are enabled, return zero counts immediately
        if (empty($enabled_post_types)) {
            wp_send_json_success(array(
                'total' => 0,
                'indexed' => 0,
                'pending' => 0,
                'enabled_types' => array(),
                'enabled_count' => 0
            ));
            return;
        }

        // Get manual selections
        $manual_selections = get_option('listeo_ai_search_manual_selections', array());
        $embeddings_table = Listeo_AI_Search_Database_Manager::get_embeddings_table_name();
        $chunk_post_type = Listeo_AI_Content_Chunker::CHUNK_POST_TYPE;

        // Debug log
        if (get_option('listeo_ai_search_debug_mode', false)) {
            error_log('[AI Chat] ajax_get_total_count called');
            error_log('[AI Chat] Enabled post types: ' . json_encode($enabled_post_types));
            error_log('[AI Chat] Manual selections: ' . json_encode($manual_selections));
        }

        // Calculate total and indexed counts respecting manual selections
        $total = 0;
        $indexed = 0;
        $type_breakdown = array(); // Store per-type breakdown

        foreach ($enabled_post_types as $post_type) {
            // Three states for manual selection:
            // 1. Not set (not in array) → count all posts
            // 2. Set to empty array ([]) → count 0 posts (user explicitly selected 0)
            // 3. Set to array with IDs ([123, 456]) → count those posts

            if (array_key_exists($post_type, $manual_selections)) {
                // Manual selection is active for this post type
                $selected_ids = is_array($manual_selections[$post_type])
                    ? array_filter(array_map('intval', $manual_selections[$post_type]))
                    : array();

                if (empty($selected_ids)) {
                    // User explicitly selected 0 posts - count as 0
                    // Don't add anything to totals
                    continue;
                }

                // Count only selected posts
                $placeholders = implode(',', array_fill(0, count($selected_ids), '%d'));

                $type_total = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM {$wpdb->posts}
                     WHERE ID IN ($placeholders) AND post_status = 'publish'",
                    ...$selected_ids
                ));

                // Count posts with direct embedding OR chunks
                $type_indexed = (int) $wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(DISTINCT p.ID)
                     FROM {$wpdb->posts} p
                     WHERE p.ID IN ($placeholders)
                     AND p.post_status = 'publish'
                     AND (
                         EXISTS (SELECT 1 FROM {$embeddings_table} e WHERE e.listing_id = p.ID)
                         OR EXISTS (
                             SELECT 1 FROM {$wpdb->posts} chunk
                             INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                             WHERE chunk.post_type = %s
                             AND chunk_parent.meta_key = '_chunk_parent_id'
                             AND chunk_parent.meta_value = p.ID
                         )
                     )",
                    ...array_merge($selected_ids, array($chunk_post_type))
                ));

                $total += $type_total;
                $indexed += $type_indexed;

                // Store breakdown for this type
                $post_type_obj = get_post_type_object($post_type);
                if ($post_type_obj && $type_total > 0) {
                    $type_breakdown[] = array(
                        'label' => $post_type_obj->label,
                        'total' => $type_total,
                        'indexed' => $type_indexed
                    );
                }
            } else {
                // No manual selection - count all published posts
                // Exclude listeo-booking products (hidden booking products)
                if ($post_type === 'product') {
                    $type_total = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(DISTINCT p.ID)
                         FROM {$wpdb->posts} p
                         WHERE p.post_type = %s AND p.post_status = 'publish'
                         AND NOT EXISTS (
                             SELECT 1
                             FROM {$wpdb->term_relationships} tr
                             INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                             INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                             WHERE tr.object_id = p.ID
                             AND tt.taxonomy = 'product_cat'
                             AND t.slug = 'listeo-booking'
                         )",
                        $post_type
                    ));

                    // Count posts with direct embedding OR chunks
                    $type_indexed = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(DISTINCT p.ID)
                         FROM {$wpdb->posts} p
                         WHERE p.post_type = %s AND p.post_status = 'publish'
                         AND NOT EXISTS (
                             SELECT 1
                             FROM {$wpdb->term_relationships} tr
                             INNER JOIN {$wpdb->term_taxonomy} tt ON tr.term_taxonomy_id = tt.term_taxonomy_id
                             INNER JOIN {$wpdb->terms} t ON tt.term_id = t.term_id
                             WHERE tr.object_id = p.ID
                             AND tt.taxonomy = 'product_cat'
                             AND t.slug = 'listeo-booking'
                         )
                         AND (
                             EXISTS (SELECT 1 FROM {$embeddings_table} e WHERE e.listing_id = p.ID)
                             OR EXISTS (
                                 SELECT 1 FROM {$wpdb->posts} chunk
                                 INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                                 WHERE chunk.post_type = %s
                                 AND chunk_parent.meta_key = '_chunk_parent_id'
                                 AND chunk_parent.meta_value = p.ID
                             )
                         )",
                        $post_type,
                        $chunk_post_type
                    ));
                } else {
                    $type_total = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(*) FROM {$wpdb->posts}
                         WHERE post_type = %s AND post_status = 'publish'",
                        $post_type
                    ));

                    // Count posts with direct embedding OR chunks
                    $type_indexed = (int) $wpdb->get_var($wpdb->prepare(
                        "SELECT COUNT(DISTINCT p.ID)
                         FROM {$wpdb->posts} p
                         WHERE p.post_type = %s AND p.post_status = 'publish'
                         AND (
                             EXISTS (SELECT 1 FROM {$embeddings_table} e WHERE e.listing_id = p.ID)
                             OR EXISTS (
                                 SELECT 1 FROM {$wpdb->posts} chunk
                                 INNER JOIN {$wpdb->postmeta} chunk_parent ON chunk.ID = chunk_parent.post_id
                                 WHERE chunk.post_type = %s
                                 AND chunk_parent.meta_key = '_chunk_parent_id'
                                 AND chunk_parent.meta_value = p.ID
                             )
                         )",
                        $post_type,
                        $chunk_post_type
                    ));
                }

                $total += $type_total;
                $indexed += $type_indexed;

                // Store breakdown for this type
                $post_type_obj = get_post_type_object($post_type);
                if ($post_type_obj && $type_total > 0) {
                    $type_breakdown[] = array(
                        'label' => $post_type_obj->label,
                        'total' => $type_total,
                        'indexed' => $type_indexed
                    );
                }
            }
        }

        // Get enabled type labels for display
        $type_labels = array();
        foreach ($enabled_post_types as $post_type) {
            $post_type_obj = get_post_type_object($post_type);
            if ($post_type_obj) {
                $type_labels[] = $post_type_obj->label;
            }
        }

        wp_send_json_success(array(
            'total' => (int) $total,
            'indexed' => (int) $indexed,
            'pending' => max(0, (int) $total - (int) $indexed),
            'enabled_types' => $type_labels,
            'enabled_count' => count($enabled_post_types),
            'type_breakdown' => $type_breakdown
        ));
    }

    /**
     * AJAX: Add custom post types to the training-ready list
     */
    public function ajax_add_custom_post_types() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $selected_types = isset($_POST['post_types']) ? (array) $_POST['post_types'] : array();
        $selected_types = array_map('sanitize_text_field', $selected_types);

        if (empty($selected_types)) {
            wp_send_json_error(__('No post types selected', 'ai-chat-search'));
        }

        // Validate all selected types exist
        foreach ($selected_types as $post_type) {
            if (!post_type_exists($post_type)) {
                wp_send_json_error(sprintf(__('Invalid post type: %s', 'ai-chat-search'), $post_type));
            }
        }

        // Get current custom types
        $custom_post_types = get_option('listeo_ai_search_custom_post_types', array());
        if (!is_array($custom_post_types)) {
            $custom_post_types = array();
        }

        // Add selected types (merge and remove duplicates)
        $custom_post_types = array_unique(array_merge($custom_post_types, $selected_types));
        update_option('listeo_ai_search_custom_post_types', $custom_post_types);

        // Also automatically enable the newly added types
        $enabled_post_types = get_option('listeo_ai_search_enabled_post_types', array('listing'));
        if (!is_array($enabled_post_types)) {
            $enabled_post_types = array('listing');
        }
        $enabled_post_types = array_unique(array_merge($enabled_post_types, $selected_types));
        update_option('listeo_ai_search_enabled_post_types', $enabled_post_types);

        wp_send_json_success(array(
            'message' => sprintf(
                _n(
                    '%d post type added successfully',
                    '%d post types added successfully',
                    count($selected_types),
                    'ai-chat-search'
                ),
                count($selected_types)
            ),
            'added_types' => $selected_types,
            'reload_required' => true
        ));
    }

    /**
     * AJAX: Remove custom post type
     */
    public function ajax_remove_custom_post_type() {
        check_ajax_referer('listeo_ai_universal_settings', 'nonce');

        if (!current_user_can('manage_options')) {
            wp_send_json_error('Insufficient permissions');
        }

        $post_type = sanitize_text_field($_POST['post_type']);

        // Validate post type exists
        if (!post_type_exists($post_type)) {
            wp_send_json_error('Invalid post type');
        }

        // Don't allow removing default types
        $default_types = array('listing', 'post', 'page', 'product');
        if (in_array($post_type, $default_types)) {
            wp_send_json_error(__('Cannot remove default post types', 'ai-chat-search'));
        }

        // Get current custom types
        $custom_post_types = get_option('listeo_ai_search_custom_post_types', array());
        if (!is_array($custom_post_types)) {
            $custom_post_types = array();
        }

        // Remove from custom types list
        $custom_post_types = array_diff($custom_post_types, array($post_type));
        update_option('listeo_ai_search_custom_post_types', array_values($custom_post_types));

        // Also remove from enabled types
        $enabled_post_types = get_option('listeo_ai_search_enabled_post_types', array('listing'));
        if (is_array($enabled_post_types)) {
            $enabled_post_types = array_diff($enabled_post_types, array($post_type));
            update_option('listeo_ai_search_enabled_post_types', array_values($enabled_post_types));
        }

        // Clear manual selections for this type
        $manual_selections = get_option('listeo_ai_search_manual_selections', array());
        if (isset($manual_selections[$post_type])) {
            unset($manual_selections[$post_type]);
            update_option('listeo_ai_search_manual_selections', $manual_selections);
        }

        wp_send_json_success(array(
            'message' => sprintf(__('Post type "%s" removed successfully', 'ai-chat-search'), $post_type),
            'post_type' => $post_type
        ));
    }
}

// Initialize
new Listeo_AI_Search_Universal_Settings();
