/**
 * AI Chat Script
 *
 * Handles chat functionality for shortcode
 */

(function ($) {
  "use strict";

  /**
   * Debug logging helper - only logs when debug mode is enabled
   */
  const debugLog = function (...args) {
    if (
      typeof listeoAiChatConfig !== "undefined" &&
      listeoAiChatConfig.debugMode
    ) {
      console.log("[AI Chat]", ...args);
    }
  };

  const debugError = function (...args) {
    if (
      typeof listeoAiChatConfig !== "undefined" &&
      listeoAiChatConfig.debugMode
    ) {
      console.error("[AI Chat ERROR]", ...args);
    }
  };

  /**
   * Get headers for REST API requests
   * Only includes X-WP-Nonce for logged-in users to prevent stale nonce errors
   * from cached pages (CDN/Cloudflare). Logged-out users don't need nonce since
   * all chat endpoints have public permission_callback.
   *
   * @return {Object} Headers object for fetch/ajax requests
   */
  const getRequestHeaders = function () {
    var headers = {
      "Content-Type": "application/json",
      "X-Page-URL": window.location.href, // Track which page chat is used on
    };

    // Only send nonce for logged-in users (they won't have cached pages)
    // This prevents "rest_cookie_invalid_nonce" errors from CDN-cached pages
    if (listeoAiChatConfig.isLoggedIn && listeoAiChatConfig.nonce) {
      headers["X-WP-Nonce"] = listeoAiChatConfig.nonce;
    }

    return headers;
  };

  /**
   * Report error to backend for server-side logging
   * Fire-and-forget - doesn't block error handling
   * @param {string} errorType - Error type
   * @param {string} context - Where error occurred
   * @param {Object} details - Error details
   */
  const reportErrorToBackend = function (errorType, context, details) {
    // Don't report if no API base configured
    if (!listeoAiChatConfig || !listeoAiChatConfig.apiBase) {
      return;
    }

    // Fire-and-forget POST to backend error logging endpoint
    try {
      $.ajax({
        url: listeoAiChatConfig.apiBase + "/log-client-error",
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify({
          error_type: errorType,
          context: context,
          details: {
            status: details.status,
            statusText: details.statusText,
            readyState: details.readyState,
            timestamp: details.timestamp,
            responsePreview: details.responsePreview,
            request_id: details.serverError?.request_id || null,
            v: "1.6.9+",
          },
        }),
        timeout: 5000, // Short timeout - this is fire-and-forget
      });
    } catch (e) {
      // Silently ignore - this is best-effort logging
    }
  };

  /**
   * Detailed error handler - analyzes XHR errors and provides diagnostic info
   * Always logs to console AND reports to backend for server-side debugging
   * @param {Object} xhr - jQuery XHR object
   * @param {string} context - Where the error occurred (e.g., 'chat-proxy', 'tool-call')
   * @return {Object} { userMessage: string, errorType: string, details: object }
   */
  const analyzeError = function (xhr, context) {
    var errorType = "unknown";
    var userMessage = listeoAiChatConfig.strings.errorGeneral;
    var details = {
      context: context,
      status: xhr.status,
      statusText: xhr.statusText,
      readyState: xhr.readyState,
      timestamp: new Date().toISOString(),
    };

    // Analyze error type based on XHR state
    if (xhr.status === 0) {
      if (xhr.readyState === 0) {
        // Request never sent - likely blocked or network down
        errorType = "network_blocked";
        userMessage =
          listeoAiChatConfig.strings.errorNetwork ||
          "Network error - request could not be sent. Please check your connection.";
      } else if (xhr.readyState === 4) {
        // Request completed but status 0 - connection dropped mid-request
        errorType = "connection_dropped";
        userMessage =
          listeoAiChatConfig.strings.errorConnection ||
          "Connection was interrupted. Please try again.";
      } else {
        // Other readyState with status 0 - likely timeout or abort
        errorType = "connection_interrupted";
        userMessage =
          listeoAiChatConfig.strings.errorConnection ||
          "Connection was interrupted. Please try again.";
      }
    } else if (xhr.status === 408 || xhr.statusText === "timeout") {
      errorType = "timeout";
      userMessage =
        listeoAiChatConfig.strings.errorTimeout ||
        "Request timed out. Please try again.";
    } else if (xhr.status === 429) {
      errorType = "rate_limit";
      userMessage =
        listeoAiChatConfig.strings.errorRateLimit ||
        "Too many requests. Please wait a moment and try again.";
    } else if (xhr.status >= 500) {
      errorType = "server_error";
      userMessage =
        listeoAiChatConfig.strings.errorServer ||
        "Server error occurred. Please try again.";
      if (xhr.responseJSON && xhr.responseJSON.error) {
        details.serverError = xhr.responseJSON.error;
      }
    } else if (xhr.status >= 400) {
      errorType = "client_error";
      userMessage =
        listeoAiChatConfig.strings.errorGeneral ||
        "An error occurred. Please try again.";
      if (xhr.responseJSON && xhr.responseJSON.error) {
        details.serverError = xhr.responseJSON.error;
      }
    } else if (xhr.responseJSON && xhr.responseJSON.error) {
      errorType = "api_error";
      details.serverError = xhr.responseJSON.error;
    }

    // Always try to extract actual error message from response (overrides generic messages)
    // This ensures users see the real error like "Cookie check failed" or "Rate limit exceeded"
    if (xhr.responseJSON) {
      var serverMsg = null;
      // AI Chat format: {"error": {"message": "..."}}
      if (xhr.responseJSON.error && xhr.responseJSON.error.message) {
        serverMsg = xhr.responseJSON.error.message;
      }
      // WordPress format: {"message": "..."}
      else if (xhr.responseJSON.message) {
        serverMsg = xhr.responseJSON.message;
      }
      // Simple format: {"error": "string"}
      else if (typeof xhr.responseJSON.error === "string") {
        serverMsg = xhr.responseJSON.error;
      }

      if (serverMsg) {
        userMessage = serverMsg;
      }
    }

    // Try to get response text for additional diagnostics
    try {
      if (xhr.responseText) {
        details.responsePreview = xhr.responseText.substring(0, 500);
      }
    } catch (e) {
      details.responseTextError = "Could not read response text";
    }

    // Always log errors to console for debugging (even without debug mode)
    console.error("[AI Chat] Error in " + context + ":", {
      type: errorType,
      status: xhr.status,
      statusText: xhr.statusText,
      readyState: xhr.readyState,
      details: details,
    });

    // Report error to backend for server-side logging (fire-and-forget)
    reportErrorToBackend(errorType, context, details);

    return {
      userMessage: userMessage,
      errorType: errorType,
      details: details,
    };
  };

  /**
   * Configure payload for GPT-5 vs GPT-4 models
   * GPT-5 models use max_completion_tokens and don't support temperature
   * @param {Object} payload - The API payload to configure
   * @param {Object} chatConfig - The chat configuration object
   */
  const configureModelPayload = function (payload, chatConfig) {
    var model = chatConfig.model;

    if (model.startsWith("gpt-5")) {
      payload.max_completion_tokens = chatConfig.max_tokens;
      payload.verbosity = "medium";
      // GPT-5.1/5.2 with reasoning_effort: none for fast, intelligent responses
      // (5.2 is mapped to 5.1 on backend due to broken tool calling)
      if (model === "gpt-5.1" || model === "gpt-5.2") {
        payload.reasoning_effort = "none";
      }
      // GPT-5-mini with reasoning_effort: minimal for speed
      if (model === "gpt-5-mini") {
        payload.reasoning_effort = "minimal";
      }
    } else {
      payload.max_tokens = chatConfig.max_tokens;
      payload.temperature = chatConfig.temperature;
    }
  };

  /**
   * Log model debug information
   * @param {Object} payload - The API payload
   * @param {string} context - Context description for the log
   * @param {Object} chatConfig - The chat configuration object
   */
  const logModelDebug = function (payload, context, chatConfig) {
    debugLog("========== OpenAI API Call (" + context + ") ==========");
    debugLog("Model:", chatConfig.model);
    if (chatConfig.model.startsWith("gpt-5")) {
      debugLog("GPT-5 Model Detected - Using max_completion_tokens");
      debugLog("Max Completion Tokens:", payload.max_completion_tokens);
      debugLog("Verbosity:", payload.verbosity);
      if (payload.reasoning_effort) {
        debugLog("Reasoning Effort:", payload.reasoning_effort);
      }
    } else {
      debugLog("Max Tokens:", payload.max_tokens);
      debugLog("Temperature:", payload.temperature);
    }
    debugLog("=".repeat(55));
  };

  /**
   * Log API request summary with model params
   * @param {Object} payload - The API payload
   */
  const logApiRequest = function (payload) {
    var isGpt5 = payload.model && payload.model.startsWith("gpt-5");
    debugLog(
      "🚀 API REQUEST | Model:",
      payload.model,
      "| Params:",
      isGpt5
        ? {
            reasoning_effort: payload.reasoning_effort || "(default)",
            max_completion_tokens: payload.max_completion_tokens,
            verbosity: payload.verbosity,
          }
        : {
            max_tokens: payload.max_tokens,
            temperature: payload.temperature,
          },
    );
  };

  /**
   * Generate loading indicator HTML based on configured style
   * @param {string} text - The loading text to display (e.g., "Thinking...", "Searching listings...")
   * @return {string} HTML string for the loader
   */
  const generateLoaderHTML = function (text) {
    // Check loading style from config (default: 'spinner')
    var loadingStyle =
      typeof listeoAiChatConfig !== "undefined" &&
      listeoAiChatConfig.loadingStyle
        ? listeoAiChatConfig.loadingStyle
        : "spinner";

    if (loadingStyle === "dots") {
      // Dots only animation (no text)
      return (
        '<div class="listeo-ai-chat-loader-wrapper">' +
        '<div class="listeo-ai-chat-typing-dots">' +
        "<span></span>" +
        "<span></span>" +
        "<span></span>" +
        "</div>" +
        "</div>"
      );
    }

    // Default: Spinner + Text with shimmer effect
    return (
      '<span class="listeo-ai-chat-loading"></span> <span class="listeo-ai-chat-shimmer-text">' +
      text +
      "</span>"
    );
  };

  // Flag to ensure global handlers are only bound once
  var globalHandlersBound = false;

  /**
   * Bind global event handlers (contact form) - only once, not per instance
   */
  function bindGlobalHandlers() {
    if (globalHandlersBound) {
      return;
    }
    globalHandlersBound = true;

    debugLog("Binding global contact form handlers (once)");

    // Contact form close button handler
    $(document).on("click", ".listeo-ai-contact-form-close", function (e) {
      e.preventDefault();
      var $overlay = $(this).closest(".listeo-ai-contact-form-overlay");
      $overlay.fadeOut(200);
      debugLog("Contact form closed");
    });

    // Contact form submission handler
    $(document).on("submit", ".listeo-ai-contact-form-body", function (e) {
      e.preventDefault();
      var $form = $(this);
      var $overlay = $form.closest(".listeo-ai-contact-form-overlay");
      var $submit = $form.find(".listeo-ai-contact-form-submit");
      var $msgDiv = $form.find(".listeo-ai-contact-form-message");
      var $buttonText = $submit.find(".button-text");
      var $spinner = $submit.find(".button-spinner");

      // Prevent double submission
      if ($submit.prop("disabled")) {
        debugLog("Contact form already submitting, ignoring");
        return;
      }

      // Get form data
      var name = $form.find('input[name="name"]').val().trim();
      var email = $form.find('input[name="email"]').val().trim();
      var msgContent = $form.find('textarea[name="message"]').val().trim();

      // Basic validation
      if (!name || !email || !msgContent) {
        $msgDiv
          .removeClass("success")
          .addClass("error")
          .text(
            listeoAiChatConfig.strings.contactFormFillAll ||
              "Please fill in all fields.",
          )
          .show();
        return;
      }

      // Show loading state
      $submit.prop("disabled", true);
      $buttonText.hide();
      $spinner.show();
      $msgDiv.hide();

      // Submit form via REST API
      $.ajax({
        url: listeoAiChatConfig.apiBase + "/contact-form",
        method: "POST",
        headers: getRequestHeaders(),
        contentType: "application/json",
        data: JSON.stringify({
          name: name,
          email: email,
          message: msgContent,
          source: "button",
        }),
        success: function (response) {
          debugLog("Contact form submitted successfully", response);
          $msgDiv
            .removeClass("error")
            .addClass("success")
            .text(
              response.message ||
                listeoAiChatConfig.strings.contactFormSent ||
                "Message sent successfully!",
            )
            .show();

          // Reset form
          $form[0].reset();

          // Close overlay after 2 seconds
          setTimeout(function () {
            $overlay.fadeOut(200);
            $msgDiv.hide();
          }, 2000);
        },
        error: function (xhr) {
          debugLog("Contact form error", xhr);
          var errorMsg =
            listeoAiChatConfig.strings.contactFormError ||
            "Failed to send message. Please try again.";
          if (xhr.responseJSON && xhr.responseJSON.message) {
            errorMsg = xhr.responseJSON.message;
          }
          $msgDiv
            .removeClass("success")
            .addClass("error")
            .text(errorMsg)
            .show();
        },
        complete: function () {
          $submit.prop("disabled", false);
          $buttonText.show();
          $spinner.hide();
        },
      });
    });
  }

  /**
   * Initialize custom tooltips for elements with data-chat-tooltip attribute
   */
  function initTooltips() {
    var $tooltip = null;
    var hideTimeout = null;

    // Create tooltip element if it doesn't exist
    function getTooltip() {
      if (!$tooltip) {
        $tooltip = $('<div class="listeo-ai-tooltip"></div>').appendTo('body');
      }
      return $tooltip;
    }

    // Show tooltip on hover
    $(document).on('mouseenter', '[data-chat-tooltip]', function() {
      var $el = $(this);
      var text = $el.attr('data-chat-tooltip');

      // Don't show tooltip if element has recording or transcribing state
      if (!text || $el.hasClass('recording') || $el.hasClass('transcribing')) return;

      clearTimeout(hideTimeout);

      var tooltip = getTooltip();
      tooltip.text(text);

      // Get element position relative to viewport (for fixed positioning)
      var rect = $el[0].getBoundingClientRect();
      var elWidth = rect.width;

      // Show tooltip to calculate its dimensions
      tooltip.css({
        visibility: 'hidden',
        display: 'block'
      });

      var tooltipWidth = tooltip.outerWidth();
      var tooltipHeight = tooltip.outerHeight();

      // Calculate position (centered above element, using viewport coordinates)
      var left = rect.left + (elWidth / 2);
      var top = rect.top - tooltipHeight - 8; // 8px gap + arrow

      // Ensure tooltip doesn't go off-screen
      if (left - (tooltipWidth / 2) < 10) {
        left = (tooltipWidth / 2) + 10;
      }
      if (left + (tooltipWidth / 2) > $(window).width() - 10) {
        left = $(window).width() - (tooltipWidth / 2) - 10;
      }

      tooltip.css({
        left: left,
        top: top,
        visibility: 'visible',
        display: 'block'
      });

      // Trigger animation
      setTimeout(function() {
        tooltip.addClass('visible');
      }, 10);
    });

    // Hide tooltip on mouse leave
    $(document).on('mouseleave', '[data-chat-tooltip]', function() {
      if ($tooltip) {
        $tooltip.removeClass('visible');
        hideTimeout = setTimeout(function() {
          if ($tooltip) {
            $tooltip.css('display', 'none');
          }
        }, 150);
      }
    });

    // Hide tooltip immediately when element gets recording/transcribing class
    // Using MutationObserver to detect class changes
    var observer = new MutationObserver(function(mutations) {
      mutations.forEach(function(mutation) {
        if (mutation.type === 'attributes' && mutation.attributeName === 'class') {
          var $target = $(mutation.target);
          if ($target.hasClass('recording') || $target.hasClass('transcribing')) {
            if ($tooltip) {
              $tooltip.removeClass('visible').css('display', 'none');
            }
          }
        }
      });
    });

    // Observe mic buttons for class changes
    $('[data-chat-tooltip].listeo-ai-chat-mic-btn').each(function() {
      observer.observe(this, { attributes: true, attributeFilter: ['class'] });
    });

    // Also observe dynamically added mic buttons
    $(document).on('DOMNodeInserted', function(e) {
      var $el = $(e.target);
      if ($el.hasClass('listeo-ai-chat-mic-btn') && $el.attr('data-chat-tooltip')) {
        observer.observe(e.target, { attributes: true, attributeFilter: ['class'] });
      }
    });
  }

  // Initialize all chat instances
  $(document).ready(function () {
    debugLog("Initializing...");
    debugLog("jQuery loaded:", typeof $ !== "undefined");
    debugLog("Found chat wrappers:", $(".listeo-ai-chat-wrapper").length);

    // Check if config is loaded
    if (typeof listeoAiChatConfig === "undefined") {
      console.error(
        "ERROR: listeoAiChatConfig is not defined! Script may not be enqueued properly.",
      );
      return;
    }

    debugLog("Config:", listeoAiChatConfig);

    // Initialize custom tooltips
    initTooltips();

    // Bind global handlers once (contact form)
    bindGlobalHandlers();

    $(".listeo-ai-chat-wrapper").each(function () {
      debugLog("Initializing chat instance:", $(this).attr("id"));
      try {
        new ListeoAIChat($(this));
      } catch (error) {
        console.error("Failed to initialize chat:", error);
      }
    });
  });

  /**
   * Chat class
   */
  function ListeoAIChat($wrapper) {
    this.$wrapper = $wrapper;
    this.chatId = $wrapper.attr("id");
    this.$messages = $wrapper.find(".listeo-ai-chat-messages");
    this.$input = $wrapper.find(".listeo-ai-chat-input");
    this.$sendBtn = $wrapper.find(".listeo-ai-chat-send-btn");

    // Menu dropdown elements
    this.$menuTrigger = $wrapper.find(".listeo-ai-chat-menu-trigger");
    this.$menuDropdown = $wrapper.find(".listeo-ai-chat-menu-dropdown");
    this.$clearBtn = $wrapper.find(".listeo-ai-chat-clear-btn");
    this.$expandBtn = $wrapper.find(".listeo-ai-chat-expand-btn");

    this.conversationHistory = [];
    this.chatConfig = null;
    this.isProcessing = false;
    this.configLoaded = false;
    this.storageKey = "listeo_ai_chat_" + this.chatId;
    this.rateLimitStorageKey = "listeo_chat_rate_limit";

    // Generate unique session ID for conversation tracking
    this.sessionId = this.getOrCreateSessionId();

    // Read per-instance hideImages setting from data attribute (overrides global config)
    var dataHideImages = $wrapper.data("hide-images");
    this.hideImages =
      dataHideImages !== undefined
        ? parseInt(dataHideImages)
        : listeoAiChatConfig.hideImages;

    debugLog("[Chat Init] Hide Images Config:", {
      dataAttribute: dataHideImages,
      globalConfig: listeoAiChatConfig.hideImages,
      finalValue: this.hideImages,
    });

    // Rate limiting configuration (from backend or defaults)
    this.rateLimits = {
      tier1: { limit: listeoAiChatConfig.rateLimits?.tier1 || 10, window: 60 }, // 10/min
      tier2: { limit: listeoAiChatConfig.rateLimits?.tier2 || 30, window: 900 }, // 30/15min
      tier3: {
        limit: listeoAiChatConfig.rateLimits?.tier3 || 100,
        window: 86400,
      }, // 100/day
    };

    // Loaded listing context (for single listing pages)
    this.loadedListing = null;
    this.loadedListingStorageKey = "listeo_chat_loaded_listing_" + this.chatId;

    // Loaded product context (for single product pages - WooCommerce)
    this.loadedProduct = null;
    this.loadedProductStorageKey = "listeo_chat_loaded_product_" + this.chatId;

    // Track if chat has been expanded (for style 2)
    this.isExpanded = false;

    // Image input handling
    this.attachedImage = null; // { base64: string, mimeType: string, name: string }
    this.$imageBtn = $wrapper.find(".listeo-ai-chat-image-btn");
    this.$imageInput = $wrapper.find(".listeo-ai-chat-image-input");

    this.init();
  }

  ListeoAIChat.prototype = {
    /**
     * Initialize chat
     */
    init: function () {
      var self = this;

      debugLog("Chat init:", {
        chatId: this.chatId,
        hasMessages: this.$messages.length,
        hasInput: this.$input.length,
        hasSendBtn: this.$sendBtn.length,
      });

      // Disable send button until config loads
      this.$sendBtn.prop("disabled", true);

      // Load config (synchronous - uses inline data from wp_localize_script)
      this.loadConfig();

      // Load conversation from localStorage (this clears/restores messages)
      this.loadConversation();

      // Show API notification AFTER loadConversation (which clears messages)
      if (this.configLoaded && !this.chatConfig.api_configured) {
        this.addMessage("system", listeoAiChatConfig.strings.apiNotConfigured);
      }

      // Load previously loaded listing from localStorage (if any)
      this.loadPersistedListingContext();

      // Detect if we're on a single listing page and add "Talk about X" button
      this.detectAndAddListingButton();

      // Load previously loaded product from localStorage (if any)
      this.loadPersistedProductContext();

      // Detect if we're on a single product page and add "Talk about X" button
      this.detectAndAddProductButton();

      // Restore expand/collapse state for floating widget
      this.restoreExpandState();

      // Event listeners
      this.$sendBtn.on("click", function () {
        debugLog("Send button clicked");
        self.sendMessage();
      });

      this.$input.on("keydown", function (e) {
        // Check isComposing for IME support (Korean, Japanese, Chinese)
        if (e.key === "Enter" && !e.shiftKey && !e.isComposing) {
          e.preventDefault();
          self.sendMessage();
        }
      });

      // Menu dropdown toggle
      this.$menuTrigger.on("click", function (e) {
        e.stopPropagation();
        self.toggleMenu();
      });

      // Close menu when clicking outside
      $(document).on("click", function (e) {
        if (!$(e.target).closest(".listeo-ai-chat-menu").length) {
          self.closeMenu();
        }
      });

      // Close menu on Escape key
      $(document).on("keydown", function (e) {
        if (e.key === "Escape") {
          self.closeMenu();
        }
      });

      // Clear conversation menu item
      this.$clearBtn.on("click", function () {
        debugLog("Clear button clicked");
        self.closeMenu();
        self.clearConversation();
      });

      // Expand/collapse chat menu item (floating widget only)
      this.$expandBtn.on("click", function () {
        debugLog("Expand/Collapse button clicked");
        self.closeMenu();
        self.toggleExpandChat();
      });

      // Image input button handling
      if (this.$imageBtn.length && this.$imageInput.length) {
        this.$imageBtn.on("click", function () {
          self.$imageInput.trigger("click");
        });

        this.$imageInput.on("change", function (e) {
          var file = e.target.files[0];
          if (file) {
            self.handleImageSelect(file);
          }
        });
      }

      // Show more button (event delegation for dynamically added buttons)
      this.$messages.on("click", ".listeo-ai-show-more-btn", function () {
        var $btn = $(this);
        var $list = $btn.prev(".listeo-ai-results-list");

        // Show all hidden listings
        $list
          .find(".listeo-ai-listing-hidden")
          .removeClass("listeo-ai-listing-hidden");

        // Hide the button
        $btn.remove();
      });

      // Popular search tag click handler (event delegation)
      $(document).on("click", ".popular-search-tag", function () {
        var $tag = $(this);
        var query = $tag.data("query");
        var $popularSearches = $tag.closest(".listeo-ai-popular-searches");
        var targetChatId = $popularSearches.data("chat-id");

        debugLog("Popular search tag clicked:", {
          query: query,
          targetChatId: targetChatId,
          currentChatId: self.chatId,
        });

        // Only handle if this tag is for the current chat instance
        if (targetChatId === self.chatId) {
          debugLog("Inserting popular search into input:", query);

          // Insert query into input field
          self.$input.val(query);

          // Focus on input
          self.$input.focus();

          // Optionally auto-send the message
          self.sendMessage();
        }
      });

      // Quick action button click handler (event delegation)
      $(document).on("click", ".listeo-ai-quick-btn", function (e) {
        e.preventDefault();
        var $btn = $(this);
        var type = $btn.data("type");
        var value = $btn.data("value");
        var buttonText = $btn.text().trim();
        var $chatWrapper = $btn.closest(".listeo-ai-chat-wrapper");
        var targetChatId = $chatWrapper.attr("id");

        debugLog("Quick button clicked:", {
          type: type,
          value: value,
          buttonText: buttonText,
          targetChatId: targetChatId,
          currentChatId: self.chatId,
        });

        // Only handle if this button is for the current chat instance
        if (targetChatId === self.chatId) {
          if (type === "url" && value) {
            // Open external URL in same tab
            window.location.href = value;
            debugLog("Opening URL:", value);
          } else if (type === "contact") {
            // Show contact form overlay
            var $overlay = $chatWrapper.find(".listeo-ai-contact-form-overlay");
            $overlay.fadeIn(200);
            $overlay.find('input[name="name"]').focus();
            debugLog("Showing contact form");
          } else if (type === "chat") {
            // Use value if set, otherwise use button text as the message
            var message = value || buttonText;
            if (message) {
              self.$input.val(message);
              self.$input.focus();
              self.sendMessage();
              debugLog("Sending quick message:", message);
            }
          }
        }
      });

      // Note: Contact form handlers are bound globally in bindGlobalHandlers()
      // to prevent duplicate bindings when multiple chat instances exist
    },

    /**
     * Load chat configuration from inline data (no API call needed)
     * Config is now passed via wp_localize_script to eliminate HTTP request
     */
    loadConfig: function () {
      var self = this;

      debugLog("[CONFIG] Loading chat config from inline data (no API call)");

      // Use inline config from wp_localize_script
      // This eliminates the /chat-config API call and avoids duplicate requests
      if (listeoAiChatConfig.chatConfig) {
        self.chatConfig = listeoAiChatConfig.chatConfig;
        self.configLoaded = true;

        debugLog("[CONFIG] ✅ Loaded from inline data:", {
          enabled: self.chatConfig.enabled,
          model: self.chatConfig.model,
          listeo_available: self.chatConfig.listeo_available,
          tools_count: self.chatConfig.tools ? self.chatConfig.tools.length : 0,
          api_configured: self.chatConfig.api_configured,
        });

        // API notification is now shown in init() after loadConversation()
        // to prevent it being cleared by $messages.empty()

        // Enable send button now that config is ready
        self.$sendBtn.prop("disabled", false);

        // Enable listing context button if present
        $(".listeo-ai-load-listing-btn").prop("disabled", false);

        // Enable product context button if present
        $(".listeo-ai-load-product-btn").prop("disabled", false);
      } else {
        // Fallback: chatConfig not available in inline data
        debugError("[CONFIG] chatConfig not found in listeoAiChatConfig");
        self.addMessage(
          "system",
          "⚠️ Chat configuration not available. Please refresh the page.",
        );
        self.configLoaded = false;
      }
    },

    /**
     * Check if chat is in Style 2 (elementor-chat-style wrapper)
     */
    isStyle2: function () {
      return this.$wrapper.parent().hasClass("elementor-chat-style");
    },

    /**
     * Expand chat to 80vh (Style 2 only)
     * Called on first message or when loading past conversation
     */
    expandChat: function () {
      if (!this.isStyle2() || this.isExpanded) {
        return; // Already expanded or not Style 2
      }

      debugLog("Expanding chat to 80vh (Style 2 animation)");

      this.$wrapper.addClass("expanded");
      this.isExpanded = true;

      // Scroll to bottom after expansion animation completes
      var self = this;
      setTimeout(function () {
        self.$messages.scrollTop(self.$messages[0].scrollHeight);
      }, 600); // Match CSS transition duration
    },

    /**
     * Send message - Dual Mode Architecture
     *
     * Mode 1 (Listeo Available): Function Calling with Listeo Tools
     *   Flow: User Question → OpenAI with tools → Tool calls → Execute → AI Answer
     *
     * Mode 2 (No Listeo): RAG-First Architecture
     *   Flow: User Question → Universal Search (top 3) → Get Content → AI Answer
     */
    sendMessage: function () {
      var self = this;
      var message = this.$input.val().trim();
      var hasImage = this.attachedImage !== null;

      // Allow sending if there's text OR an image
      if ((!message && !hasImage) || this.isProcessing) {
        return;
      }

      // Check if config is still loading
      if (!this.configLoaded) {
        this.addMessage("system", listeoAiChatConfig.strings.loadingConfig);
        debugLog("[SEND] Config not loaded yet, waiting...");
        return;
      }

      // Check if config loaded
      if (!this.chatConfig) {
        this.addMessage("system", listeoAiChatConfig.strings.errorConfig);
        return;
      }

      // Check if enabled
      if (!this.chatConfig.enabled) {
        this.addMessage("system", listeoAiChatConfig.strings.chatDisabled);
        return;
      }

      // Check rate limits (client-side)
      var rateLimitCheck = this.checkRateLimit();
      if (!rateLimitCheck.allowed) {
        this.addMessage("system", rateLimitCheck.message);
        debugLog("[Rate Limit] Blocked:", rateLimitCheck);
        return;
      }

      // Record message timestamp for rate limiting
      this.recordMessage();

      // Check if this is the first real message (expand chat for Style 2)
      var isFirstMessage = this.conversationHistory.length === 0;

      // Build message content (with image if attached)
      var messageContent = this.buildMessageContent(message);
      var displayContent = hasImage ? this.getUserMessageDisplay(message) : message;

      // Store image reference before clearing
      var attachedImageData = this.attachedImage;

      // Add user message (with image preview if attached)
      this.addMessage("user", displayContent);
      this.$input.val("");
      this.isProcessing = true;
      this.$sendBtn.prop("disabled", true);

      // Clear attached image after sending
      if (hasImage) {
        this.clearAttachedImage();
      }

      // Expand chat on first message (Style 2 only)
      if (isFirstMessage) {
        this.expandChat();

        // Hide quick buttons after first message if configured
        if (listeoAiChatConfig.quickButtonsVisibility === "hide_after_first") {
          var $quickBtns = self.$wrapper.find(".listeo-ai-chat-quick-buttons");
          $quickBtns.find(".listeo-ai-quick-btn").fadeOut(150);
          $quickBtns.delay(150).slideUp(200);
          debugLog("[Quick Buttons] Hidden after first message");
        }
      }

      // Add loading indicator
      var loadingId = "loading-" + Date.now();
      this.addMessage(
        "assistant",
        generateLoaderHTML(listeoAiChatConfig.strings.searchingDatabase),
        loadingId,
      );

      // ALWAYS use function calling mode - LLM decides whether to search or answer from context
      debugLog("===== FUNCTION CALLING MODE =====");
      debugLog(
        "Tools available:",
        this.chatConfig.tools ? this.chatConfig.tools.length : 0,
      );
      debugLog(
        "LLM will decide whether to call tools or answer directly from conversation context",
      );
      debugLog("Has image:", !!attachedImageData);
      this.sendMessageWithFunctionCalling(messageContent, loadingId);
    },

    /**
     * Send message with function calling (Listeo tools mode)
     * OpenAI decides which tool to call, we execute it, and send results back
     */
    sendMessageWithFunctionCalling: function (userMessage, loadingId) {
      var self = this;

      // Check if listing/product tools are available (need more history for complex flows)
      var hasComplexTools = self.chatConfig.tools && self.chatConfig.tools.some(function(t) {
        return t.function && (t.function.name === "search_listings" || t.function.name === "search_products");
      });

      // Get valid history slice (10 for complex tools, 6 for universal-only)
      var recentHistory = self.getValidHistorySlice(hasComplexTools ? 10 : 6);

      // Build messages array (server handles system prompt injection for security)
      var messages = recentHistory.concat([
        { role: "user", content: userMessage },
      ]);

      // Build payload with tools (only if tools are available)
      var payload = {
        model: self.chatConfig.model,
        messages: messages,
      };

      // Check if message was from speech-to-text transcription
      if (self.$wrapper.data('speech-pending')) {
        payload.is_transcribed = true;
        self.$wrapper.data('speech-pending', false); // Clear flag after use
        debugLog('[Speech-to-Text] Message was transcribed from voice');
      }

      // If listing context is loaded (via "Talk about this listing" button),
      // send listing ID so server can inject context into system prompt
      if (self.loadedListing && self.loadedListing.id) {
        payload.listing_context_id = self.loadedListing.id;
        debugLog(
          "[Listing Context] Sending listing_context_id:",
          self.loadedListing.id,
        );
      }

      // If product context is loaded (via "Talk about this product" button),
      // send product ID so server can inject context into system prompt
      if (self.loadedProduct && self.loadedProduct.id) {
        payload.product_context_id = self.loadedProduct.id;
        debugLog(
          "[Product Context] Sending product_context_id:",
          self.loadedProduct.id,
        );
      }

      // Tools are now generated server-side for security
      // We only send tool_choice to tell the server to use auto mode
      if (self.chatConfig.tools && self.chatConfig.tools.length > 0) {
        payload.tool_choice = "auto";
        // Disable parallel tool calls for OpenAI (prevents chaining issues with GPT-5.2)
        // Must only be sent when tools are present
        var model = self.chatConfig.model;
        var isOSeries = model.startsWith("o1") || model.startsWith("o3") || model.startsWith("o4");
        var isOpenAI = model.startsWith("gpt-") || model.startsWith("chatgpt-");
        if (isOpenAI && !isOSeries) {
          payload.parallel_tool_calls = false;
        }
      }

      // Configure model-specific parameters (GPT-5 vs GPT-4)
      configureModelPayload(payload, self.chatConfig);
      logModelDebug(payload, "Function Calling", self.chatConfig);

      debugLog("===== SENDING TO CHAT-PROXY =====");
      debugLog("Messages:", messages.length);
      debugLog("Tools (server-side):", self.chatConfig.tools ? self.chatConfig.tools.length : 0);

      logApiRequest(payload);

      // Send to OpenAI - with silent retry on failure
      (function doChatProxyRequest(attempt) {
        $.ajax({
          url: listeoAiChatConfig.apiBase + "/chat-proxy",
          method: "POST",
          headers: $.extend({}, getRequestHeaders(), {
            "X-Session-ID": self.sessionId,
          }),
          data: JSON.stringify(payload),
          success: function (data) {
            // Check if response indicates an error (success: false)
            if (data.success === false) {
              if (attempt < 2) {
                debugLog(
                  "Chat proxy returned error, retrying... (attempt " +
                    attempt +
                    ")",
                );
                doChatProxyRequest(attempt + 1);
                return;
              }
              // 2nd failure - show detailed error
              var errorDetail =
                data.error?.message || data.error?.type || "Unknown error";
              debugError("Chat proxy error (final):", data.error);
              self.$messages.find("#" + loadingId).remove();
              self.addMessage(
                "system",
                listeoAiChatConfig.strings.errorGeneral +
                  " [" +
                  errorDetail +
                  "]",
              );
              self.isProcessing = false;
              self.$sendBtn.prop("disabled", false);
              return;
            }

            var assistantMessage = data.choices[0].message;

            debugLog("===== OPENAI RESPONSE =====");
            debugLog("Has tool_calls:", !!assistantMessage.tool_calls);

            // Check if AI wants to call tools
            if (
              assistantMessage.tool_calls &&
              assistantMessage.tool_calls.length > 0
            ) {
              debugLog(
                "Tool calls requested:",
                assistantMessage.tool_calls.length,
              );

              // Update loading message
              self.$messages
                .find("#" + loadingId + " .listeo-ai-chat-message-content")
                .html(
                  generateLoaderHTML(
                    listeoAiChatConfig.strings.searchingDatabase,
                  ),
                );

              // Execute tool calls
              self.executeToolCalls(userMessage, assistantMessage, loadingId);
            } else {
              // No tools called - AI responded directly (greetings, clarifications, simple questions)
              debugLog("No tool calls, displaying direct response");
              debugLog(
                "This handles: greetings, thanks, clarifications, simple questions",
              );

              // Remove loading
              self.$messages.find("#" + loadingId).remove();

              // Display AI's direct response
              var content = assistantMessage.content || "No response received.";
              self.addMessage("assistant", content);

              // Update conversation history
              self.conversationHistory.push(
                { role: "user", content: userMessage },
                { role: "assistant", content: content },
              );

              // Save conversation
              self.saveConversation();

              self.isProcessing = false;
              self.$sendBtn.prop("disabled", false);
            }
          },
          error: function (xhr) {
            if (attempt < 2) {
              debugLog(
                "Chat proxy XHR error, retrying... (attempt " + attempt + ")",
              );
              doChatProxyRequest(attempt + 1);
            } else {
              var errorInfo = analyzeError(xhr, "chat-proxy");
              self.$messages.find("#" + loadingId).remove();
              self.addMessage(
                "system",
                errorInfo.userMessage,
              );
              self.isProcessing = false;
              self.$sendBtn.prop("disabled", false);
            }
          },
        });
      })(1);
    },

    /**
     * Execute tool calls (search_listings, get_listing_details, or search_universal_content)
     */
    executeToolCalls: function (userMessage, assistantMessage, loadingId) {
      var self = this;
      var toolCall = assistantMessage.tool_calls[0]; // Handle first tool call
      var functionName = toolCall.function.name;
      var functionArgs = JSON.parse(toolCall.function.arguments);

      debugLog("===== EXECUTING TOOL CALL =====");
      debugLog("Function:", functionName);
      debugLog("Arguments:", functionArgs);

      if (functionName === "search_listings") {
        // Call Listeo hybrid search
        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(
            generateLoaderHTML(listeoAiChatConfig.strings.searchingListings),
          );

        $.ajax({
          url: listeoAiChatConfig.apiBase + "/listeo-hybrid-search",
          method: "POST",
          contentType: "application/json",
          data: JSON.stringify(functionArgs),
          success: function (response) {
            if (
              response.success &&
              response.results &&
              response.results.length > 0
            ) {
              debugLog("Search results:", response.results.length);

              // Show results grid
              var gridHTML = self.formatListingsGrid(response.results);
              self.$messages.find("#" + loadingId).remove();
              self.addMessage("assistant", gridHTML);

              // Update loading for AI response
              var responseLoadingId = "loading-response-" + Date.now();
              self.addMessage(
                "assistant",
                generateLoaderHTML(listeoAiChatConfig.strings.analyzingResults),
                responseLoadingId,
              );

              // Send results back to OpenAI for natural language response
              self.getFinalResponse(
                userMessage,
                assistantMessage,
                toolCall,
                response,
                responseLoadingId,
              );
            } else {
              // No results - check if there's a notice (e.g., no embeddings trained)
              debugLog("No results found");

              // Check for notice from backend (e.g., no embeddings)
              if (response.notice && response.notice_type === "no_embeddings") {
                self.$messages.find("#" + loadingId).remove();
                self.addMessage(
                  "system",
                  response.notice,
                );
                self.isProcessing = false;
                self.$sendBtn.prop("disabled", false);
                return;
              }

              // Update loading indicator to show AI is thinking about no results
              self.$messages
                .find("#" + loadingId + " .listeo-ai-chat-message-content")
                .html(
                  generateLoaderHTML(
                    listeoAiChatConfig.strings.thinkingAboutQuery ||
                      "Thinking...",
                  ),
                );

              var noResultsMsg = {
                total: 0,
                results: [],
              };

              // Still send to OpenAI so it can respond naturally
              self.getFinalResponse(
                userMessage,
                assistantMessage,
                toolCall,
                noResultsMsg,
                loadingId,
              );
            }
          },
          error: function (xhr) {
            var errorInfo = analyzeError(xhr, "search_listings");
            self.$messages.find("#" + loadingId).remove();
            self.addMessage(
              "system",
              listeoAiChatConfig.strings.searchFailed,
            );
            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
          },
        });
      } else if (functionName === "get_listing_details") {
        // Get listing details (supports multiple IDs for comparison)
        // Normalize to array for both legacy (listing_id) and new (listing_ids) formats
        var listingIds = functionArgs.listing_ids || [functionArgs.listing_id];
        if (!Array.isArray(listingIds)) {
          listingIds = [listingIds];
        }

        var loadingText = listingIds.length > 1
          ? (listeoAiChatConfig.strings.comparingListings || "Comparing listings...")
          : listeoAiChatConfig.strings.analyzingListing;

        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(generateLoaderHTML(loadingText));

        self.getListingDetails(
          listingIds,
          userMessage,
          assistantMessage,
          toolCall,
          loadingId,
        );
      } else if (functionName === "search_universal_content") {
        // Search universal content (posts/pages/products/docs)
        debugLog("Calling universal content search for:", functionArgs.query);

        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(
            generateLoaderHTML(listeoAiChatConfig.strings.searchingSiteContent),
          );

        // Call RAG endpoint directly - it handles everything server-side
        // IMPORTANT: Don't pass chat_history here because:
        // 1. RAG endpoint doesn't use function calling
        // 2. Chat history may contain tool messages that will break the request
        // 3. This is being called AS a tool, so history isn't needed
        // Uses silent retry - retries once on failure, shows error only on 2nd failure
        // Check if user message contains an image (for chat history logging)
        var messageHasImage = Array.isArray(userMessage) && userMessage.some(function(part) {
          return part.type === 'image_url';
        });

        (function doRagRequest(attempt) {
          $.ajax({
            url: listeoAiChatConfig.apiBase + "/rag-chat",
            method: "POST",
            headers: $.extend({}, getRequestHeaders(), {
              "X-Session-ID": self.sessionId,
            }),
            data: JSON.stringify({
              query: functionArgs.query,
              original_question: self.extractTextFromMessage(userMessage), // Text only, no images
              chat_history: [], // Empty history - this is a tool call, not a conversation
              top_results: functionArgs.top_results || 5, // Server uses DB setting, this is just fallback
              has_image: messageHasImage, // For chat history logging
            }),
            success: function (response) {
              if (response.success) {
                debugLog(
                  "RAG response received:",
                  response.sources?.length || 0,
                  "sources",
                );

                // Remove loading
                self.$messages.find("#" + loadingId).remove();

                // Handle empty answer from RAG endpoint
                var answer = response.answer;
                if (!answer || answer.trim() === "") {
                  debugError("RAG returned empty answer - check server logs");
                  answer =
                    listeoAiChatConfig.strings.noResultsGeneric ||
                    "I couldn't find relevant information. Try different keywords or be more specific about what you're looking for.";
                }

                // Display answer
                self.addMessage("assistant", answer);

                // Display source attribution (if sources exist)
                if (response.sources && response.sources.length > 0) {
                  var sourcesHTML = self.formatSourceAttribution(
                    response.sources,
                  );
                  self.addMessage("assistant", sourcesHTML);
                }

                // Update conversation history - MUST include complete tool calling sequence
                self.conversationHistory.push(
                  { role: "user", content: userMessage },
                  assistantMessage, // Assistant message WITH tool_calls
                  {
                    role: "tool",
                    tool_call_id: toolCall.id,
                    content: JSON.stringify({
                      answer: answer,
                      sources: response.sources || [],
                    }),
                  },
                  { role: "assistant", content: answer }, // Final response
                );

                // Save conversation
                self.saveConversation();

                self.isProcessing = false;
                self.$sendBtn.prop("disabled", false);
              } else if (attempt < 2) {
                // Silent retry on first failure
                debugLog(
                  "RAG request failed, retrying... (attempt " + attempt + ")",
                );
                doRagRequest(attempt + 1);
              } else {
                // 2nd failure - show detailed error
                var errorDetail =
                  response.error?.message ||
                  response.error?.type ||
                  response.error?.code ||
                  "Unknown error";
                debugError("RAG endpoint error (final):", response.error);
                self.$messages.find("#" + loadingId).remove();
                self.addMessage(
                  "system",
                  listeoAiChatConfig.strings.errorGeneral +
                    " [" +
                    errorDetail +
                    "]",
                );
                self.isProcessing = false;
                self.$sendBtn.prop("disabled", false);
              }
            },
            error: function (xhr) {
              if (attempt < 2) {
                // Silent retry on first XHR failure
                debugLog(
                  "RAG request XHR error, retrying... (attempt " +
                    attempt +
                    ")",
                );
                doRagRequest(attempt + 1);
              } else {
                // 2nd failure - show detailed error
                var errorInfo = analyzeError(xhr, "rag-universal-search");
                self.$messages.find("#" + loadingId).remove();
                self.addMessage(
                  "system",
                  errorInfo.userMessage,
                );
                self.isProcessing = false;
                self.$sendBtn.prop("disabled", false);
              }
            },
          });
        })(1);
      } else if (functionName === "search_products") {
        // Search WooCommerce products
        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(
            generateLoaderHTML(listeoAiChatConfig.strings.searchingProducts),
          );

        $.ajax({
          url: listeoAiChatConfig.apiBase + "/woocommerce-product-search",
          method: "POST",
          contentType: "application/json",
          data: JSON.stringify(functionArgs),
          success: function (response) {
            if (
              response.success &&
              response.results &&
              response.results.length > 0
            ) {
              debugLog("Product search results:", response.results.length);

              // Show products grid (same container as listings, but with price instead of location)
              var gridHTML = self.formatProductsGrid(response.results);
              self.$messages.find("#" + loadingId).remove();
              self.addMessage("assistant", gridHTML);

              // Update loading for AI response
              var responseLoadingId = "loading-response-" + Date.now();
              self.addMessage(
                "assistant",
                generateLoaderHTML(
                  listeoAiChatConfig.strings.analyzingProducts,
                ),
                responseLoadingId,
              );

              // Send results back to OpenAI for natural language response
              self.getFinalResponse(
                userMessage,
                assistantMessage,
                toolCall,
                response,
                responseLoadingId,
              );
            } else {
              // No results - still need to show thinking indicator
              debugLog("No products found");

              // Update loading indicator to show AI is thinking about no results
              self.$messages
                .find("#" + loadingId + " .listeo-ai-chat-message-content")
                .html(
                  generateLoaderHTML(
                    listeoAiChatConfig.strings.thinkingAboutQuery ||
                      "Thinking...",
                  ),
                );

              var noResultsMsg = {
                total: 0,
                results: [],
              };

              // Still send to OpenAI so it can respond naturally
              self.getFinalResponse(
                userMessage,
                assistantMessage,
                toolCall,
                noResultsMsg,
                loadingId,
              );
            }
          },
          error: function (xhr) {
            var errorInfo = analyzeError(xhr, "search_products");
            self.$messages.find("#" + loadingId).remove();
            self.addMessage(
              "system",
              listeoAiChatConfig.strings.productSearchFailed,
            );
            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
          },
        });
      } else if (functionName === "get_product_details") {
        // Get WooCommerce product details (supports multiple IDs for comparison)
        // Normalize to array for both legacy (product_id) and new (product_ids) formats
        var productIds = functionArgs.product_ids || [functionArgs.product_id];
        if (!Array.isArray(productIds)) {
          productIds = [productIds];
        }

        var loadingText = productIds.length > 1
          ? (listeoAiChatConfig.strings.comparingProducts || "Comparing products...")
          : listeoAiChatConfig.strings.gettingProductDetails;

        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(generateLoaderHTML(loadingText));

        self.getProductDetails(
          productIds,
          userMessage,
          assistantMessage,
          toolCall,
          loadingId,
        );
      } else if (functionName === "check_order_status") {
        // Check WooCommerce order status
        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(
            generateLoaderHTML(listeoAiChatConfig.strings.checkingOrderStatus),
          );

        self.getOrderStatus(
          functionArgs.order_number,
          functionArgs.billing_email,
          userMessage,
          assistantMessage,
          toolCall,
          loadingId,
        );
      } else if (functionName === "send_contact_message") {
        // Send contact message via AI
        debugLog("Sending contact message via AI tool");
        self.$messages
          .find("#" + loadingId + " .listeo-ai-chat-message-content")
          .html(
            generateLoaderHTML(
              listeoAiChatConfig.strings.sendingMessage || "Sending message...",
            ),
          );

        $.ajax({
          url: listeoAiChatConfig.apiBase + "/contact-form",
          method: "POST",
          headers: getRequestHeaders(),
          contentType: "application/json",
          processData: false,
          data: JSON.stringify({
            name: functionArgs.name || "",
            email: functionArgs.email || "",
            message: functionArgs.message || "",
            source: "llm",
            conversation_id: self.sessionId,
          }),
          success: function (response) {
            debugLog("Contact form response:", response);

            var toolResult = {
              success: response.success || false,
              message:
                response.message ||
                (response.success
                  ? "Message sent successfully!"
                  : "Failed to send message."),
            };

            // Remove loading
            self.$messages.find("#" + loadingId).remove();

            // Add loading for AI response
            var responseLoadingId = "loading-response-" + Date.now();
            self.addMessage(
              "assistant",
              generateLoaderHTML(listeoAiChatConfig.strings.loading),
              responseLoadingId,
            );

            // Send result back to OpenAI for natural language response
            self.getFinalResponse(
              userMessage,
              assistantMessage,
              toolCall,
              toolResult,
              responseLoadingId,
            );
          },
          error: function (xhr) {
            debugError("Contact form error:", xhr);

            var errorMessage = "Failed to send message.";
            if (xhr.responseJSON && xhr.responseJSON.message) {
              errorMessage = xhr.responseJSON.message;
            }

            var toolResult = {
              success: false,
              message: errorMessage,
            };

            // Remove loading
            self.$messages.find("#" + loadingId).remove();

            // Add loading for AI response
            var responseLoadingId = "loading-response-" + Date.now();
            self.addMessage(
              "assistant",
              generateLoaderHTML(listeoAiChatConfig.strings.loading),
              responseLoadingId,
            );

            // Send error result back to OpenAI
            self.getFinalResponse(
              userMessage,
              assistantMessage,
              toolCall,
              toolResult,
              responseLoadingId,
            );
          },
        });
      } else {
        debugError("Unknown function:", functionName);
        self.$messages.find("#" + loadingId).remove();
        self.addMessage("system", listeoAiChatConfig.strings.unknownFunction);
        self.isProcessing = false;
        self.$sendBtn.prop("disabled", false);
      }
    },

    /**
     * Build enriched context from retrieved content
     */
    buildEnrichedContext: function (contentResponses, sourceResults) {
      var context = "RELEVANT CONTENT FROM SITE:\n\n";

      contentResponses.forEach(function (response, index) {
        // Handle both direct response and array response from $.when
        var data = response[0] || response;

        if (data.success && data.structured_content) {
          var source = sourceResults[index];
          context += "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n";
          context += "SOURCE " + (index + 1) + ": " + data.title + "\n";
          context +=
            "Type: " +
            data.post_type.charAt(0).toUpperCase() +
            data.post_type.slice(1) +
            "\n";
          context += "URL: " + data.url + "\n";
          context += "━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n\n";
          context += data.structured_content + "\n\n\n";
        }
      });

      return context;
    },

    /**
     * Answer with RAG (server-side search + context + OpenAI in ONE call)
     * Simple non-streaming AJAX approach
     */
    answerWithContext: function (
      userMessage,
      enrichedContext,
      sourceResults,
      loadingId,
    ) {
      var self = this;

      // Build conversation history (last 3 exchanges = 6 messages)
      var recentHistory = [];
      var historySlice = self.conversationHistory.slice(-6);
      historySlice.forEach(function (msg) {
        recentHistory.push({ role: msg.role, content: msg.content });
      });

      // Check if user message contains an image (for chat history logging)
      var messageHasImage = Array.isArray(userMessage) && userMessage.some(function(part) {
        return part.type === 'image_url';
      });

      var payload = {
        query: userMessage,
        chat_history: recentHistory,
        top_results: 5, // Server uses DB setting, this is just fallback
        has_image: messageHasImage, // For chat history logging
      };

      debugLog("===== CALLING RAG ENDPOINT (SERVER-SIDE RAG) =====");
      debugLog("🔍 Query:", userMessage);
      debugLog("📚 Chat history length:", recentHistory.length, "messages");
      debugLog("🎯 Top results to retrieve:", payload.top_results);
      debugLog("🌐 Endpoint:", listeoAiChatConfig.apiBase + "/rag-chat");
      debugLog("📦 Full payload:", payload);
      debugLog("⚙️  Listeo available:", self.chatConfig.listeo_available);
      debugLog(
        "🔧 Post types will be filtered server-side based on Listeo availability",
      );
      debugLog("===== SYSTEM PROMPT =====");
      debugLog("(Generated server-side for security)");
      debugLog("===== END SYSTEM PROMPT =====");

      // Uses silent retry - retries once on failure, shows error only on 2nd failure
      (function doRagChatRequest(attempt) {
        $.ajax({
          url: listeoAiChatConfig.apiBase + "/rag-chat",
          method: "POST",
          headers: $.extend({}, getRequestHeaders(), {
            "X-Session-ID": self.sessionId,
          }),
          data: JSON.stringify(payload),
          success: function (response) {
            debugLog("===== RAG ENDPOINT RESPONSE =====");
            debugLog("✅ Response received");
            debugLog("📄 Full response:", response);

            if (!response.success) {
              if (attempt < 2) {
                // Silent retry on first failure
                debugLog(
                  "RAG chat failed, retrying... (attempt " + attempt + ")",
                );
                doRagChatRequest(attempt + 1);
                return;
              }
              // 2nd failure - show detailed error
              var errorDetail =
                response.error?.message ||
                response.error?.type ||
                response.error?.code ||
                "Unknown error";
              debugError("❌ RAG endpoint returned success:false (final)");
              debugError("Error object:", response.error);
              self.$messages.find("#" + loadingId).remove();
              self.addMessage(
                "system",
                listeoAiChatConfig.strings.errorGeneral +
                  " [" +
                  errorDetail +
                  "]",
              );
              self.isProcessing = false;
              self.$sendBtn.prop("disabled", false);
              return;
            }

            var answer = response.answer;
            var sources = response.sources || [];

            debugLog("===== RAG RESPONSE DETAILS =====");
            if (response.query_info) {
              debugLog("📝 Query Transformation:");
              debugLog("  Original query:", response.query_info.original_query);
              debugLog(
                "  Optimized query:",
                response.query_info.optimized_query,
              );
            }
            debugLog("💬 Answer length:", answer ? answer.length : 0, "chars");
            debugLog("📚 Sources found:", sources.length);
            if (sources.length > 0) {
              debugLog("📖 Source details:");
              sources.forEach(function (source, idx) {
                debugLog(
                  "  " +
                    (idx + 1) +
                    ". " +
                    source.title +
                    " (" +
                    source.type +
                    ") - " +
                    source.url,
                );
              });
            } else {
              debugLog("⚠️  No sources returned from search");
            }
            debugLog("⏱️  Performance:", response.performance);
            debugLog("🤖 Model used:", response.model);
            if (response.usage) {
              debugLog("💰 Token usage:", response.usage);
            }

            // Remove loading
            self.$messages.find("#" + loadingId).remove();

            // Display answer
            if (answer) {
              self.addMessage("assistant", answer);
              debugLog("✅ Answer displayed to user");
            } else {
              debugError("⚠️  Empty answer received");
            }

            // Display source attribution (if sources exist)
            if (sources && sources.length > 0) {
              var sourcesHTML = self.formatSourceAttribution(sources);
              self.addMessage("assistant", sourcesHTML);
              debugLog("✅ Source attribution displayed");
            }

            // Update conversation history (simple, no tool calls)
            self.conversationHistory.push(
              { role: "user", content: userMessage },
              { role: "assistant", content: answer },
            );

            // Save conversation
            self.saveConversation();

            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
            debugLog("✅ RAG flow completed successfully");
          },
          error: function (xhr) {
            if (attempt < 2) {
              // Silent retry on first XHR failure
              debugLog(
                "RAG chat XHR error, retrying... (attempt " + attempt + ")",
              );
              doRagChatRequest(attempt + 1);
            } else {
              // 2nd failure - show detailed error
              var errorInfo = analyzeError(xhr, "rag-chat");
              self.$messages.find("#" + loadingId).remove();
              self.addMessage(
                "system",
                errorInfo.userMessage,
              );
              self.isProcessing = false;
              self.$sendBtn.prop("disabled", false);
            }
          },
        });
      })(1);
    },

    /**
     * Format source attribution cards
     */
    formatSourceAttribution: function (sourceResults) {
      // var self = this;
      // var html = '<div class="listeo-ai-sources">';
      // html += '<div class="listeo-ai-sources-label">📚 Sources used:</div>';
      // html += '<div class="listeo-ai-sources-list">';
      // sourceResults.forEach(function(source, index) {
      //     var postTypeLabel = source.post_type ? source.post_type.charAt(0).toUpperCase() + source.post_type.slice(1) : 'Content';
      //     var icon = self.getIconForPostType(source.post_type);
      //     html += '<a href="' + source.url + '" class="listeo-ai-source-card" target="_blank">';
      //     html += '  <span class="source-icon">' + icon + '</span>';
      //     html += '  <span class="source-info">';
      //     html += '    <span class="source-title">' + source.title + '</span>';
      //     html += '    <span class="source-type">' + postTypeLabel + '</span>';
      //     html += '  </span>';
      //     html += '</a>';
      // });
      // html += '</div></div>';
      // return html;
    },

    /**
     * Get icon for post type
     */
    getIconForPostType: function (postType) {
      var icons = {
        post: '<i class="fa fa-file-text-o"></i>',
        page: '<i class="fa fa-file-o"></i>',
        listing: '<i class="fa fa-map-marker"></i>',
        product: '<i class="fa fa-shopping-cart"></i>',
      };
      return icons[postType] || '<i class="fa fa-file-o"></i>';
    },

    /**
     * Search listings via API
     */
    searchListings: function (params, callback) {
      // Use Listeo-specific endpoint if available
      var endpoint = this.chatConfig.listeo_available
        ? "/listeo-hybrid-search"
        : "/universal-search";

      debugLog(
        "[searchListings] Using endpoint:",
        endpoint,
        "(Listeo available:",
        this.chatConfig.listeo_available + ")",
      );

      $.ajax({
        url: listeoAiChatConfig.apiBase + endpoint,
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify(params),
        success: function (response) {
          callback(response);
        },
        error: function (xhr) {
          callback({
            success: false,
            error: xhr.responseJSON?.error || "Search failed",
          });
        },
      });
    },

    /**
     * Universal content search via API (any post type)
     */
    searchContent: function (params, callback) {
      $.ajax({
        url: listeoAiChatConfig.apiBase + "/universal-search",
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify(params),
        success: function (response) {
          callback(response);
        },
        error: function (xhr) {
          callback({
            success: false,
            error: xhr.responseJSON?.error || "Universal search failed",
          });
        },
      });
    },

    /**
     * Get content details via API (any post type)
     */
    getContentDetails: function (
      postId,
      userMessage,
      assistantMessage,
      toolCall,
      loadingId,
    ) {
      var self = this;

      $.ajax({
        url: listeoAiChatConfig.apiBase + "/get-content",
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify({ post_id: postId }),
        success: function (response) {
          if (response.success) {
            debugLog("===== CONTENT DETAILS =====");
            debugLog("Post ID:", response.post_id);
            debugLog("Post Type:", response.post_type);
            debugLog("Title:", response.title);
            debugLog(
              "Structured content length:",
              response.structured_content.length,
            );

            // Update loading message
            self.$messages
              .find("#" + loadingId + " .listeo-ai-chat-message-content")
              .html(
                generateLoaderHTML(listeoAiChatConfig.strings.analyzingContent),
              );

            // Send structured content to OpenAI for natural response
            self.getDetailsResponse(
              userMessage,
              assistantMessage,
              toolCall,
              response,
              loadingId,
            );
          } else {
            self.$messages.find("#" + loadingId).remove();
            self.addMessage(
              "system",
              listeoAiChatConfig.strings.contentNotFound,
            );
            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
          }
        },
        error: function (xhr) {
          self.$messages.find("#" + loadingId).remove();
          self.addMessage(
            "system",
            listeoAiChatConfig.strings.errorGettingContent,
          );
          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
      });
    },

    /**
     * Format universal content grid (for posts, pages, products, etc.)
     */
    formatContentGrid: function (results) {
      var self = this;
      var html = '<div class="listeo-ai-results-list">';

      results.forEach(function (item, index) {
        var thumbnail =
          item.featured_image || listeoAiChatConfig.placeholderImage || "";
        var title = item.title || "Untitled";
        var excerpt = item.excerpt || "";
        var postType = item.post_type || "post";
        var url = item.url || "#";

        // Best Match badge for first result
        var bestMatchBadge = "";
        if (index === 0) {
          bestMatchBadge =
            '<div class="match-badge best">' +
            (listeoAiChatConfig.strings.bestMatch || "Best Match") +
            "</div>";
        }

        // Add hidden class to items after the first 3
        var hiddenClass = index >= 3 ? " listeo-ai-listing-hidden" : "";

        // Format post type label
        var postTypeLabel =
          postType.charAt(0).toUpperCase() + postType.slice(1);

        html +=
          '<a href="' +
          url +
          '" class="listeo-ai-listing-item' +
          hiddenClass +
          '">';

        // Only render thumbnail if hideImages is not enabled
        if (!self.hideImages && thumbnail) {
          html += '  <div class="listeo-ai-listing-thumbnail">';
          html += '    <img src="' + thumbnail + '" alt="' + title + '">';
          html += "  </div>";
        }

        html += '  <div class="listeo-ai-listing-details">';
        html += '    <div class="listeo-ai-listing-main">';
        html +=
          '      <h3 class="listeo-ai-listing-title">' +
          title +
          " " +
          bestMatchBadge +
          "</h3>";
        if (excerpt) {
          html +=
            '      <p class="listeo-ai-listing-excerpt">' + excerpt + "</p>";
        }
        html += '      <div class="listeo-ai-listing-meta">';
        html +=
          '        <span class="content-type"><i class="fa fa-file-text-o"></i> ' +
          postTypeLabel +
          "</span>";
        html += "      </div>";
        html += "    </div>";
        html += "  </div>";
        html += "</a>";
      });

      html += "</div>";

      // Add "Show more" button if more than 3 results
      if (results.length > 3) {
        html +=
          '<button class="listeo-ai-show-more-btn">' +
          listeoAiChatConfig.strings.showMore.replace(
            "%d",
            results.length - 3,
          ) +
          "</button>";
      }

      return html;
    },

    /**
     * Safely slice conversation history without breaking tool calling sequences
     * OpenAI requires: assistant[tool_calls] → tool → assistant (complete sequence)
     */
    getValidHistorySlice: function (maxMessages) {
      // FIX: Always validate history, not just when slicing
      // Previously, history <= maxMessages was returned without validation,
      // causing API errors when corrupted tool_calls sequences existed
      var sliced = this.conversationHistory.length <= maxMessages
        ? this.conversationHistory.slice() // Copy to avoid mutating original
        : this.conversationHistory.slice(-maxMessages);

      // Remove any orphaned 'tool' messages at the start (tool without preceding assistant)
      while (sliced.length > 0 && sliced[0].role === "tool") {
        debugLog(
          "[History Validation] Removing orphaned tool message from start",
        );
        sliced.shift();
      }

      // Comprehensive validation: scan entire array and remove incomplete sequences
      var validated = [];
      for (var i = 0; i < sliced.length; i++) {
        var msg = sliced[i];

        // If this is an assistant message with tool_calls
        if (msg.role === "assistant" && msg.tool_calls) {
          // Check if next message is a tool response
          var nextMsg = sliced[i + 1];
          if (!nextMsg || nextMsg.role !== "tool") {
            debugLog(
              "[History Validation] Removing assistant+tool_calls at index " +
                i +
                " - no tool response follows",
            );
            // Skip this message and all remaining (incomplete sequence at end)
            break;
          }

          // Valid sequence start - add assistant message
          validated.push(msg);
        } else if (msg.role === "tool") {
          // Tool message should only appear after assistant+tool_calls (already validated above)
          validated.push(msg);
        } else {
          // Regular user or assistant message (no tool_calls)
          validated.push(msg);
        }
      }

      debugLog(
        "[History Validation] Original: " +
          sliced.length +
          " messages, Validated: " +
          validated.length +
          " messages",
      );
      return validated;
    },

    /**
     * Get listing details via API (supports multiple IDs for comparison)
     * @param {Array} listingIds - Array of listing IDs to fetch
     */
    getListingDetails: function (
      listingIds,
      userMessage,
      assistantMessage,
      toolCall,
      loadingId,
    ) {
      var self = this;

      // Ensure we have an array
      if (!Array.isArray(listingIds)) {
        listingIds = [listingIds];
      }

      // Use Listeo-specific endpoint if available
      var endpoint = this.chatConfig.listeo_available
        ? "/listeo-listing-details"
        : "/get-content";

      // Build params - use listing_ids for multiple, listing_id for single (backward compat with get-content)
      var param;
      if (this.chatConfig.listeo_available) {
        param = listingIds.length > 1
          ? { listing_ids: listingIds }
          : { listing_id: listingIds[0] };
      } else {
        // Fallback get-content only supports single ID
        param = { post_id: listingIds[0] };
      }

      debugLog(
        "[getListingDetails] Using endpoint:",
        endpoint,
        "(Listeo available:",
        this.chatConfig.listeo_available + ")",
        "IDs:", listingIds,
      );

      $.ajax({
        url: listeoAiChatConfig.apiBase + endpoint,
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify(param),
        success: function (response) {
          if (response.success) {
            debugLog("===== LISTING DETAILS =====");

            // Normalize response structure for getDetailsResponse
            // Backend returns different formats for single vs multiple
            var normalizedResponse;
            if (response.listings && Array.isArray(response.listings)) {
              // Multiple listings - combine structured content
              debugLog("Multiple listings:", response.count);
              var combinedContent = response.listings.map(function(listing, index) {
                return "=== LISTING " + (index + 1) + ": " + listing.title + " ===\n" +
                       "URL: " + listing.url + "\n\n" +
                       listing.structured_content;
              }).join("\n\n" + "=".repeat(50) + "\n\n");

              normalizedResponse = {
                success: true,
                listing_id: response.listings.map(function(l) { return l.listing_id; }).join(", "),
                title: response.listings.map(function(l) { return l.title; }).join(" vs "),
                structured_content: combinedContent,
                is_comparison: true,
                count: response.count
              };
            } else {
              // Single listing - use as-is
              debugLog("Listing/Post ID:", response.listing_id || response.post_id);
              debugLog("Title:", response.title);
              normalizedResponse = response;
            }

            debugLog("Structured content length:", normalizedResponse.structured_content.length);

            // Update loading message
            var loadingText = normalizedResponse.is_comparison
              ? (listeoAiChatConfig.strings.comparingListings || "Comparing listings...")
              : listeoAiChatConfig.strings.analyzingListing;

            self.$messages
              .find("#" + loadingId + " .listeo-ai-chat-message-content")
              .html(generateLoaderHTML(loadingText));

            // Send structured content to OpenAI for natural response
            self.getDetailsResponse(
              userMessage,
              assistantMessage,
              toolCall,
              normalizedResponse,
              loadingId,
            );
          } else {
            self.$messages.find("#" + loadingId).remove();
            self.addMessage(
              "system",
              listeoAiChatConfig.strings.listingNotFound,
            );
            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
          }
        },
        error: function (xhr) {
          self.$messages.find("#" + loadingId).remove();
          self.addMessage(
            "system",
            listeoAiChatConfig.strings.errorGettingDetails,
          );
          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
      });
    },

    /**
     * Get product details via API (WooCommerce) - supports multiple IDs for comparison
     * @param {Array} productIds - Array of product IDs to fetch
     */
    getProductDetails: function (
      productIds,
      userMessage,
      assistantMessage,
      toolCall,
      loadingId,
    ) {
      var self = this;

      // Ensure we have an array
      if (!Array.isArray(productIds)) {
        productIds = [productIds];
      }

      // Build params - use product_ids for multiple, product_id for single (backward compat)
      var param = productIds.length > 1
        ? { product_ids: productIds }
        : { product_id: productIds[0] };

      debugLog("[getProductDetails] Fetching products:", productIds);

      $.ajax({
        url: listeoAiChatConfig.apiBase + "/woocommerce-product-details",
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify(param),
        success: function (response) {
          if (response.success) {
            debugLog("===== PRODUCT DETAILS =====");

            // Normalize response structure for getDetailsResponse
            // Backend returns different formats for single vs multiple
            var normalizedResponse;
            if (response.products && Array.isArray(response.products)) {
              // Multiple products - combine structured content
              debugLog("Multiple products:", response.count);
              var combinedContent = response.products.map(function(product, index) {
                return "=== PRODUCT " + (index + 1) + ": " + product.title + " ===\n" +
                       "URL: " + product.url + "\n\n" +
                       product.structured_content;
              }).join("\n\n" + "=".repeat(50) + "\n\n");

              normalizedResponse = {
                success: true,
                product_id: response.products.map(function(p) { return p.product_id; }).join(", "),
                title: response.products.map(function(p) { return p.title; }).join(" vs "),
                structured_content: combinedContent,
                is_comparison: true,
                count: response.count
              };
            } else {
              // Single product - use as-is
              debugLog("Product ID:", response.product_id);
              debugLog("Title:", response.title);
              normalizedResponse = response;
            }

            debugLog("Structured content length:", normalizedResponse.structured_content.length);

            // Update loading message
            var loadingText = normalizedResponse.is_comparison
              ? (listeoAiChatConfig.strings.comparingProducts || "Comparing products...")
              : listeoAiChatConfig.strings.analyzingProduct;

            self.$messages
              .find("#" + loadingId + " .listeo-ai-chat-message-content")
              .html(generateLoaderHTML(loadingText));

            // Send structured content to OpenAI for natural response
            self.getDetailsResponse(
              userMessage,
              assistantMessage,
              toolCall,
              normalizedResponse,
              loadingId,
            );
          } else {
            self.$messages.find("#" + loadingId).remove();
            self.addMessage(
              "system",
              listeoAiChatConfig.strings.productNotFound,
            );
            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
          }
        },
        error: function (xhr) {
          self.$messages.find("#" + loadingId).remove();
          self.addMessage(
            "system",
            listeoAiChatConfig.strings.errorGettingProduct,
          );
          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
      });
    },

    /**
     * Get order status via API (WooCommerce)
     */
    getOrderStatus: function (
      orderNumber,
      billingEmail,
      userMessage,
      assistantMessage,
      toolCall,
      loadingId,
    ) {
      var self = this;

      // Build request data
      var requestData = {
        order_number: orderNumber,
      };

      // Add billing email if provided
      if (billingEmail) {
        requestData.billing_email = billingEmail;
      }

      debugLog("===== ORDER STATUS REQUEST =====");
      debugLog("Order Number:", orderNumber);
      debugLog("Billing Email:", billingEmail || "Not provided");

      $.ajax({
        url: listeoAiChatConfig.apiBase + "/woocommerce-order-status",
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify(requestData),
        success: function (response) {
          if (response.success) {
            debugLog("===== ORDER STATUS RESPONSE =====");
            debugLog("Order Number:", response.order_number);
            debugLog("Status:", response.status);
            debugLog(
              "Structured content length:",
              response.structured_content.length,
            );

            // Update loading message
            self.$messages
              .find("#" + loadingId + " .listeo-ai-chat-message-content")
              .html(
                generateLoaderHTML(
                  listeoAiChatConfig.strings.analyzingOrderDetails,
                ),
              );

            // Send structured content to OpenAI for natural response
            self.getDetailsResponse(
              userMessage,
              assistantMessage,
              toolCall,
              response,
              loadingId,
            );
          } else {
            // Check if email verification is required
            if (response.requires_email) {
              debugLog("Order verification required - asking for email");
              self.$messages.find("#" + loadingId).remove();
              self.addMessage(
                "system",
                response.error ||
                  listeoAiChatConfig.strings.orderVerificationRequired,
              );

              // Suggest user provide email in next message
              var suggestionMsg =
                "<p>💡 <strong>Tip:</strong> Please provide the email address you used when placing the order.</p>";
              self.addMessage("assistant", suggestionMsg);
            } else {
              self.$messages.find("#" + loadingId).remove();
              self.addMessage(
                "system",
                response.error || listeoAiChatConfig.strings.orderNotFound,
              );
            }

            self.isProcessing = false;
            self.$sendBtn.prop("disabled", false);
          }
        },
        error: function (xhr) {
          debugError("===== ORDER STATUS ERROR =====");
          debugError("Status:", xhr.status);
          debugError("Response:", xhr.responseJSON);

          self.$messages.find("#" + loadingId).remove();

          var errorMsg = listeoAiChatConfig.strings.errorGettingOrder;
          if (xhr.responseJSON && xhr.responseJSON.error) {
            errorMsg = xhr.responseJSON.error;
          } else if (xhr.status === 403) {
            errorMsg = listeoAiChatConfig.strings.orderVerificationRequired;
          } else if (xhr.status === 404) {
            errorMsg = listeoAiChatConfig.strings.orderNotFound;
          }

          self.addMessage("system", errorMsg);
          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
      });
    },

    /**
     * Get AI response for listing details
     */
    getDetailsResponse: function (
      userMessage,
      assistantMessage,
      toolCall,
      detailsResponse,
      loadingId,
    ) {
      var self = this;

      // Get valid history slice
      var recentHistory = self.getValidHistorySlice(10);

      // Build API payload (server handles system prompt injection for security)
      var payload = {
        model: self.chatConfig.model,
        messages: recentHistory.concat([
          { role: "user", content: userMessage },
          assistantMessage,
          {
            role: "tool",
            tool_call_id: toolCall.id,
            content: detailsResponse.structured_content, // Semantic embedding content
          },
        ]),
      };

      // If listing context is loaded, send listing ID for server-side injection
      if (self.loadedListing && self.loadedListing.id) {
        payload.listing_context_id = self.loadedListing.id;
      }

      // If product context is loaded, send product ID for server-side injection
      if (self.loadedProduct && self.loadedProduct.id) {
        payload.product_context_id = self.loadedProduct.id;
      }

      // Configure model-specific parameters (GPT-5 vs GPT-4)
      configureModelPayload(payload, self.chatConfig);
      logModelDebug(payload, "Listing Details", self.chatConfig);

      debugLog("===== SENDING LISTING DETAILS TO OPENAI =====");
      debugLog("Payload messages count:", payload.messages.length);
      debugLog(
        "Tool response content length:",
        detailsResponse.structured_content.length,
        "chars",
      );
      debugLog(
        "Content preview (first 500 chars):",
        detailsResponse.structured_content.substring(0, 500),
      );
      debugLog(
        "Content preview (last 500 chars):",
        detailsResponse.structured_content.substring(
          detailsResponse.structured_content.length - 500,
        ),
      );
      debugLog("Full payload:", payload);
      logApiRequest(payload);

      $.ajax({
        url: listeoAiChatConfig.apiBase + "/chat-proxy",
        method: "POST",
        headers: $.extend({}, getRequestHeaders(), {
          "X-Session-ID": self.sessionId,
        }),
        data: JSON.stringify(payload),
        success: function (data) {
          var finalMessage = data.choices[0].message.content;

          // Check for empty content
          if (!finalMessage || finalMessage.trim() === "") {
            debugError("[AI Chat ERROR] getDetailsResponse - LLM returned empty content");
            debugError("[AI Chat ERROR] Full response:", JSON.stringify(data, null, 2));
            debugError("[AI Chat ERROR] Finish reason:", data.choices?.[0]?.finish_reason);
            if (data.choices?.[0]?.message?.tool_calls) {
              debugError("[AI Chat ERROR] AI returned more tool_calls instead of content");
            }
            finalMessage = listeoAiChatConfig.strings.noResultsGeneric ||
              "I couldn't process that request. Please try again.";
          }

          // Remove loading indicator
          self.$messages.find("#" + loadingId).remove();

          // Add AI response
          self.addMessage("assistant", finalMessage);

          // Update history - MUST include complete tool calling sequence
          self.conversationHistory.push(
            { role: "user", content: userMessage },
            assistantMessage, // Assistant message WITH tool_calls
            {
              role: "tool",
              tool_call_id: toolCall.id,
              content: detailsResponse.structured_content,
            },
            { role: "assistant", content: finalMessage }, // Final response
          );

          // Save conversation
          self.saveConversation();

          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
        error: function (xhr) {
          var errorInfo = analyzeError(xhr, "getDetailsResponse");
          self.$messages.find("#" + loadingId).remove();
          self.addMessage("system", errorInfo.userMessage);
          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
      });
    },

    /**
     * Get final response from OpenAI
     */
    getFinalResponse: function (
      userMessage,
      assistantMessage,
      toolCall,
      apiResult,
      loadingId,
    ) {
      var self = this;

      // Get valid history slice
      var recentHistory = self.getValidHistorySlice(10);

      // Detect if this is a product search or listing search based on tool name
      var isProductSearch = toolCall.function.name === "search_products";

      // Format results based on search type
      var condensedResults = {
        original_question: self.extractTextFromMessage(userMessage), // Text only, no images
        total: apiResult.total,
      };

      if (isProductSearch) {
        // Product-specific format with price and stock info
        condensedResults.products = apiResult.results
          ? apiResult.results.map(function (r) {
              return {
                id: r.id,
                title: r.title,
                price: r.price?.formatted || "",
                stock_status: r.stock_status || "",
                on_sale: r.on_sale || false,
                url: r.url || "",
              };
            })
          : [];
      } else {
        // Listing-specific format with address
        condensedResults.listings = apiResult.results
          ? apiResult.results.map(function (r) {
              var listing = {
                id: r.id,
                title: r.title,
                address: r.location?.address || "",
                url: r.url || "",
              };
              // Include event dates if available (cache-agnostic: only if PHP provides it)
              if (r.event_dates) {
                listing.event_dates = r.event_dates;
              }
              return listing;
            })
          : [];
      }

      debugLog("===== CONDENSED RESULTS SENT TO AI =====");
      debugLog(JSON.stringify(condensedResults, null, 2));

      // Build API payload for final response (server handles system prompt for security)
      var payload = {
        model: self.chatConfig.model,
        messages: recentHistory.concat([
          { role: "user", content: userMessage },
          assistantMessage,
          {
            role: "tool",
            tool_call_id: toolCall.id,
            content: JSON.stringify(condensedResults),
          },
        ]),
      };

      // If listing context is loaded, send listing ID for server-side injection
      if (self.loadedListing && self.loadedListing.id) {
        payload.listing_context_id = self.loadedListing.id;
      }

      // If product context is loaded, send product ID for server-side injection
      if (self.loadedProduct && self.loadedProduct.id) {
        payload.product_context_id = self.loadedProduct.id;
      }

      // Configure model-specific parameters (GPT-5 vs GPT-4)
      configureModelPayload(payload, self.chatConfig);
      logModelDebug(payload, "Tool Response Summary", self.chatConfig);

      debugLog("===== SENDING MESSAGE TO OPENAI (SECOND CALL - SUMMARY) =====");
      debugLog(
        "Conversation history length:",
        recentHistory.length,
        "messages",
      );
      debugLog("Listing context ID:", payload.listing_context_id || "none");
      debugLog("Full messages array:", payload.messages);
      debugLog("Condensed results being sent:", condensedResults);
      debugLog("Complete payload:", payload);
      logApiRequest(payload);

      $.ajax({
        url: listeoAiChatConfig.apiBase + "/chat-proxy",
        method: "POST",
        headers: $.extend({}, getRequestHeaders(), {
          "X-Session-ID": self.sessionId,
        }),
        data: JSON.stringify(payload),
        success: function (data) {
          var finalMessage =
            data.choices && data.choices[0] && data.choices[0].message
              ? data.choices[0].message.content
              : null;

          // Handle empty content (shouldn't happen with parallel_tool_calls: false)
          if (!finalMessage || finalMessage.trim() === "") {
            var finishReason = data.choices?.[0]?.finish_reason;
            var responseMessage = data.choices?.[0]?.message;

            debugError("[AI Chat ERROR] LLM returned empty content");
            debugError("[AI Chat ERROR] Finish reason:", finishReason);

            // Log if AI tried to chain tool calls (indicates parallel_tool_calls didn't work)
            if (finishReason === "tool_calls" && responseMessage?.tool_calls) {
              debugError("[AI Chat ERROR] AI attempted chained tool_calls despite parallel_tool_calls: false");
              debugError("[AI Chat ERROR] Tools:", responseMessage.tool_calls.map(function(tc) { return tc.function.name; }));
            }

            finalMessage =
              listeoAiChatConfig.strings.noResultsGeneric ||
              "I couldn't find results matching your search. Try different keywords or be more specific about what you're looking for.";
          }

          // Remove loading indicator
          self.$messages.find("#" + loadingId).remove();

          // Add text response ONLY (grid already shown above)
          self.addMessage("assistant", finalMessage);

          // Update history - MUST include complete tool calling sequence
          self.conversationHistory.push(
            { role: "user", content: userMessage },
            assistantMessage, // Assistant message WITH tool_calls
            {
              role: "tool",
              tool_call_id: toolCall.id,
              content: JSON.stringify(condensedResults),
            },
            { role: "assistant", content: finalMessage }, // Final response
          );

          // Save conversation to localStorage
          self.saveConversation();

          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
        error: function (xhr) {
          var errorInfo = analyzeError(xhr, "getFinalResponse");
          self.$messages.find("#" + loadingId).remove();
          self.addMessage("system", errorInfo.userMessage);
          self.isProcessing = false;
          self.$sendBtn.prop("disabled", false);
        },
      });
    },

    /**
     * Format listings grid
     */
    formatListingsGrid: function (results) {
      var self = this; // Capture 'this' for use in forEach callback
      var html = '<div class="listeo-ai-results-list">';

      results.forEach(function (listing, index) {
        // Use theme placeholder (matches frontend)
        var thumbnail =
          listing.featured_image || listeoAiChatConfig.placeholderImage || "";
        var title = listing.title || "Untitled";
        var excerpt = listing.excerpt || "";
        var location = listing.location?.address || "";
        var rating = listing.rating?.average || 0;
        var ratingCount = listing.rating?.count || 0;
        var url = listing.url || "#";

        // Best Match badge for first result (matches frontend)
        var bestMatchBadge = "";
        if (index === 0) {
          bestMatchBadge =
            '<div class="match-badge best">' +
            (listeoAiChatConfig.strings.bestMatch || "Best Match") +
            "</div>";
        }

        // Add hidden class to items after the first 3
        var hiddenClass = index >= 3 ? " listeo-ai-listing-hidden" : "";
        // Add no-thumbnail class if thumbnail is missing
        var noThumbnailClass = (!self.hideImages && thumbnail) ? "" : " no-thumbnail";

        html +=
          '<a href="' +
          url +
          '" class="listeo-ai-listing-item' +
          hiddenClass +
          noThumbnailClass +
          '">';

        // Only render thumbnail if hideImages is not enabled AND we have a valid thumbnail
        if (!self.hideImages && thumbnail) {
          html += '  <div class="listeo-ai-listing-thumbnail">';
          html += '    <img src="' + thumbnail + '" alt="' + title + '">';
          html += "  </div>";
        }

        html += '  <div class="listeo-ai-listing-details">';
        html += '    <div class="listeo-ai-listing-main">';
        html +=
          '      <h3 class="listeo-ai-listing-title">' +
          title +
          " " +
          bestMatchBadge +
          "</h3>";
        if (excerpt) {
          html +=
            '      <p class="listeo-ai-listing-excerpt">' + excerpt + "</p>";
        }
        html += '      <div class="listeo-ai-listing-meta">';
        if (location) {
          html +=
            '        <span class="address"><i class="fa fa-map-marker"></i> ' +
            location +
            "</span>";
        }
        if (rating > 0) {
          html +=
            '        <span class="listeo-ai-listing-rating"><svg class="rating-icon" width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><path d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z"/></svg> ' +
            parseFloat(rating).toFixed(1) +
            "</span>";
        }
        html += "      </div>";
        html += "    </div>";
        html += "  </div>";
        html += "</a>";
      });

      html += "</div>";

      // Add "Show more" button if more than 3 results
      if (results.length > 3) {
        html +=
          '<button class="listeo-ai-show-more-btn">' +
          listeoAiChatConfig.strings.showMore.replace(
            "%d",
            results.length - 3,
          ) +
          "</button>";
      }

      return html;
    },

    /**
     * Format products grid (for WooCommerce products)
     */
    formatProductsGrid: function (results) {
      var self = this;
      var html = '<div class="listeo-ai-results-list">';

      results.forEach(function (product, index) {
        // Use theme placeholder (matches frontend)
        var thumbnail =
          product.featured_image || listeoAiChatConfig.placeholderImage || "";
        var title = product.title || "Untitled";
        var excerpt = product.excerpt || "";
        var price = product.price?.formatted || "";
        var regularPrice = product.price?.regular || "";
        var salePrice = product.price?.sale || null;
        var onSale = product.on_sale || false;
        var stockStatus = product.stock_status || "";
        var rating = product.rating?.average || 0;
        var ratingCount = product.rating?.count || 0;
        var url = product.url || "#";

        // Best Match badge for first result
        var bestMatchBadge = "";
        if (index === 0) {
          bestMatchBadge =
            '<div class="match-badge best">' +
            (listeoAiChatConfig.strings.bestMatch || "Best Match") +
            "</div>";
        }

        // Add hidden class to items after the first 3
        var hiddenClass = index >= 3 ? " listeo-ai-listing-hidden" : "";
        // Add no-thumbnail class if thumbnail is missing
        var noThumbnailClass = (!self.hideImages && thumbnail) ? "" : " no-thumbnail";

        html +=
          '<a href="' +
          url +
          '" class="listeo-ai-listing-item' +
          hiddenClass +
          noThumbnailClass +
          '">';

        // Only render thumbnail if hideImages is not enabled AND we have a valid thumbnail
        if (!self.hideImages && thumbnail) {
          html += '  <div class="listeo-ai-listing-thumbnail">';
          html += '    <img src="' + thumbnail + '" alt="' + title + '">';
          html += "  </div>";
        }

        html += '  <div class="listeo-ai-listing-details">';
        html += '    <div class="listeo-ai-listing-main">';
        html +=
          '      <h3 class="listeo-ai-listing-title">' +
          title +
          " " +
          bestMatchBadge +
          "</h3>";
        if (excerpt) {
          html +=
            '      <p class="listeo-ai-listing-excerpt">' + excerpt + "</p>";
        }
        html += '      <div class="listeo-ai-listing-meta">';

        // Price display (with sale price handling)
        if (price) {
          if (onSale && salePrice) {
            html +=
              '        <span class="product-price"><span class="regular-price">' +
              regularPrice +
              '</span> <span class="sale-price">' +
              salePrice +
              "</span></span>";
          } else {
            html += '        <span class="product-price">' + price + "</span>";
          }
        }

        // Stock status
        if (stockStatus === "instock") {
          html +=
            '        <span class="stock-status in-stock"><svg class="stock-icon" width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm-2 15l-5-5 1.41-1.41L10 14.17l7.59-7.59L19 8l-9 9z"/></svg> ' +
            listeoAiChatConfig.strings.inStock +
            "</span>";
        } else if (stockStatus === "outofstock") {
          html +=
            '        <span class="stock-status out-of-stock"><svg class="stock-icon" width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><path d="M12 2C6.47 2 2 6.47 2 12s4.47 10 10 10 10-4.47 10-10S17.53 2 12 2zm5 13.59L15.59 17 12 13.41 8.41 17 7 15.59 10.59 12 7 8.41 8.41 7 12 10.59 15.59 7 17 8.41 13.41 12 17 15.59z"/></svg> ' +
            listeoAiChatConfig.strings.outOfStock +
            "</span>";
        }

        // Rating
        if (rating > 0) {
          html +=
            '        <span class="listeo-ai-listing-rating"><svg class="rating-icon" width="14" height="14" viewBox="0 0 24 24" fill="currentColor"><path d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z"/></svg> ' +
            parseFloat(rating).toFixed(1) +
            "</span>";
        }

        html += "      </div>";
        html += "    </div>";
        html += "  </div>";
        html += "</a>";
      });

      html += "</div>";

      // Add "Show more" button if more than 3 results
      if (results.length > 3) {
        html +=
          '<button class="listeo-ai-show-more-btn">' +
          listeoAiChatConfig.strings.showMore.replace(
            "%d",
            results.length - 3,
          ) +
          "</button>";
      }

      return html;
    },

    /**
     * Add message to chat
     * @param {string} role - Message role (user, assistant, system)
     * @param {string} content - Message content (text or HTML)
     * @param {string} id - Optional message ID
     * @param {boolean} skipAnimation - Skip typing animation (for restored messages)
     */
    addMessage: function (role, content, id, skipAnimation) {
      // Skip if content is empty/undefined
      if (!content) {
        console.warn("Skipping message with empty content");
        return;
      }

      var $message = $(
        '<div class="listeo-ai-chat-message listeo-ai-chat-message-' +
          role +
          '"></div>',
      );
      if (id) {
        $message.attr("id", id);
      }

      // Check if content contains search results grid or other non-streamable content
      var isResultsGrid = content.indexOf("listeo-ai-results-list") !== -1;
      var isSourceAttribution = content.indexOf("listeo-ai-sources") !== -1;
      var isLoadingMessage = id && id.indexOf("loading") !== -1;

      if (isResultsGrid) {
        $message.addClass("chat-message-results");
      }

      // Add avatar for assistant messages if avatar is configured
      if (role === "assistant" && listeoAiChatConfig.chatAvatarUrl) {
        $message.addClass("has-avatar");
        var $avatar = $(
          '<img class="listeo-ai-chat-message-avatar" src="' +
            listeoAiChatConfig.chatAvatarUrl +
            '" alt="" />',
        );
        $message.append($avatar);
      }

      var $content = $('<div class="listeo-ai-chat-message-content"></div>');

      // Check if content contains HTML
      if (content.indexOf("<") !== -1) {
        $content.html(content);
      } else {
        $content.text(content);
      }

      $message.append($content);
      this.$messages.append($message);

      // Apply typing animation for NEW assistant text responses only
      // Skip for: restored messages, grids, sources, loading indicators
      var shouldAnimate = listeoAiChatConfig.typingAnimation &&
                          role === "assistant" &&
                          !skipAnimation &&
                          !isResultsGrid &&
                          !isSourceAttribution &&
                          !isLoadingMessage;

      // For assistant messages in floating widget, scroll the message itself to top so it's visible
      // (results grid will be above, user question further above)
      // Skip this for shortcodes - they're larger and don't need this behavior
      var isFloatingWidget = this.$wrapper.closest(".listeo-floating-chat-popup").length > 0;
      if (role === "assistant" && !isResultsGrid && isFloatingWidget) {
        $message[0].scrollIntoView({ block: "start", behavior: "instant" });
      } else {
        // For user/system messages, result grids, and shortcodes - scroll to bottom as normal
        this.$messages.scrollTop(this.$messages[0].scrollHeight);
      }

      if (shouldAnimate) {
        this.animateTyping($content[0]);
      }
    },

    /**
     * Animate typing effect for message content
     * Wraps words in spans and reveals them with staggered timing
     */
    animateTyping: function (contentElement) {
      var self = this;
      var WORDS_PER_BATCH = 3; // Words to reveal per tick
      var TICK_INTERVAL = 50; // ms between batches
      var FADE_TRAIL_LENGTH = 5; // Number of words in fade trail

      // Add streaming-active class to enable hiding of unrevealed words
      contentElement.classList.add("streaming-active");

      // Hide block elements initially - they'll be revealed when their content appears
      var hiddenElements = contentElement.querySelectorAll("li, p, h1, h2, h3, h4, h5, h6");
      hiddenElements.forEach(function (el) {
        el.classList.add("stream-block-hidden");
      });

      // Use TreeWalker to find all text nodes (preserves HTML structure)
      var walker = document.createTreeWalker(
        contentElement,
        NodeFilter.SHOW_TEXT,
        {
          acceptNode: function (node) {
            // Skip text nodes inside code/pre elements - don't animate code
            if (node.parentNode.closest("code, pre")) {
              return NodeFilter.FILTER_REJECT;
            }
            return NodeFilter.FILTER_ACCEPT;
          }
        },
        false
      );

      var textNodes = [];
      while (walker.nextNode()) {
        textNodes.push(walker.currentNode);
      }

      // Wrap each word in a span
      var allWordSpans = [];
      textNodes.forEach(function (textNode) {
        var text = textNode.textContent;
        if (!text.trim()) return; // Skip empty text nodes

        // Split into words while preserving whitespace
        var parts = text.split(/(\s+)/);
        var fragment = document.createDocumentFragment();

        parts.forEach(function (part) {
          if (part.match(/^\s+$/)) {
            // Whitespace - keep as text node
            fragment.appendChild(document.createTextNode(part));
          } else if (part) {
            // Word - wrap in span
            var span = document.createElement("span");
            span.className = "stream-word";
            span.textContent = part;
            fragment.appendChild(span);
            allWordSpans.push(span);
          }
        });

        textNode.parentNode.replaceChild(fragment, textNode);
      });

      // If no words to animate, remove class and exit
      if (allWordSpans.length === 0) {
        contentElement.classList.remove("streaming-active");
        hiddenElements.forEach(function (el) {
          el.classList.remove("stream-block-hidden");
        });
        return;
      }

      // Animate words appearing in batches
      var currentIndex = 0;

      function revealBatch() {
        var endIndex = Math.min(currentIndex + WORDS_PER_BATCH, allWordSpans.length);

        // Reveal current batch
        for (var i = currentIndex; i < endIndex; i++) {
          var span = allWordSpans[i];
          span.classList.add("visible");

          // Check if this word is inside a hidden block element - reveal it
          var parentBlock = span.closest(".stream-block-hidden");
          if (parentBlock) {
            parentBlock.classList.remove("stream-block-hidden");
          }
        }

        // Update fade trail - last N visible words get fading classes
        for (var j = 0; j < allWordSpans.length; j++) {
          var span = allWordSpans[j];
          // Remove all trail classes
          span.classList.remove("trail-1", "trail-2", "trail-3", "trail-4", "trail-5");

          // Add trail classes to last few visible words
          if (span.classList.contains("visible")) {
            var distanceFromEnd = endIndex - 1 - j;
            if (distanceFromEnd >= 0 && distanceFromEnd < FADE_TRAIL_LENGTH) {
              span.classList.add("trail-" + (distanceFromEnd + 1));
            }
          }
        }

        currentIndex = endIndex;

        // For shortcodes, scroll to bottom during animation so user sees latest content
        // For floating widget, no scroll - keeps user's message visible at top
        var isFloatingWidget = self.$wrapper.closest(".listeo-floating-chat-popup").length > 0;
        if (!isFloatingWidget) {
          self.$messages.scrollTop(self.$messages[0].scrollHeight);
        }

        // Continue or finish
        if (currentIndex < allWordSpans.length) {
          setTimeout(revealBatch, TICK_INTERVAL);
        } else {
          // Animation complete - remove streaming class and unwrap spans
          contentElement.classList.remove("streaming-active");
          hiddenElements.forEach(function (el) {
            el.classList.remove("stream-block-hidden");
          });
          allWordSpans.forEach(function (span) {
            // Move all child nodes out of span (preserves images, emojis, etc.)
            while (span.firstChild) {
              span.parentNode.insertBefore(span.firstChild, span);
            }
            span.parentNode.removeChild(span);
          });
        }
      }

      // Start animation
      revealBatch();
    },

    /**
     * Save conversation to localStorage
     */
    saveConversation: function () {
      try {
        var data = {
          history: this.conversationHistory,
          messages: [],
          timestamp: Date.now(),
        };

        // Save message HTML for display (skip listing-action messages)
        this.$messages.find(".listeo-ai-chat-message").each(function () {
          var $msg = $(this);

          // Skip listing-action messages (don't persist them)
          if ($msg.hasClass("listeo-ai-chat-message-listing-action")) {
            return; // continue to next message
          }

          var role = "assistant";
          if ($msg.hasClass("listeo-ai-chat-message-user")) {
            role = "user";
          } else if ($msg.hasClass("listeo-ai-chat-message-system")) {
            role = "system";
          }

          data.messages.push({
            role: role,
            content: $msg.find(".listeo-ai-chat-message-content").html(),
          });
        });

        localStorage.setItem(this.storageKey, JSON.stringify(data));
        debugLog("Conversation saved to localStorage");
      } catch (e) {
        debugError("Failed to save conversation:", e);
      }
    },

    /**
     * Load conversation from localStorage
     */
    loadConversation: function () {
      try {
        var data = localStorage.getItem(this.storageKey);
        if (!data) {
          debugLog(
            "No saved conversation found - showing fresh welcome message",
          );
          // Replace HTML welcome message with current setting from JS
          this.$messages.empty();
          this.addMessage("system", listeoAiChatConfig.strings.welcomeMessage, null, true);
          return;
        }

        data = JSON.parse(data);

        // Check if conversation is less than 24 hours old
        var hoursSinceLastMessage =
          (Date.now() - data.timestamp) / (1000 * 60 * 60);
        if (hoursSinceLastMessage > 24) {
          debugLog("Conversation expired (older than 24 hours)");
          localStorage.removeItem(this.storageKey);
          // Show fresh welcome message
          this.$messages.empty();
          this.addMessage("system", listeoAiChatConfig.strings.welcomeMessage, null, true);
          return;
        }

        // Restore conversation history
        this.conversationHistory = data.history || [];

        // Clear welcome message and restore messages
        this.$messages.empty();

        var self = this;
        if (data.messages && data.messages.length > 0) {
          data.messages.forEach(function (msg) {
            // Skip messages with empty content
            if (msg.content) {
              // Pass true for skipAnimation - don't animate restored messages
              self.addMessage(msg.role, msg.content, null, true);
            }
          });
          debugLog(
            "Loaded " + data.messages.length + " messages from localStorage",
          );

          // Expand chat when loading past conversation (Style 2 only)
          this.expandChat();

          // Hide quick buttons if there are existing messages and setting is "hide_after_first"
          if (listeoAiChatConfig.quickButtonsVisibility === "hide_after_first") {
            var $quickBtns = this.$wrapper.find(".listeo-ai-chat-quick-buttons");
            $quickBtns.find(".listeo-ai-quick-btn").hide();
            $quickBtns.hide();
            debugLog("[Quick Buttons] Hidden (existing conversation loaded)");
          }
        } else {
          // Show welcome message if no saved messages
          this.addMessage("system", listeoAiChatConfig.strings.welcomeMessage, null, true);
        }
      } catch (e) {
        debugError("Failed to load conversation:", e);
        // Clear and show fresh welcome message on error
        this.$messages.empty();
        this.addMessage("system", listeoAiChatConfig.strings.welcomeMessage, null, true);
      }
    },

    /**
     * Get or create a unique session ID for conversation tracking
     */
    getOrCreateSessionId: function () {
      var sessionKey = "listeo_ai_session_" + this.chatId;
      var sessionId = localStorage.getItem(sessionKey);

      if (!sessionId) {
        // Generate unique ID: timestamp + random string
        sessionId =
          Date.now().toString(36) + Math.random().toString(36).substr(2, 9);
        localStorage.setItem(sessionKey, sessionId);
      }

      return sessionId;
    },

    /**
     * Toggle menu dropdown
     */
    toggleMenu: function () {
      var isOpen = this.$menuDropdown.attr("data-state") === "open";
      if (isOpen) {
        this.closeMenu();
      } else {
        this.openMenu();
      }
    },

    /**
     * Open menu dropdown
     */
    openMenu: function () {
      this.$menuDropdown.attr("data-state", "open");
      this.$menuTrigger.attr("aria-expanded", "true");
    },

    /**
     * Close menu dropdown
     */
    closeMenu: function () {
      this.$menuDropdown.attr("data-state", "closed");
      this.$menuTrigger.attr("aria-expanded", "false");
    },

    /**
     * Toggle expand/collapse chat (floating widget only)
     */
    toggleExpandChat: function () {
      var $popup = this.$wrapper.closest(".listeo-floating-chat-popup");
      if ($popup.length) {
        $popup.toggleClass("is-expanded");
        var isExpanded = $popup.hasClass("is-expanded");
        // Remember preference in localStorage
        localStorage.setItem("listeo_chat_expanded", isExpanded ? "1" : "0");
        debugLog("Chat expanded:", isExpanded);
      }
    },

    /**
     * Restore expand/collapse state from localStorage (floating widget only)
     */
    restoreExpandState: function () {
      var $popup = this.$wrapper.closest(".listeo-floating-chat-popup");
      if ($popup.length) {
        var savedState = localStorage.getItem("listeo_chat_expanded");
        if (savedState === "1") {
          $popup.addClass("is-expanded");
          debugLog("Restored expanded state from localStorage");
        }
      }
    },

    /**
     * Clear conversation
     */
    clearConversation: function () {
      // Clear localStorage
      localStorage.removeItem(this.storageKey);

      // Clear loaded listing context
      this.loadedListing = null;
      this.clearLoadedListingFromStorage();

      // Clear loaded product context
      this.loadedProduct = null;
      this.clearLoadedProductFromStorage();

      // Clear conversation history
      this.conversationHistory = [];

      // Clear messages and show welcome (keep session ID for analytics continuity)
      this.$messages.empty();
      this.addMessage("system", listeoAiChatConfig.strings.welcomeMessage, null, true);

      // Re-detect and add listing button if on listing page
      this.detectAndAddListingButton();

      // Re-detect and add product button if on product page
      this.detectAndAddProductButton();

      // Show quick buttons again after clearing conversation (if hidden)
      if (listeoAiChatConfig.quickButtonsVisibility === "hide_after_first") {
        var $quickBtns = this.$wrapper.find(".listeo-ai-chat-quick-buttons");
        $quickBtns.find(".listeo-ai-quick-btn").show();
        $quickBtns.show();
        debugLog("[Quick Buttons] Restored after conversation clear");
      }

      debugLog("Conversation cleared (session ID preserved)");
    },

    /**
     * Check rate limits (client-side using localStorage)
     * Returns {allowed: boolean, message: string}
     */
    checkRateLimit: function () {
      var now = Date.now();
      var timestamps = this.getMessageTimestamps();

      // Clean up old timestamps (older than 24 hours)
      timestamps = timestamps.filter(function (ts) {
        return now - ts < 86400000; // 24 hours in ms
      });

      // Check Tier 1: X messages per minute
      var tier1Window = this.rateLimits.tier1.window * 1000; // Convert to ms
      var tier1Count = timestamps.filter(function (ts) {
        return now - ts < tier1Window;
      }).length;

      if (tier1Count >= this.rateLimits.tier1.limit) {
        var tier1Wait = Math.ceil(
          (timestamps[timestamps.length - tier1Count] + tier1Window - now) /
            1000,
        );
        return {
          allowed: false,
          tier: "tier1",
          message: this.getRateLimitMessage(
            "minute",
            this.rateLimits.tier1.limit,
            tier1Wait,
          ),
        };
      }

      // Check Tier 2: X messages per 15 minutes
      var tier2Window = this.rateLimits.tier2.window * 1000;
      var tier2Count = timestamps.filter(function (ts) {
        return now - ts < tier2Window;
      }).length;

      if (tier2Count >= this.rateLimits.tier2.limit) {
        var tier2Wait = Math.ceil(
          (timestamps[timestamps.length - tier2Count] + tier2Window - now) /
            1000,
        );
        return {
          allowed: false,
          tier: "tier2",
          message: this.getRateLimitMessage(
            "15 minutes",
            this.rateLimits.tier2.limit,
            tier2Wait,
          ),
        };
      }

      // Check Tier 3: X messages per day
      var tier3Window = this.rateLimits.tier3.window * 1000;
      var tier3Count = timestamps.filter(function (ts) {
        return now - ts < tier3Window;
      }).length;

      if (tier3Count >= this.rateLimits.tier3.limit) {
        var tier3Wait = Math.ceil(
          (timestamps[timestamps.length - tier3Count] + tier3Window - now) /
            1000,
        );
        return {
          allowed: false,
          tier: "tier3",
          message: this.getRateLimitMessage(
            "day",
            this.rateLimits.tier3.limit,
            tier3Wait,
          ),
        };
      }

      // Save cleaned timestamps
      this.saveMessageTimestamps(timestamps);

      return { allowed: true };
    },

    /**
     * Record message timestamp
     */
    recordMessage: function () {
      var timestamps = this.getMessageTimestamps();
      timestamps.push(Date.now());
      this.saveMessageTimestamps(timestamps);
    },

    /**
     * Get message timestamps from localStorage
     */
    getMessageTimestamps: function () {
      try {
        var data = localStorage.getItem(this.rateLimitStorageKey);
        if (!data) return [];

        var parsed = JSON.parse(data);
        return Array.isArray(parsed) ? parsed : [];
      } catch (e) {
        debugError("[Rate Limit] Failed to parse timestamps:", e);
        return [];
      }
    },

    /**
     * Save message timestamps to localStorage
     */
    saveMessageTimestamps: function (timestamps) {
      try {
        localStorage.setItem(
          this.rateLimitStorageKey,
          JSON.stringify(timestamps),
        );
      } catch (e) {
        debugError("[Rate Limit] Failed to save timestamps:", e);
      }
    },

    /**
     * Get human-readable rate limit error message
     */
    getRateLimitMessage: function (period, limit, waitSeconds) {
      var waitText = "";
      if (waitSeconds >= 3600) {
        // Hours for daily limits
        var hours = Math.ceil(waitSeconds / 3600);
        var hourText =
          hours > 1
            ? listeoAiChatConfig.strings.hours
            : listeoAiChatConfig.strings.hour;
        waitText = hours + " " + hourText;
      } else if (waitSeconds >= 60) {
        // Minutes
        var minutes = Math.ceil(waitSeconds / 60);
        var minuteText =
          minutes > 1
            ? listeoAiChatConfig.strings.minutes
            : listeoAiChatConfig.strings.minute;
        waitText = minutes + " " + minuteText;
      } else {
        // Seconds
        var secondText =
          waitSeconds > 1
            ? listeoAiChatConfig.strings.seconds
            : listeoAiChatConfig.strings.second;
        waitText = waitSeconds + " " + secondText;
      }

      return (
        "⏱️ " +
        listeoAiChatConfig.strings.rateLimitPrefix +
        " " +
        limit +
        " " +
        listeoAiChatConfig.strings.rateLimitSuffix +
        " " +
        period +
        ". " +
        listeoAiChatConfig.strings.rateLimitWait +
        " " +
        waitText +
        " " +
        listeoAiChatConfig.strings.rateLimitBeforeTrying
      );
    },

    /**
     * Detect if we're on a single listing page and add "Talk about X" button
     */
    detectAndAddListingButton: function () {
      var self = this;

      // Check if we're on a single listing page
      var listingId = this.getCurrentListingId();
      if (!listingId) {
        debugLog("[Listing Context] Not on a listing page, skipping button");
        return;
      }

      debugLog("[Listing Context] Detected listing page, ID:", listingId);

      // Check if this listing is already loaded
      if (this.loadedListing && this.loadedListing.id === listingId) {
        debugLog(
          "[Listing Context] This listing is already loaded, no button needed",
        );
        return; // Don't show button if already loaded
      }

      // Get listing title from page
      var listingTitle = this.getCurrentListingTitle();

      // Build button HTML (only add title if found)
      var buttonHtml = "";
      if (listingTitle) {
        buttonHtml +=
          '<h4 class="listeo-ai-listing-context-title">' +
          listingTitle +
          "</h4>";
      }

      // Only disable button if config hasn't loaded yet (config is now loaded synchronously before this)
      var disabledAttr = this.configLoaded ? "" : " disabled";
      buttonHtml +=
        '<button class="listeo-ai-load-listing-btn" data-listing-id="' +
        listingId +
        '"' + disabledAttr + '>' +
        '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">' +
        '<path d="M19 3H5C3.89 3 3 3.89 3 5V19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.89 20.1 3 19 3ZM19 19H5V5H19V19Z" fill="currentColor"/>' +
        '<path d="M7 7H17V9H7V7ZM7 11H17V13H7V11ZM7 15H14V17H7V15Z" fill="currentColor"/>' +
        "</svg>" +
        '<span class="btn-text">' +
        listeoAiChatConfig.strings.talkAboutListing +
        "</span>" +
        "</button>";

      // Add as a special message
      this.addMessage("listing-action", buttonHtml, "listing-context-btn");

      // Button click handler (use event delegation since button is inside messages)
      this.$messages.on("click", ".listeo-ai-load-listing-btn", function () {
        var $btn = $(this);
        // Load listing
        self.loadListingContext(listingId, self.getCurrentListingTitle());
      });
    },

    /**
     * Get current listing ID from page
     */
    getCurrentListingId: function () {
      var bodyClasses = $("body").attr("class") || "";

      // Check if it's a single listing page
      var isSingleListing = bodyClasses.indexOf("single-listing") !== -1;

      // Make sure we're NOT on admin or other pages
      var isAdminPage =
        bodyClasses.indexOf("wp-admin") !== -1 ||
        (bodyClasses.indexOf("admin-bar") !== -1 &&
          window.location.href.indexOf("/wp-admin/") !== -1);

      if (!isSingleListing || isAdminPage) {
        return null; // Not a single listing page or is admin
      }

      // Method 1: Check body class (e.g., postid-1234)
      var postIdMatch = bodyClasses.match(/postid-(\d+)/);
      if (postIdMatch) {
        var postId = parseInt(postIdMatch[1]);
        return postId;
      }

      // Method 2: Check for data attribute on listing container
      var $listingContainer = $(
        ".single-listing-wrapper, .listing-single-container, [data-listing-id]",
      );
      if ($listingContainer.length) {
        var dataId = $listingContainer.data("listing-id");
        if (dataId) {
          return parseInt(dataId);
        }
      }

      // Method 3: Check global WordPress JS object (if available)
      if (typeof listeo_core !== "undefined" && listeo_core.post_id) {
        return parseInt(listeo_core.post_id);
      }

      return null;
    },

    /**
     * Get current listing title from page
     */
    getCurrentListingTitle: function () {
      // Get from listing titlebar only
      var $titlebar = $(".listing-titlebar-title h1");
      if ($titlebar.length) {
        return $titlebar.text().trim();
      }

      // No title found - return empty string
      return "";
    },

    /**
     * Load listing context into chat
     */
    loadListingContext: function (listingId, listingTitle) {
      var self = this;
      var $btn = $(".listeo-ai-load-listing-btn");

      // Show loading state
      $btn.addClass("loading").prop("disabled", true);
      $btn.find(".btn-text").text(listeoAiChatConfig.strings.loadingButton);

      debugLog("[Listing Context] Fetching listing details for ID:", listingId);

      // Use Listeo-specific endpoint if available (config is now guaranteed to be loaded)
      var endpoint = this.chatConfig.listeo_available
        ? "/listeo-listing-details"
        : "/get-content";
      var param = this.chatConfig.listeo_available
        ? { listing_id: listingId }
        : { post_id: listingId };

      debugLog(
        "[loadListingContext] Using endpoint:",
        endpoint,
        "(Listeo available:",
        this.chatConfig.listeo_available + ")",
      );

      // Fetch listing details via API
      $.ajax({
        url: listeoAiChatConfig.apiBase + endpoint,
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify(param),
        success: function (response) {
          if (response.success) {
            // IMPORTANT: Wipe any previous listing context
            // This prevents context pollution when loading multiple listings
            self.loadedListing = {
              id: listingId,
              title: response.title,
              url: response.url,
              content: response.structured_content,
            };

            // Persist to localStorage
            self.saveLoadedListingToStorage();

            debugLog(
              "[Listing Context] Loaded listing:",
              self.loadedListing.title,
            );

            // Remove the button (hide it)
            $btn.closest(".listeo-ai-chat-message").fadeOut(300, function () {
              $(this).remove();
            });

            // Add confirmation message with SVG icon
            var listingIcon =
              '<svg class="context-loaded-icon" width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M19 3H5C3.89 3 3 3.89 3 5V19C3 20.1 3.89 21 5 21H19C20.1 21 21 20.1 21 19V5C21 3.89 20.1 3 19 3ZM19 19H5V5H19V19Z" fill="currentColor"/><path d="M7 7H17V9H7V7ZM7 11H17V13H7V11ZM7 15H14V17H7V15Z" fill="currentColor"/></svg>';
            self.addMessage(
              "system",
              listingIcon +
                " <strong>" +
                listeoAiChatConfig.strings.listingContextLoaded +
                '</strong> <a href="' +
                response.url +
                '">' +
                response.title +
                "</a>",
            );
          } else {
            self.handleListingLoadError(
              listeoAiChatConfig.strings.failedLoadDetails,
            );
            $btn.removeClass("loading").prop("disabled", false);
            $btn
              .find(".btn-text")
              .text(listeoAiChatConfig.strings.talkAboutListing);
          }
        },
        error: function (xhr) {
          debugError("[Listing Context] Error loading listing:", xhr);
          self.handleListingLoadError(
            listeoAiChatConfig.strings.errorLoadingListing,
          );
          $btn.removeClass("loading").prop("disabled", false);
          $btn
            .find(".btn-text")
            .text(listeoAiChatConfig.strings.talkAboutListing);
        },
      });
    },

    /**
     * Handle listing load error
     */
    handleListingLoadError: function (message) {
      this.addMessage("system", "⚠️ " + message);
    },

    /**
     * Save loaded listing to localStorage
     */
    saveLoadedListingToStorage: function () {
      try {
        if (this.loadedListing) {
          localStorage.setItem(
            this.loadedListingStorageKey,
            JSON.stringify(this.loadedListing),
          );
          debugLog(
            "[Listing Context] Saved to localStorage:",
            this.loadedListing.title,
          );
        }
      } catch (e) {
        debugError("[Listing Context] Failed to save to localStorage:", e);
      }
    },

    /**
     * Load persisted listing context from localStorage
     */
    loadPersistedListingContext: function () {
      try {
        var data = localStorage.getItem(this.loadedListingStorageKey);
        if (data) {
          this.loadedListing = JSON.parse(data);
          debugLog(
            "[Listing Context] Restored from localStorage:",
            this.loadedListing.title,
          );
        }
      } catch (e) {
        debugError("[Listing Context] Failed to load from localStorage:", e);
      }
    },

    /**
     * Clear loaded listing from localStorage
     */
    clearLoadedListingFromStorage: function () {
      try {
        localStorage.removeItem(this.loadedListingStorageKey);
        debugLog("[Listing Context] Cleared from localStorage");
      } catch (e) {
        debugError("[Listing Context] Failed to clear from localStorage:", e);
      }
    },

    // ========================================
    // PRODUCT CONTEXT FUNCTIONS (WooCommerce)
    // ========================================

    /**
     * Detect single product page and add "Talk about this product" button
     */
    detectAndAddProductButton: function () {
      var self = this;

      // Get product ID from page
      var productId = this.getCurrentProductId();

      if (!productId) {
        debugLog("[Product Context] Not on a product page, skipping button");
        return;
      }

      debugLog("[Product Context] Detected product page, ID:", productId);

      // Check if this product is already loaded
      if (this.loadedProduct && this.loadedProduct.id === productId) {
        debugLog(
          "[Product Context] This product is already loaded, no button needed",
        );
        return;
      }

      // Get product title from page
      var productTitle = this.getCurrentProductTitle();

      // Build button HTML
      var buttonHtml = "";
      if (productTitle) {
        buttonHtml +=
          '<h4 class="listeo-ai-product-context-title">' +
          productTitle +
          "</h4>";
      }

      // Only disable button if config hasn't loaded yet (config is now loaded synchronously before this)
      var disabledAttr = this.configLoaded ? "" : " disabled";
      buttonHtml +=
        '<button class="listeo-ai-load-product-btn" data-product-id="' +
        productId +
        '"' + disabledAttr + '>' +
        '<svg width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">' +
        '<path d="M7 18c-1.1 0-1.99.9-1.99 2S5.9 22 7 22s2-.9 2-2-.9-2-2-2zM1 2v2h2l3.6 7.59-1.35 2.45c-.16.28-.25.61-.25.96 0 1.1.9 2 2 2h12v-2H7.42c-.14 0-.25-.11-.25-.25l.03-.12.9-1.63h7.45c.75 0 1.41-.41 1.75-1.03l3.58-6.49c.08-.14.12-.31.12-.48 0-.55-.45-1-1-1H5.21l-.94-2H1zm16 16c-1.1 0-1.99.9-1.99 2s.89 2 1.99 2 2-.9 2-2-.9-2-2-2z" fill="currentColor"/>' +
        "</svg>" +
        '<span class="btn-text">' +
        listeoAiChatConfig.strings.talkAboutProduct +
        "</span>" +
        "</button>";

      // Add as a special message
      this.addMessage("product-action", buttonHtml, "product-context-btn");

      // Button click handler
      this.$messages.on("click", ".listeo-ai-load-product-btn", function () {
        var $btn = $(this);
        self.loadProductContext(productId, self.getCurrentProductTitle());
      });
    },

    /**
     * Get current product ID from page (WooCommerce)
     */
    getCurrentProductId: function () {
      var bodyClasses = $("body").attr("class") || "";

      // Check if it's a single product page
      var isSingleProduct = bodyClasses.indexOf("single-product") !== -1;

      // Make sure we're NOT on admin
      var isAdminPage =
        bodyClasses.indexOf("wp-admin") !== -1 ||
        (bodyClasses.indexOf("admin-bar") !== -1 &&
          window.location.href.indexOf("/wp-admin/") !== -1);

      if (!isSingleProduct || isAdminPage) {
        return null;
      }

      // Method 1: Check body class (e.g., postid-1234)
      var postIdMatch = bodyClasses.match(/postid-(\d+)/);
      if (postIdMatch) {
        return parseInt(postIdMatch[1]);
      }

      // Method 2: Check for data attribute on product container
      var $productContainer = $("[data-product_id], .product[data-id]");
      if ($productContainer.length) {
        var dataId =
          $productContainer.data("product_id") || $productContainer.data("id");
        if (dataId) {
          return parseInt(dataId);
        }
      }

      // Method 3: Check for hidden input with product ID
      var $productIdInput = $(
        'input[name="product_id"], input[name="add-to-cart"]',
      );
      if ($productIdInput.length) {
        var inputVal = $productIdInput.val();
        if (inputVal) {
          return parseInt(inputVal);
        }
      }

      return null;
    },

    /**
     * Get current product title from page
     */
    getCurrentProductTitle: function () {
      // WooCommerce product title
      var $productTitle = $(
        ".product_title, .woocommerce-product-title, h1.entry-title",
      );
      if ($productTitle.length) {
        return $productTitle.first().text().trim();
      }

      return "";
    },

    /**
     * Load product context into chat
     */
    loadProductContext: function (productId, productTitle) {
      var self = this;
      var $btn = $(".listeo-ai-load-product-btn");

      // Show loading state
      $btn.addClass("loading").prop("disabled", true);
      $btn.find(".btn-text").text(listeoAiChatConfig.strings.loadingButton);

      debugLog("[Product Context] Fetching product details for ID:", productId);

      // Use get-content endpoint for products
      $.ajax({
        url: listeoAiChatConfig.apiBase + "/get-content",
        method: "POST",
        contentType: "application/json",
        data: JSON.stringify({ post_id: productId }),
        success: function (response) {
          if (response.success) {
            // Store product context
            self.loadedProduct = {
              id: productId,
              title: response.title,
              url: response.url,
              content: response.structured_content,
            };

            // Persist to localStorage
            self.saveLoadedProductToStorage();

            debugLog(
              "[Product Context] Loaded product:",
              self.loadedProduct.title,
            );

            // Remove the button
            $btn.closest(".listeo-ai-chat-message").fadeOut(300, function () {
              $(this).remove();
            });

            // Add confirmation message with SVG icon
            var productIcon =
              '<svg class="context-loaded-icon" width="16" height="16" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg"><path d="M7 18c-1.1 0-1.99.9-1.99 2S5.9 22 7 22s2-.9 2-2-.9-2-2-2zM1 2v2h2l3.6 7.59-1.35 2.45c-.16.28-.25.61-.25.96 0 1.1.9 2 2 2h12v-2H7.42c-.14 0-.25-.11-.25-.25l.03-.12.9-1.63h7.45c.75 0 1.41-.41 1.75-1.03l3.58-6.49c.08-.14.12-.31.12-.48 0-.55-.45-1-1-1H5.21l-.94-2H1zm16 16c-1.1 0-1.99.9-1.99 2s.89 2 1.99 2 2-.9 2-2-.9-2-2-2z" fill="currentColor"/></svg>';
            self.addMessage(
              "system",
              productIcon +
                " <strong>" +
                listeoAiChatConfig.strings.productContextLoaded +
                '</strong> <a href="' +
                response.url +
                '">' +
                response.title +
                "</a>",
            );
          } else {
            self.handleProductLoadError(
              listeoAiChatConfig.strings.failedLoadProductDetails,
            );
            $btn.removeClass("loading").prop("disabled", false);
            $btn
              .find(".btn-text")
              .text(listeoAiChatConfig.strings.talkAboutProduct);
          }
        },
        error: function (xhr) {
          debugError("[Product Context] Error loading product:", xhr);
          self.handleProductLoadError(
            listeoAiChatConfig.strings.errorLoadingProduct,
          );
          $btn.removeClass("loading").prop("disabled", false);
          $btn
            .find(".btn-text")
            .text(listeoAiChatConfig.strings.talkAboutProduct);
        },
      });
    },

    /**
     * Handle product load error
     */
    handleProductLoadError: function (message) {
      this.addMessage("system", "⚠️ " + message);
    },

    /**
     * Save loaded product to localStorage
     */
    saveLoadedProductToStorage: function () {
      try {
        if (this.loadedProduct) {
          localStorage.setItem(
            this.loadedProductStorageKey,
            JSON.stringify(this.loadedProduct),
          );
          debugLog(
            "[Product Context] Saved to localStorage:",
            this.loadedProduct.title,
          );
        }
      } catch (e) {
        debugError("[Product Context] Failed to save to localStorage:", e);
      }
    },

    /**
     * Load persisted product context from localStorage
     */
    loadPersistedProductContext: function () {
      try {
        var data = localStorage.getItem(this.loadedProductStorageKey);
        if (data) {
          this.loadedProduct = JSON.parse(data);
          debugLog(
            "[Product Context] Restored from localStorage:",
            this.loadedProduct.title,
          );
        }
      } catch (e) {
        debugError("[Product Context] Failed to load from localStorage:", e);
      }
    },

    /**
     * Clear loaded product from localStorage
     */
    clearLoadedProductFromStorage: function () {
      try {
        localStorage.removeItem(this.loadedProductStorageKey);
        debugLog("[Product Context] Cleared from localStorage");
      } catch (e) {
        debugError("[Product Context] Failed to clear from localStorage:", e);
      }
    },

    /**
     * Handle image file selection
     */
    handleImageSelect: function (file) {
      var self = this;

      // Allowed MIME types (include image/jpg as fallback - some browsers report it instead of image/jpeg)
      var allowedTypes = ["image/jpeg", "image/jpg", "image/png", "image/gif", "image/webp"];

      // Validate file type
      if (allowedTypes.indexOf(file.type) === -1) {
        debugLog("[Image] Invalid file type:", file.type);
        this.addMessage("system", listeoAiChatConfig.strings.imageInvalidFormat || "Invalid image format. Allowed: JPEG, PNG, GIF, WebP.");
        return;
      }

      // Validate file size (max 4MB)
      var maxSize = 4 * 1024 * 1024;
      if (file.size > maxSize) {
        debugLog("[Image] File too large:", file.size);
        this.addMessage("system", listeoAiChatConfig.strings.imageTooLarge || "Image is too large. Maximum size is 4MB.");
        return;
      }

      // Read file and check resolution
      var reader = new FileReader();
      reader.onload = function (e) {
        var base64 = e.target.result;

        // Create image to check dimensions
        var img = new Image();
        img.onload = function () {
          var maxDimension = 3000;

          // Check resolution (max 3000x3000)
          if (img.width > maxDimension || img.height > maxDimension) {
            debugLog("[Image] Resolution too large:", img.width + "x" + img.height);
            self.addMessage("system", listeoAiChatConfig.strings.imageResolutionTooLarge || "Image resolution is too large. Maximum is 3000x3000 pixels.");
            return;
          }

          // Store the attached image
          self.attachedImage = {
            base64: base64,
            mimeType: file.type,
            name: file.name,
            width: img.width,
            height: img.height,
          };

          // Update button state
          self.$imageBtn.addClass("has-image");

          debugLog("[Image] Image attached:", {
            name: file.name,
            type: file.type,
            size: file.size,
            dimensions: img.width + "x" + img.height,
          });
        };
        img.src = base64;
      };
      reader.readAsDataURL(file);
    },

    /**
     * Clear attached image
     */
    clearAttachedImage: function () {
      this.attachedImage = null;
      this.$imageBtn.removeClass("has-image");
      if (this.$imageInput.length) {
        this.$imageInput.val("");
      }
      debugLog("[Image] Image cleared");
    },

    /**
     * Build message content with optional image
     * Returns array format for multimodal or string for text-only
     */
    buildMessageContent: function (text) {
      if (!this.attachedImage) {
        return text;
      }

      // Build multimodal content array (OpenAI format)
      var content = [];

      // Add image first
      content.push({
        type: "image_url",
        image_url: {
          url: this.attachedImage.base64,
          detail: "auto",
        },
      });

      // Add text if provided (no default - vision models understand images without prompts)
      if (text && text.trim()) {
        content.push({
          type: "text",
          text: text,
        });
      }

      return content;
    },

    /**
     * Get display text for user message (for UI display)
     */
    getUserMessageDisplay: function (text) {
      var html = "";

      if (this.attachedImage) {
        html += '<img src="' + this.attachedImage.base64 + '" alt="Attached image" class="listeo-ai-chat-user-image" />';
      }

      if (text && text.trim()) {
        html += '<span>' + this.escapeHtml(text) + '</span>';
      }

      return html || listeoAiChatConfig.strings.imageAttached || "[Image attached]";
    },

    /**
     * Escape HTML special characters
     */
    escapeHtml: function (text) {
      var div = document.createElement("div");
      div.textContent = text;
      return div.innerHTML;
    },

    /**
     * Extract text from message content
     * Handles both string and multimodal (array) content
     * Used for original_question in search tools (don't send images to search)
     */
    extractTextFromMessage: function (content) {
      if (typeof content === "string") {
        return content;
      }

      if (Array.isArray(content)) {
        var textParts = [];
        for (var i = 0; i < content.length; i++) {
          var part = content[i];
          if (part.type === "text" && part.text) {
            textParts.push(part.text);
          }
        }
        return textParts.join(" ");
      }

      return "";
    },
  };
})(jQuery);
