/**
 * AI Chat Search - Admin UI Module
 *
 * Handles general UI interactions: sticky footers, collapsible sections,
 * shortcode generator, visual radio cards, translation installer.
 *
 * @package AI_Chat_Search
 * @since 1.0.0
 */

(function($) {
    'use strict';

    var i18n = window.listeo_ai_search_i18n || {};

    /**
     * Position sticky footer to match form width
     */
    function initStickyFooters() {
        function positionStickyFooter($form) {
            var $footer = $form.find('.airs-sticky-footer');
            if ($form.length && $footer.length) {
                var formOffset = $form.offset();
                var formWidth = $form.outerWidth();
                $footer.css({
                    left: formOffset.left + 'px',
                    width: formWidth + 'px'
                });
            }
        }

        // Position all sticky footers
        function positionAllFooters() {
            $('#settings-form, #ai-chat-settings-form').each(function() {
                positionStickyFooter($(this));
            });
        }

        positionAllFooters();
        $(window).on('resize', positionAllFooters);
    }

    /**
     * Initialize collapsible sections
     */
    function initCollapsibleSections() {
        // Toggle handler
        $('.airs-collapsible-header').on('click', function() {
            var $header = $(this);
            var $content = $header.next('.airs-collapsible-content');
            var sectionId = $header.data('section');

            $header.toggleClass('is-open');
            $content.toggleClass('is-open');

            // Save state to localStorage
            if (sectionId) {
                var isOpen = $header.hasClass('is-open');
                localStorage.setItem('airs_collapse_' + sectionId, isOpen ? 'open' : 'closed');
            }
        });

        // Restore saved states
        $('.airs-collapsible-header').each(function() {
            var $header = $(this);
            var sectionId = $header.data('section');
            if (sectionId) {
                var savedState = localStorage.getItem('airs_collapse_' + sectionId);
                if (savedState === 'open') {
                    $header.addClass('is-open');
                    $header.next('.airs-collapsible-content').addClass('is-open');
                }
            }
        });
    }

    /**
     * Initialize shortcode generator
     */
    function initShortcodeGenerator() {
        var $postTypes = $('.shortcode-post-type');
        var $placeholder = $('#shortcode-placeholder');
        var $limit = $('#shortcode-limit');
        var $output = $('#generated-shortcode');
        var $copyBtn = $('#copy-shortcode-btn');

        // Skip if elements don't exist
        if (!$output.length) return;

        var defaultPlaceholder = i18n.searchPlaceholder || 'Search anything...';

        function updateShortcode() {
            var postTypes = [];
            $postTypes.filter(':checked').each(function() {
                postTypes.push($(this).val());
            });

            var placeholder = $placeholder.val().trim();
            var limit = parseInt($limit.val()) || 10;

            var shortcode = '[ai_search_field';

            if (postTypes.length > 0) {
                shortcode += ' post_types="' + postTypes.join(',') + '"';
            }

            if (placeholder && placeholder !== defaultPlaceholder) {
                shortcode += ' placeholder="' + placeholder + '"';
            }

            if (limit !== 10) {
                shortcode += ' limit="' + limit + '"';
            }

            shortcode += ']';

            $output.val(shortcode);
        }

        // Update shortcode when options change
        $postTypes.add($placeholder).add($limit).on('change input', updateShortcode);

        // Initial generation
        updateShortcode();

        // Copy to clipboard
        $copyBtn.on('click', function() {
            $output.select();
            document.execCommand('copy');

            var $btn = $(this);
            var originalHtml = $btn.html();
            $btn.html('<svg width="14" height="14" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg" style="margin-right: 5px;"><path d="M20 6L9 17l-5-5" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/></svg>' + (i18n.copied || 'Copied!'));

            setTimeout(function() {
                $btn.html(originalHtml);
            }, 2000);
        });
    }

    /**
     * Initialize visual radio card selection
     */
    function initVisualRadioCards() {
        $('.airs-visual-radio-card input[type="radio"]').on('change', function() {
            var $group = $(this).closest('.airs-visual-radio-group');
            $group.find('.airs-visual-radio-card').removeClass('selected');
            $(this).closest('.airs-visual-radio-card').addClass('selected');
        });
    }

    /**
     * Initialize theme toggle (light/system/dark)
     */
    function initThemeToggle() {
        var $toggle = $('.airs-theme-toggle');
        if (!$toggle.length) return;

        var $buttons = $toggle.find('.airs-theme-btn');
        var $hiddenInput = $toggle.find('input[type="hidden"]');

        $buttons.on('click', function(e) {
            e.preventDefault();
            var $btn = $(this);
            var value = $btn.data('value');

            // Update active state
            $buttons.removeClass('active');
            $btn.addClass('active');

            // Update hidden input
            $hiddenInput.val(value);
        });
    }

    /**
     * Initialize translation installer
     */
    function initTranslationInstaller() {
        var $select = $('#ai_translation_locale');
        var $installBtn = $('#ai_install_translation');
        var $status = $('#ai_translation_status');

        // Skip if elements don't exist
        if (!$select.length) return;

        var config = window.listeo_ai_search_ajax || {};
        var translationNonce = config.translation_nonce || '';

        $select.on('change', function() {
            var locale = $(this).val();
            $status.html('');
            $installBtn.prop('disabled', true);

            if (!locale) return;

            if (locale.indexOf('en_') === 0 || locale === 'en') {
                $status.html('<span style="color:#0073aa;">' + (i18n.englishDefault || 'English is default, no install needed.') + '</span>');
                return;
            }

            $status.html('<span style="color:#666;">' + (i18n.checking || 'Checking...') + '</span>');

            $.post(window.ajaxurl, {
                action: 'ai_chat_search_check_translation',
                nonce: translationNonce,
                locale: locale
            }, function(response) {
                if (response.success) {
                    if (response.data.installed) {
                        $status.html('<span style="color:#46b450;">' + (i18n.translationInstalled || 'Translation already installed!') + '</span>');
                    } else if (response.data.available) {
                        $status.html('<span style="color:#0073aa;">' + (i18n.translationAvailable || 'Translation available. Click Install.') + '</span>');
                        $installBtn.prop('disabled', false);
                    } else {
                        $status.html('<span style="color:#dc3232;">' + (i18n.translationNotAvailable || 'Translation not available for this locale.') + '</span>');
                    }
                } else {
                    $status.html('<span style="color:#dc3232;">' + (response.data.message || i18n.checkFailed || 'Check failed.') + '</span>');
                }
            }).fail(function() {
                $status.html('<span style="color:#dc3232;">' + (i18n.connectionFailed || 'Connection failed.') + '</span>');
            });
        });

        $installBtn.on('click', function() {
            var locale = $select.val();
            if (!locale) return;

            $installBtn.prop('disabled', true);
            $status.html('<span style="color:#666;">' + (i18n.installing || 'Installing...') + '</span>');

            $.post(window.ajaxurl, {
                action: 'ai_chat_search_install_translation',
                nonce: translationNonce,
                locale: locale
            }, function(response) {
                if (response.success) {
                    $status.html('<span style="color:#46b450;">' + (i18n.translationInstalledSuccess || 'Translation installed successfully!') + '</span>');
                } else {
                    $status.html('<span style="color:#dc3232;">' + (response.data.message || i18n.installFailed || 'Installation failed.') + '</span>');
                    $installBtn.prop('disabled', false);
                }
            }).fail(function() {
                $status.html('<span style="color:#dc3232;">' + (i18n.connectionFailed || 'Connection failed.') + '</span>');
                $installBtn.prop('disabled', false);
            });
        });
    }

    /**
     * Initialize quality slider (min match percentage)
     */
    function initQualitySlider() {
        var $qualitySlider = $('#listeo_ai_search_min_match_percentage');
        if (!$qualitySlider.length) return;

        function updateQualitySlider($slider) {
            var value = parseInt($slider.val());
            var $badge = $('#min-match-badge');
            var $label = $('#min-match-label');
            var $display = $('#min-match-display');
            var $track = $('#min-match-track');

            var qualityClass = '';
            var labelText = '';
            var qualityClasses = 'quality-very-low quality-low quality-below-avg quality-balanced quality-high quality-very-high';

            // Update badge text
            $badge.text(value + '%');

            // Determine quality class and label based on value
            if (value < 20) {
                qualityClass = 'quality-very-low';
                labelText = i18n.qualityVeryLow || 'Loose - many results, lower relevance';
            } else if (value < 40) {
                qualityClass = 'quality-low';
                labelText = i18n.qualityLow || 'Broad - more results, some may be less relevant';
            } else if (value < 60) {
                qualityClass = 'quality-balanced';
                labelText = i18n.qualityBalanced || 'Balanced - good mix of quantity and quality';
            } else if (value < 80) {
                qualityClass = 'quality-high';
                labelText = i18n.qualityHigh || 'Quality focused - pay attention because <strong>you might start getting little results</strong>';
            } else {
                qualityClass = 'quality-very-high';
                labelText = i18n.qualityVeryHigh || 'Very strict \u2014 <strong>you might get little to no results</strong>';
            }

            // Update classes on all elements
            $badge.removeClass(qualityClasses).addClass(qualityClass);
            $display.removeClass(qualityClasses).addClass(qualityClass);
            $slider.removeClass(qualityClasses).addClass(qualityClass);
            $track.removeClass(qualityClasses).addClass(qualityClass);

            $label.html(labelText);

            // Update track glow position
            var percentage = (value / 100) * 100;
            $track.css('--slider-glow-position', percentage + '%');
        }

        updateQualitySlider($qualitySlider);
        $qualitySlider.on('input change', function() {
            updateQualitySlider($(this));
        });
    }

    /**
     * Initialize checkbox card styling
     */
    function initCheckboxCards() {
        // Hover effect
        $('.airs-checkbox-card').hover(
            function() { $(this).css('border-color', '#0073ee'); },
            function() {
                var $checkbox = $(this).find('.shortcode-post-type');
                if (!$checkbox.is(':checked')) {
                    $(this).css('border-color', '#e0e0e0');
                }
            }
        );

        // Checked state
        $('.shortcode-post-type').on('change', function() {
            var $card = $(this).closest('.airs-checkbox-card');
            if ($(this).is(':checked')) {
                $card.css({'background': '#f0f7ff', 'border-color': '#0073ee'});
            } else {
                $card.css({'background': '#fff', 'border-color': '#e0e0e0'});
            }
        }).trigger('change');
    }

    /**
     * Initialize toggleable cards with localStorage persistence
     */
    function initToggleableCards() {
        var STORAGE_KEY = 'airs_collapsed_cards';

        // Get collapsed state from localStorage
        function getCollapsedCards() {
            try {
                var stored = localStorage.getItem(STORAGE_KEY);
                return stored ? JSON.parse(stored) : {};
            } catch(e) {
                return {};
            }
        }

        // Save collapsed state to localStorage
        function saveCollapsedCards(state) {
            try {
                localStorage.setItem(STORAGE_KEY, JSON.stringify(state));
            } catch(e) {}
        }

        // Initialize: apply collapsed class, then add js-ready to disable early CSS
        var DEFAULTS = { 'database-management': true, 'semantic-search-field': true, 'developer-debug': true };
        var collapsedState = window.airsCollapsedCards || getCollapsedCards();
        for (var id in DEFAULTS) {
            if (!(id in collapsedState)) collapsedState[id] = DEFAULTS[id];
        }
        $('.airs-card-toggleable').each(function() {
            var $card = $(this);
            var toggleId = $card.data('toggle-id');
            if (toggleId && collapsedState[toggleId] === true) {
                $card.addClass('is-collapsed');
            }
            $card.addClass('js-ready');
        });

        // Handle click on card header
        $(document).on('click', '.airs-card-toggleable .airs-card-header', function(e) {
            if ($(e.target).is('button, input, select, a')) {
                return;
            }

            var $card = $(this).closest('.airs-card-toggleable');
            var toggleId = $card.data('toggle-id');

            $card.toggleClass('is-collapsed');

            if (toggleId) {
                var collapsedState = getCollapsedCards();
                collapsedState[toggleId] = $card.hasClass('is-collapsed');
                saveCollapsedCards(collapsedState);
            }
        });
    }

    /**
     * Initialize color picker fields
     */
    function initColorPicker() {
        if (!$.fn.wpColorPicker) return;

        $('.airs-color-picker').wpColorPicker();

        // Add Select/Apply button and hex input next to each swatch
        $('.airs-color-picker').each(function() {
            var $input = $(this);
            var $container = $input.closest('.wp-picker-container');
            var $swatchBtn = $container.find('.wp-color-result');
            var $holder = $container.find('.wp-picker-holder');

            // Select/Apply button
            if (!$container.find('.airs-color-action-btn').length) {
                var $actionBtn = $('<button type="button" class="airs-color-action-btn">Select</button>');
                $swatchBtn.after($actionBtn);

                $actionBtn.on('click', function(e) {
                    e.preventDefault();
                    e.stopPropagation();
                    var isOpen = $swatchBtn.attr('aria-expanded') === 'true';
                    $swatchBtn.trigger('click');
                    if (isOpen) {
                        $actionBtn.text('Select');
                    }
                });
            }

            // Hex input below the picker
            if (!$holder.find('.airs-hex-input').length) {
                var currentVal = $input.val() || $input.data('default-color') || '#000000';
                var $hexInput = $('<input type="text" class="airs-hex-input" maxlength="7" placeholder="#000000" />');
                $hexInput.val(currentVal);
                $holder.append($hexInput);

                // Hex input → picker
                $hexInput.on('input', function() {
                    var val = $(this).val();
                    if (/^#[0-9a-fA-F]{6}$/.test(val)) {
                        $input.wpColorPicker('color', val);
                    }
                });

                // Picker → hex input (on iris change)
                $input.wpColorPicker('option', 'change', function(event, ui) {
                    $hexInput.val(ui.color.toString());
                });
            }

            // Sync picker-open class, button text, and hex value
            function syncPickerState() {
                var isOpen = $swatchBtn.attr('aria-expanded') === 'true';
                $container.toggleClass('picker-open', isOpen);
                $container.find('.airs-color-action-btn').text(isOpen ? 'Apply' : 'Select');
                if (isOpen) {
                    $container.find('.airs-hex-input').val($input.val());
                }
            }

            $swatchBtn.on('click', function() {
                setTimeout(syncPickerState, 0);
            });

            // Catch all close paths (click outside, etc.)
            var observer = new MutationObserver(syncPickerState);
            observer.observe($swatchBtn[0], { attributes: true, attributeFilter: ['aria-expanded'] });
        });
    }

    /**
     * Initialize system prompt character counter
     */
    function initSystemPromptCounter() {
        var $textarea = $('#listeo_ai_chat_system_prompt');
        var $counter = $('#system-prompt-counter');

        if (!$textarea.length || !$counter.length) return;

        var maxLength = parseInt($textarea.data('max-length')) || 2000;

        function updateCounter() {
            var length = $textarea.val().length;
            $counter.text(length + ' / ' + maxLength + ' ' + (i18n.characters || 'characters'));
            var percentage = (length / maxLength) * 100;
            if (percentage > 90) {
                $counter.css('color', '#d63638');
            } else if (percentage > 75) {
                $counter.css('color', '#dba617');
            } else {
                $counter.css('color', '#666');
            }
        }

        $textarea.on('input', updateCounter);
        updateCounter();
    }

    /**
     * Initialize settings toggle handlers
     */
    function initSettingsToggles() {
        // Toggle terms notice text field visibility
        $('#listeo_ai_chat_terms_notice_enabled').on('change', function() {
            if ($(this).is(':checked')) {
                $('#listeo_ai_chat_terms_notice_text_wrapper').slideDown(300);
            } else {
                $('#listeo_ai_chat_terms_notice_text_wrapper').slideUp(300);
            }
        });

        // Toggle contact form examples field visibility
        $('#listeo_ai_contact_form_allow_ai_send').on('change', function() {
            if ($(this).is(':checked')) {
                $('#listeo_ai_contact_form_examples_wrapper').slideDown(300);
            } else {
                $('#listeo_ai_contact_form_examples_wrapper').slideUp(300);
            }
        });

        // Toggle quick buttons wrapper visibility
        $('#listeo_ai_chat_quick_buttons_enabled').on('change', function() {
            if ($(this).is(':checked')) {
                $('#listeo_ai_chat_quick_buttons_wrapper').slideDown(300);
            } else {
                $('#listeo_ai_chat_quick_buttons_wrapper').slideUp(300);
            }
        });
    }

    /**
     * Initialize quick buttons management (remove handler only - add needs PHP)
     */
    function initQuickButtonsRemove() {
        $(document).on('click', '.listeo-remove-quick-button', function() {
            var $container = $('#listeo-quick-buttons-container');
            if ($container.find('.listeo-quick-button-row').length > 1) {
                $(this).closest('.listeo-quick-button-row').remove();
                // Re-index remaining rows
                $container.find('.listeo-quick-button-row').each(function(i) {
                    $(this).find('input, select').each(function() {
                        var name = $(this).attr('name');
                        if (name) {
                            $(this).attr('name', name.replace(/\[\d+\]/, '[' + i + ']'));
                        }
                    });
                });
            }
        });
    }

    /**
     * Initialize modal handlers
     */
    function initModalHandlers() {
        // Open contact form configuration modal
        $(document).on('click', '.listeo-configure-contact-form', function() {
            $('#contact-form-config-modal').fadeIn(200);
        });

        // Close contact form modal
        $('#contact-form-modal-close, #contact-form-config-modal .airs-modal-overlay').on('click', function() {
            $('#contact-form-config-modal').fadeOut(200);
        });
    }

    /**
     * Initialize quick button type change handler
     */
    function initQuickButtonTypeChange() {
        $(document).on('change', '.listeo-quick-button-type', function() {
            var $row = $(this).closest('.listeo-quick-button-row');
            var $valueInput = $row.find('.listeo-quick-button-value');
            var $configureBtn = $row.find('.listeo-configure-contact-form');
            var type = $(this).val();

            if (type === 'contact') {
                $valueInput.hide().val('');
                $configureBtn.show();
            } else if (type === 'url') {
                $valueInput.show().attr('placeholder', i18n.placeholderUrl || 'https://example.com');
                $configureBtn.hide();
            } else {
                $valueInput.show().attr('placeholder', i18n.placeholderMessage || 'Message to send');
                $configureBtn.hide();
            }
        });
    }

    /**
     * Initialize contact form AJAX handlers (test email, save settings)
     */
    function initContactFormHandlers() {
        var config = window.listeo_ai_search_ajax || {};

        // Send test email
        $('#contact-form-test-email-btn').on('click', function() {
            var $btn = $(this);
            var $result = $('#contact-form-test-result');
            var $btnText = $btn.find('.button-text');
            var $spinner = $btn.find('.button-spinner');

            $btn.prop('disabled', true);
            $btnText.hide();
            $spinner.show();
            $result.hide();

            $.ajax({
                url: window.ajaxurl,
                method: 'POST',
                data: {
                    action: 'listeo_ai_test_contact_email',
                    nonce: config.test_email_nonce
                },
                success: function(response) {
                    if (response.success) {
                        $result.removeClass('error').addClass('success').html(response.data.message).show();
                    } else {
                        $result.removeClass('success').addClass('error').html(response.data.message).show();
                    }
                },
                error: function() {
                    $result.removeClass('success').addClass('error').html(i18n.requestFailed || 'Request failed. Please try again.').show();
                },
                complete: function() {
                    $btn.prop('disabled', false);
                    $btnText.show();
                    $spinner.hide();
                }
            });
        });

        // Save contact form settings
        $('#contact-form-save-settings-btn').on('click', function() {
            var $btn = $(this);
            var $result = $('#contact-form-save-result');
            var $btnText = $btn.find('.button-text');
            var $spinner = $btn.find('.button-spinner');

            $btn.prop('disabled', true);
            $btnText.hide();
            $spinner.show();
            $result.hide();

            $.ajax({
                url: window.ajaxurl,
                method: 'POST',
                data: {
                    action: 'listeo_ai_save_contact_form_settings',
                    nonce: config.contact_form_nonce,
                    recipient: $('#listeo_ai_contact_form_recipient').val(),
                    from_name: $('#listeo_ai_contact_form_from_name').val(),
                    from_email: $('#listeo_ai_contact_form_from_email').val(),
                    subject: $('#listeo_ai_contact_form_subject').val(),
                    success_message: $('#listeo_ai_contact_form_success_message').val()
                },
                success: function(response) {
                    if (response.success) {
                        $result.removeClass('error').addClass('success').html(response.data.message).show();
                        setTimeout(function() {
                            $('#contact-form-config-modal').fadeOut(200);
                        }, 1500);
                    } else {
                        $result.removeClass('success').addClass('error').html(response.data.message).show();
                    }
                },
                error: function() {
                    $result.removeClass('success').addClass('error').html(i18n.requestFailed || 'Request failed. Please try again.').show();
                },
                complete: function() {
                    $btn.prop('disabled', false);
                    $btnText.show();
                    $spinner.hide();
                }
            });
        });
    }

    /**
     * Initialize all UI handlers
     */
    function init() {
        initStickyFooters();
        initCollapsibleSections();
        initShortcodeGenerator();
        initVisualRadioCards();
        initThemeToggle();
        initTranslationInstaller();
        initQualitySlider();
        initCheckboxCards();
        initToggleableCards();
        initColorPicker();
        initSystemPromptCounter();
        initSettingsToggles();
        initQuickButtonsRemove();
        initQuickButtonTypeChange();
        initModalHandlers();
        initContactFormHandlers();

        console.log('AIRS Admin UI loaded');
    }

    // Initialize on document ready
    $(document).ready(init);

})(jQuery);
