/**
 * AI Chat Search - Admin Media Module
 *
 * Handles WordPress media uploader for custom icons and chat avatars.
 *
 * @package AI_Chat_Search
 * @since 1.0.0
 */

(function($) {
    'use strict';

    var i18n = window.listeo_ai_search_i18n || {};
    var customIconFrame;
    var chatAvatarFrame;

    /**
     * Initialize custom icon uploader
     */
    function initCustomIconUploader() {
        $('#listeo-upload-custom-icon').on('click', function(e) {
            e.preventDefault();

            // If the media frame already exists, reopen it
            if (customIconFrame) {
                customIconFrame.open();
                return;
            }

            // Create the media frame
            customIconFrame = wp.media({
                title: i18n.selectCustomIcon || 'Select Custom Icon',
                button: {
                    text: i18n.useThisIcon || 'Use this icon'
                },
                library: {
                    type: ['image/svg+xml']
                },
                multiple: false
            });

            // When an image is selected
            customIconFrame.on('select', function() {
                var attachment = customIconFrame.state().get('selection').first().toJSON();

                // Set the hidden input value
                $('#listeo_ai_floating_custom_icon').val(attachment.id);

                // Update the preview (escape URL for security)
                var $img = $('<img>')
                    .attr('src', attachment.url)
                    .attr('alt', 'Custom icon')
                    .css({'width': '60px', 'height': '60px', 'border-radius': '100px', 'object-fit': 'cover'});
                $('#listeo-custom-icon-preview').empty().append($img);

                // Show remove button if it doesn't exist
                if ($('#listeo-remove-custom-icon').length === 0) {
                    $('.airs-media-buttons').append(
                        '<button type="button" class="airs-button airs-button-secondary" id="listeo-remove-custom-icon" style="margin-left: 5px;">' +
                        (i18n.remove || 'Remove') + '</button>'
                    );
                }
            });

            // Open the modal
            customIconFrame.open();
        });

        // Remove custom icon handler
        $(document).on('click', '#listeo-remove-custom-icon', function(e) {
            e.preventDefault();

            // Clear the hidden input
            $('#listeo_ai_floating_custom_icon').val('');

            // Reset the preview to placeholder
            var btnColor = i18n.buttonColor || '#222222';
            var pluginUrl = i18n.pluginUrl || '';
            var placeholderHtml = '<div class="airs-media-placeholder" style="width: 60px; height: 60px; background-color: ' + btnColor + '; border-radius: 100px; display: flex; align-items: center; justify-content: center;">' +
                '<img src="' + pluginUrl + 'assets/icons/chat.svg" alt="Default icon" width="28" height="28" /></div>';
            $('#listeo-custom-icon-preview').html(placeholderHtml);

            // Remove the remove button
            $(this).remove();
        });
    }

    /**
     * Initialize chat avatar uploader
     */
    function initChatAvatarUploader() {
        $('#listeo-upload-chat-avatar').on('click', function(e) {
            e.preventDefault();

            // If the media frame already exists, reopen it
            if (chatAvatarFrame) {
                chatAvatarFrame.open();
                return;
            }

            // Create the media frame
            chatAvatarFrame = wp.media({
                title: i18n.selectChatAvatar || 'Select Chat Avatar',
                button: {
                    text: i18n.useThisImage || 'Use this image'
                },
                library: {
                    type: ['image']
                },
                multiple: false
            });

            // When an image is selected
            chatAvatarFrame.on('select', function() {
                var attachment = chatAvatarFrame.state().get('selection').first().toJSON();

                // Set the hidden input value
                $('#listeo_ai_chat_avatar').val(attachment.id);

                // Update the preview (escape URL for security)
                var $img = $('<img>')
                    .attr('src', attachment.url)
                    .attr('alt', 'Chat avatar')
                    .css({'width': '38px', 'height': '38px', 'border-radius': '100px', 'object-fit': 'cover'});
                $('#listeo-chat-avatar-preview').empty().append($img);

                // Show remove button if it doesn't exist
                if ($('#listeo-remove-chat-avatar').length === 0) {
                    $('#listeo-upload-chat-avatar').after(
                        '<button type="button" class="airs-button airs-button-secondary" id="listeo-remove-chat-avatar" style="margin-left: 5px;">' +
                        (i18n.remove || 'Remove') + '</button>'
                    );
                }
            });

            // Open the modal
            chatAvatarFrame.open();
        });

        // Remove chat avatar handler
        $(document).on('click', '#listeo-remove-chat-avatar', function(e) {
            e.preventDefault();

            // Clear the hidden input
            $('#listeo_ai_chat_avatar').val('');

            // Reset the preview to placeholder
            var placeholderHtml = '<div class="airs-media-placeholder" style="width: 38px; height: 38px; border: 2px dashed #ddd; border-radius: 100px; display: flex; align-items: center; justify-content: center; color: #999; box-sizing: content-box;">' +
                '<i class="sl sl-icon-user" style="font-size: 14px;"></i></div>';
            $('#listeo-chat-avatar-preview').html(placeholderHtml);

            // Remove the remove button
            $(this).remove();
        });
    }

    /**
     * Initialize all media handlers
     */
    function init() {
        // Check if WordPress media is available
        if (typeof wp === 'undefined' || typeof wp.media === 'undefined') {
            console.warn('AIRS Media: WordPress media library not available');
            return;
        }

        initCustomIconUploader();
        initChatAvatarUploader();

    }

    // Initialize on document ready
    $(document).ready(init);

})(jQuery);
